# -*- coding: utf-8 -*-
"""
Sistema de licencias con validación en línea para aplicaciones profesionales
Versión 2.1 - Modificada para compatibilidad con servidor PHP
"""

import json
import hashlib
import base64
import uuid
import os
import urllib.request
import urllib.error
import urllib.parse
import datetime
import time
import socket
import logging
import platform
import sys

# Configuración de logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("sistema_licencias.log"),
        logging.StreamHandler(sys.stdout)
    ]
)
logger = logging.getLogger("SistemaLicencias")

# Configuración
LICENSE_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'licencia.dat')
CACHE_FILE = os.path.join(os.path.dirname(os.path.abspath(__file__)), 'licencia_cache.json')
# URL de la API de licencias (reemplazar con la URL real en producción)
API_URL = "https://linen-chicken-624264.hostingersite.com/api-licencias/validar_licencia.php"
# Tiempo máximo de espera para validación online (segundos)
TIMEOUT_VALIDACION = 10

class SistemaLicenciasArchivoCompleto:
    """
    Sistema de licencias avanzado con validación local y en línea.
    Implementa persistencia local de licencia y caché de validación online.
    """
    
    def __init__(self, license_file=LICENSE_FILE, cache_file=CACHE_FILE, timeout=TIMEOUT_VALIDACION):
        """
        Inicializa el sistema de licencias con rutas personalizables.
        
        Args:
            license_file: Ruta al archivo de licencia local
            cache_file: Ruta al archivo de caché de validación
            timeout: Tiempo máximo de espera para conexiones (segundos)
        """
        self.license_file = license_file
        self.cache_file = cache_file
        self.timeout = timeout
        self.rutas_licencia = self.obtener_rutas_licencia()
        logger.debug(f"Sistema de licencias inicializado. Archivos posibles: {self.rutas_licencia}")

    def obtener_rutas_licencia(self):
        """
        Obtiene múltiples rutas donde buscar el archivo de licencia.
        """
        rutas = []
        
        # Si la aplicación está congelada (PyInstaller)
        if getattr(sys, 'frozen', False):
            # Ruta de la carpeta temporal de PyInstaller
            try:
                rutas.append(os.path.join(sys._MEIPASS, 'licencia.dat'))
            except AttributeError:
                logger.debug("sys._MEIPASS no disponible")
            
            # Ruta de la carpeta donde está el ejecutable
            ruta_ejecutable = os.path.dirname(sys.executable)
            rutas.append(os.path.join(ruta_ejecutable, 'licencia.dat'))
            
            # Si se extrajo en una subcarpeta 'dist'
            rutas.append(os.path.join(ruta_ejecutable, 'dist', 'licencia.dat'))
            
            # También buscar en niveles superiores
            ruta_padre = os.path.dirname(ruta_ejecutable)
            rutas.append(os.path.join(ruta_padre, 'licencia.dat'))
        
        # Siempre incluir la ruta del script actual como opción
        ruta_script = os.path.dirname(os.path.abspath(__file__))
        rutas.append(os.path.join(ruta_script, 'licencia.dat'))
        
        # La ruta pasada al constructor siempre se incluye
        if self.license_file not in rutas:
            rutas.append(self.license_file)
        
        logger.info(f"Rutas de búsqueda de licencia: {rutas}")
        return rutas

    def _obtener_id_hardware(self):
        """
        Obtiene identificador único de hardware combinando múltiples fuentes.
        El ID es consistente para la misma máquina pero difícil de replicar.
        """
        hw_components = []
        
        # 1. CPU ID (Windows)
        if platform.system() == 'Windows':
            try:
                import subprocess
                output = subprocess.check_output(['wmic', 'cpu', 'get', 'ProcessorId'], shell=True).decode()
                lines = [l.strip() for l in output.splitlines() if l.strip() and 'ProcessorId' not in l]
                if lines:
                    hw_components.append(lines[0])
            except Exception as e:
                logger.debug(f"No se pudo obtener CPU ID: {e}")
        
        # 2. Hostname
        try:
            hw_components.append(socket.gethostname())
        except Exception as e:
            logger.debug(f"No se pudo obtener hostname: {e}")
            
        # 3. Información del sistema
        hw_components.append(platform.platform())
            
        # 4. MAC address como fallback
        try:
            mac = uuid.getnode()
            hw_components.append(str(mac))
        except Exception as e:
            logger.debug(f"No se pudo obtener MAC: {e}")
            
        # 5. Si todo falla, usar un UUID persistente
        if not hw_components:
            uid_file = os.path.join(os.path.dirname(os.path.abspath(__file__)), '.hwid')
            if os.path.exists(uid_file):
                return open(uid_file, 'r').read().strip()
            hwid = str(uuid.uuid4())
            try:
                with open(uid_file, 'w') as f:
                    f.write(hwid)
                return hwid
            except Exception as e:
                logger.error(f"Error al crear archivo hwid: {e}")
                return str(uuid.uuid4())  # UUID temporal como último recurso
        
        # Combinar todos los componentes y generar hash
        combined = "-".join(hw_components)
        return hashlib.sha256(combined.encode()).hexdigest()[:32]

    def _hash(self, data):
        """Genera un hash SHA-256 del texto proporcionado."""
        return hashlib.sha256(data.encode()).hexdigest()

    def _ofuscar(self, data):
        """Codifica datos utilizando Base64 con salt."""
        salt = "SCJN2023"  # Salt para dificultar decodificación
        salted = f"{salt}{data}{salt[::-1]}"
        return base64.b64encode(salted.encode()).decode()

    def _desofuscar(self, data):
        """Decodifica datos ofuscados con Base64 y extrae el contenido original."""
        salt = "SCJN2023"
        decoded = base64.b64decode(data.encode()).decode()
        # Eliminar el salt del principio y final
        if decoded.startswith(salt) and decoded.endswith(salt[::-1]):
            return decoded[len(salt):-len(salt[::-1])]
        raise ValueError("Datos corruptos o manipulados")

    def generar_licencia_demo(self, nombre, email, tipo='Demo', dias=15):
        """
        Genera una licencia temporal hasta activación online.
        Esta función solo crea una licencia local temporal. Para una licencia completa,
        debe obtenerla a través del servidor web.
        
        Args:
            nombre: Nombre del usuario
            email: Correo electrónico del usuario
            tipo: Tipo de licencia ('Demo' o 'Full')
            dias: Días de validez para licencia temporal
            
        Returns:
            bool: True si se generó correctamente
        """
        try:
            # Datos de la licencia
            payload = {
                'nombre': nombre,
                'email': email,
                'tipo': tipo,
                'fecha_emision': datetime.datetime.utcnow().isoformat(),
                'fecha_expiracion': (datetime.datetime.utcnow() + 
                                    datetime.timedelta(days=dias)).isoformat(),
                'id_hardware': self._obtener_id_hardware(),
                'version': '2.1',
                'codigo_licencia': f"DEMO-{self._hash(email)[:10]}"  # Código demo simplificado
            }
            
            # Añadir un código de verificación para detectar manipulaciones
            verificacion = self._hash(f"{nombre}:{email}:{payload['id_hardware']}:{payload['fecha_emision']}")
            payload['verificacion'] = verificacion
            
            # Convertir a texto y ofuscar
            text = json.dumps(payload)
            obf = self._ofuscar(text)
            
            # Intentar guardar en todas las rutas posibles
            saved = False
            for ruta in self.rutas_licencia:
                try:
                    with open(ruta, 'w') as f:
                        f.write(obf)
                    logger.info(f"Licencia {tipo} generada para {email} en {ruta}")
                    self.license_file = ruta  # Actualizar la ruta principal
                    saved = True
                    break
                except Exception as e:
                    logger.warning(f"No se pudo guardar en {ruta}: {e}")
            
            if not saved:
                logger.error("No se pudo guardar la licencia en ninguna ubicación")
                return False
                
            return True
            
        except Exception as e:
            logger.error(f"Error al generar licencia: {e}")
            return False

    def registrar_licencia_servidor(self, codigo_licencia, email, nombre):
        """
        Registra una licencia obtenida del servidor en el archivo local.
        
        Args:
            codigo_licencia: Código generado por el servidor
            email: Email del usuario
            nombre: Nombre del usuario
            
        Returns:
            bool: True si se registró correctamente
        """
        try:
            # Datos de la licencia
            payload = {
                'nombre': nombre,
                'email': email,
                'tipo': 'Full',
                'fecha_emision': datetime.datetime.utcnow().isoformat(),
                'fecha_expiracion': (datetime.datetime.utcnow() + 
                                    datetime.timedelta(days=365)).isoformat(),  # Por defecto 1 año
                'id_hardware': self._obtener_id_hardware(),
                'version': '2.1',
                'codigo_licencia': codigo_licencia  # Usar el código generado por el servidor
            }
            
            # Añadir un código de verificación para detectar manipulaciones
            verificacion = self._hash(f"{nombre}:{email}:{payload['id_hardware']}:{payload['fecha_emision']}")
            payload['verificacion'] = verificacion
            
            # Convertir a texto y ofuscar
            text = json.dumps(payload)
            obf = self._ofuscar(text)
            
            # Intentar guardar en todas las rutas posibles
            saved = False
            for ruta in self.rutas_licencia:
                try:
                    with open(ruta, 'w') as f:
                        f.write(obf)
                    logger.info(f"Licencia registrada para {email} con código {codigo_licencia} en {ruta}")
                    self.license_file = ruta  # Actualizar la ruta principal
                    saved = True
                    break
                except Exception as e:
                    logger.warning(f"No se pudo guardar en {ruta}: {e}")
            
            if not saved:
                logger.error("No se pudo guardar la licencia en ninguna ubicación")
                return False
                
            return True
            
        except Exception as e:
            logger.error(f"Error al registrar licencia: {e}")
            return False

    def _guardar_cache_validacion(self, datos):
        """
        Guarda la información de validación en caché local.
        
        Args:
            datos: Diccionario con información de la licencia
        """
        try:
            datos['timestamp'] = time.time()
            
            # Intentar guardar en el mismo directorio que la licencia
            cache_dir = os.path.dirname(self.license_file)
            cache_file = os.path.join(cache_dir, 'licencia_cache.json')
            
            try:
                with open(cache_file, 'w') as f:
                    json.dump(datos, f)
                logger.debug(f"Cache de validación guardado en {cache_file}")
                self.cache_file = cache_file  # Actualizar ruta de caché
                return
            except Exception as e:
                logger.warning(f"No se pudo guardar caché en {cache_file}: {e}")
            
            # Si falla, intentar con la ruta original
            with open(self.cache_file, 'w') as f:
                json.dump(datos, f)
            logger.debug(f"Cache de validación guardado en {self.cache_file}")
        except Exception as e:
            logger.error(f"Error al guardar caché de validación: {e}")

    def _leer_cache_validacion(self):
        """
        Lee la información de la licencia desde la caché.
        
        Returns:
            dict: Datos de validación o None si no hay caché válida
        """
        # Lista de posibles ubicaciones para el caché
        rutas_cache = []
        
        # Mismas carpetas que las licencias pero con nombre de archivo de caché
        for ruta_lic in self.rutas_licencia:
            dir_lic = os.path.dirname(ruta_lic)
            rutas_cache.append(os.path.join(dir_lic, 'licencia_cache.json'))
        
        # Añadir la ruta original del caché
        if self.cache_file not in rutas_cache:
            rutas_cache.append(self.cache_file)
        
        # Intentar leer desde todas las ubicaciones
        for cache_path in rutas_cache:
            if not os.path.exists(cache_path):
                continue
                
            try:
                with open(cache_path, 'r') as f:
                    cache = json.load(f)
                    
                # Verificar si la caché es válida (menos de 7 días)
                if time.time() - cache.get('timestamp', 0) > 7 * 24 * 60 * 60:
                    logger.debug(f"Caché expirada en {cache_path}")
                    continue
                    
                logger.debug(f"Usando caché de validación de {cache_path}")
                self.cache_file = cache_path  # Actualizar ruta del caché
                return cache
                
            except Exception as e:
                logger.error(f"Error al leer caché de validación en {cache_path}: {e}")
        
        return None

    def _validar_online(self, email, licencia_info):
        """
        Valida la licencia contra el servidor en línea.
        Versión actualizada para usar el código de licencia del servidor.
        
        Args:
            email: Email del usuario
            licencia_info: Información completa de la licencia local
            
        Returns:
            dict: Respuesta del servidor o estado offline
        """
        # Usar el código de licencia almacenado en el archivo local
        # Este es el código generado por el servidor PHP
        codigo_licencia = licencia_info.get('codigo_licencia', '')
        
        try:
            # Log para depuración
            logger.debug(f"Validando licencia online: Email={email}, Código={codigo_licencia}")
            
            # Preparar datos para la solicitud
            datos_solicitud = {
                'email': email,
                'codigo_licencia': codigo_licencia,
                'id_hardware': licencia_info.get('id_hardware', '')  # Incluir para posible uso futuro
            }
            
            # Serializar a JSON
            data = json.dumps(datos_solicitud).encode('utf-8')
            
            # Crear la solicitud HTTP
            req = urllib.request.Request(API_URL)
            req.add_header('Content-Type', 'application/json; charset=utf-8')
            req.add_header('Content-Length', len(data))
            req.add_header('User-Agent', 'SCJN-LicenseClient/2.1')
            
            # Enviar solicitud con timeout
            response = urllib.request.urlopen(req, data, timeout=self.timeout)
            result = json.loads(response.read().decode('utf-8'))
            
            # Registrar la respuesta para depuración
            logger.debug(f"Respuesta del servidor: {result}")
            
            # Si la licencia es válida, guardar en caché
            if result.get('valido', False):
                self._guardar_cache_validacion(result)
                logger.info("Licencia validada correctamente en línea")
            else:
                logger.warning(f"Validación online fallida: {result.get('mensaje', 'Sin mensaje')}")
            
            return result
            
        except urllib.error.URLError as e:
            logger.warning(f"Error de conexión al validar licencia: {e}")
            return {'valido': False, 'mensaje': f'Error de conexión: {str(e)}', 'offline': True}
            
        except Exception as e:
            logger.error(f"Error inesperado al validar licencia online: {e}")
            return {'valido': False, 'mensaje': f'Error: {str(e)}', 'offline': True}

    def _leer_archivo_licencia(self):
        """
        Lee el archivo de licencia desde cualquiera de las ubicaciones posibles.
        
        Returns:
            dict: Datos de la licencia o None si no se encuentra
        """
        for ruta in self.rutas_licencia:
            if not os.path.exists(ruta):
                logger.debug(f"No existe archivo de licencia en {ruta}")
                continue
            
            try:
                with open(ruta, 'r') as f:
                    obf = f.read().strip()
                text = self._desofuscar(obf)
                payload = json.loads(text)
                
                # Si llegamos aquí, encontramos una licencia válida
                logger.info(f"Archivo de licencia encontrado en: {ruta}")
                self.license_file = ruta  # Actualizar la ruta principal
                return payload
                
            except Exception as e:
                logger.warning(f"Error al leer archivo de licencia en {ruta}: {e}")
        
        return None

    def verificar_licencia(self):
        """
        Verifica la licencia usando primero caché, luego validación online, y finalmente licencia local.
        
        Returns:
            bool: True si la licencia es válida, False en caso contrario
        """
        logger.info("Iniciando verificación de licencia")
        
        # 1. Verificar si existe una licencia local en cualquiera de las rutas posibles
        payload = self._leer_archivo_licencia()
        if not payload:
            logger.warning("No se encontró archivo de licencia válido en ninguna ubicación")
            return False
        
        # Validar integridad de la licencia
        if 'verificacion' in payload:
            verificacion_original = payload['verificacion']
            verificacion_calculada = self._hash(f"{payload.get('nombre')}:{payload.get('email')}:{payload.get('id_hardware')}:{payload.get('fecha_emision')}")
            
            if verificacion_original != verificacion_calculada:
                logger.warning("Licencia manipulada: verificación no coincide")
                return False
        
        # 3. Verificar hardware (solo para validación offline)
        hw_actual = self._obtener_id_hardware()
        if payload.get('id_hardware') != hw_actual:
            logger.warning(f"Licencia asociada a otro hardware")
            # No retornamos False aquí, dejamos que la validación online decida
        
        # Obtener datos para verificación online
        email = payload.get('email')
        
        # 4. Primero intentar usar la caché
        cache = self._leer_cache_validacion()
        if cache and cache.get('valido', False):
            logger.info("Licencia verificada desde caché")
            # Mostrar algunos detalles de la licencia
            for k in ['nombre', 'tipo', 'hasta']:
                if k in cache:
                    logger.info(f"  {k}: {cache[k]}")
            return True
        
        # 5. Si no hay caché válida, intentar validación online
        logger.info("Intentando validación online...")
        resultado = self._validar_online(email, payload)
        
        if resultado.get('valido', False):
            logger.info("Licencia validada online correctamente")
            return True
        
        # 6. Si falla la validación online por conexión, verificar offline
        if resultado.get('offline', False):
            logger.warning("Validación online fallida. Realizando verificación offline...")
            
            # Para validación offline, el hardware DEBE coincidir
            if payload.get('id_hardware') != hw_actual:
                logger.warning("Licencia asociada a otro hardware, validación offline fallida")
                return False
            
            try:
                fecha_actual = datetime.datetime.utcnow()
                fecha_expiracion = datetime.datetime.fromisoformat(payload.get('fecha_expiracion'))
                
                if fecha_actual > fecha_expiracion:
                    logger.warning(f"Licencia expirada: {fecha_expiracion}")
                    return False
                
                # Licencia válida temporalmente offline
                logger.info(f"Licencia válida temporalmente (offline hasta {fecha_expiracion})")
                return True
                
            except Exception as e:
                logger.error(f"Error en verificación offline: {e}")
                return False
        else:
            # Fallo de validación online (el servidor dijo que no es válida)
            logger.warning(f"Licencia rechazada por el servidor: {resultado.get('mensaje', 'Sin detalles')}")
            return False

    def activar_licencia_desde_servidor(self, email, codigo_licencia, nombre):
        """
        Nueva función para activar una licencia generada en el servidor.
        
        Args:
            email: Email del usuario
            codigo_licencia: Código de licencia generado por el servidor
            nombre: Nombre del usuario
            
        Returns:
            bool: True si la activación fue exitosa
        """
        try:
            # Registrar la licencia localmente
            if not self.registrar_licencia_servidor(codigo_licencia, email, nombre):
                logger.error("No se pudo registrar la licencia localmente")
                return False
                
            # Verificar que la licencia sea válida
            if not self.verificar_licencia():
                logger.error("La licencia registrada no es válida")
                return False
                
            logger.info(f"Licencia activada correctamente para {email}")
            return True
            
        except Exception as e:
            logger.error(f"Error al activar licencia: {e}")
            return False

def verificar_licencia_aplicacion():
    """
    Función simplificada para verificar la licencia de la aplicación.
    Para usar directamente en la aplicación principal.
    
    Returns:
        bool: True si la licencia es válida, False en caso contrario
    """
    sistema = SistemaLicenciasArchivoCompleto()
    return sistema.verificar_licencia()

def obtener_rutas_licencia():
    """
    Función que devuelve las rutas donde se busca el archivo de licencia.
    Útil para informar al usuario dónde debe colocar su archivo de licencia.
    
    Returns:
        list: Lista de rutas donde se busca el archivo de licencia
    """
    sistema = SistemaLicenciasArchivoCompleto()
    return sistema.obtener_rutas_licencia()

def main():
    """Función principal para uso como script independiente."""
    import argparse
    parser = argparse.ArgumentParser(description="Sistema de Licencias SCJN v2.1")
    
    # Subcomandos
    subparsers = parser.add_subparsers(dest='comando', help='Comando a ejecutar')
    
    # Comando para generar licencia demo
    gen = subparsers.add_parser('generar-demo', help='Generar licencia demo local')
    gen.add_argument('--nombre', required=True, help='Nombre del usuario')
    gen.add_argument('--email', required=True, help='Email del usuario')
    gen.add_argument('--dias', type=int, default=15, help='Días de validez')
    
    # Comando para activar licencia desde servidor
    act = subparsers.add_parser('activar', help='Activar licencia desde servidor')
    act.add_argument('--codigo', required=True, help='Código de licencia')
    act.add_argument('--email', required=True, help='Email del usuario')
    act.add_argument('--nombre', required=True, help='Nombre del usuario')
    
    # Comando para verificar licencia
    subparsers.add_parser('verificar', help='Verificar licencia actual')
    
    # Comando para mostrar info de hardware
    subparsers.add_parser('hwid', help='Mostrar ID de hardware')
    
    # Parsear argumentos
    args = parser.parse_args()
    
    # Crear instancia del sistema
    sistema = SistemaLicenciasArchivoCompleto()
    
    # Ejecutar comando
    if args.comando == 'generar-demo':
        resultado = sistema.generar_licencia_demo(args.nombre, args.email, 'Demo', args.dias)
        if resultado:
            print(f"Licencia demo generada correctamente para {args.email}")
            print(f"Válida por {args.dias} días (verificación offline)")
        else:
            print("Error al generar licencia demo")
    
    elif args.comando == 'activar':
        resultado = sistema.activar_licencia_desde_servidor(args.email, args.codigo, args.nombre)
        if resultado:
            print(f"Licencia activada correctamente para {args.email}")
        else:
            print("Error al activar licencia")
    
    elif args.comando == 'verificar':
        if sistema.verificar_licencia():
            print("✅ Licencia válida")
        else:
            print("❌ Licencia no válida")
    
    elif args.comando == 'hwid':
        hwid = sistema._obtener_id_hardware()
        print(f"ID de Hardware: {hwid}")
    
    else:
        parser.print_help()

if __name__ == '__main__':
    main()