import time
import random
import logging
import json
import os
from datetime import datetime
from functools import wraps
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait, Select
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import (
    TimeoutException,
    NoSuchElementException,
    StaleElementReferenceException,
    ElementClickInterceptedException,
    WebDriverException
)

# Configuración de logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("scraper_scjn.log"),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger("ScraperSCJN")

class RetryError(Exception):
    """Excepción personalizada para errores después de agotar reintentos"""
    pass

def retry(max_attempts=3, delay=1, backoff=2, exceptions=(Exception,)):
    """
    Decorador para reintentar funciones con retraso exponencial.

    Args:
        max_attempts (int): Número máximo de intentos
        delay (float): Retraso inicial en segundos
        backoff (float): Factor de incremento para el retraso
        exceptions (tuple): Excepciones que activarán el reintento
    """
    def decorator(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            mtries, mdelay = max_attempts, delay
            last_exception = None

            while mtries > 0:
                try:
                    return func(*args, **kwargs)
                except exceptions as e:
                    mtries -= 1
                    if mtries == 0:
                        raise RetryError(f"Se agotaron los reintentos ({max_attempts}): {str(e)}")

                    # Agregar variación aleatoria (jitter) para evitar sobrecarga sincronizada
                    sleep_time = mdelay + (random.random() * mdelay * 0.5)
                    logger.warning(f"Reintento en {sleep_time:.2f}s - {func.__name__}: {str(e)}")
                    last_exception = e
                    time.sleep(sleep_time)
                    mdelay *= backoff

            # Si llegamos aquí, se agotaron los reintentos
            raise RetryError(f"Error inesperado después de {max_attempts} intentos: {last_exception}")
        return wrapper
    return decorator

class ResultadosCache:
    """
    Clase para gestionar el caché de resultados y proporcionar recuperación ante fallos.
    """
    def __init__(self, nombre_archivo=None):
        self.resultados = []
        self.ultima_pagina_procesada = 0
        self.nombre_archivo = nombre_archivo or f"cache_resultados_{datetime.now().strftime('%Y%m%d_%H%M%S')}.json"

    def agregar_resultados(self, resultados_pagina, numero_pagina):
        """Agrega resultados de una página al caché"""
        self.resultados.extend(resultados_pagina)
        self.ultima_pagina_procesada = max(self.ultima_pagina_procesada, numero_pagina)
        self._guardar_cache()

    def obtener_resultados(self):
        """Obtiene todos los resultados almacenados"""
        return self.resultados

    def obtener_ultima_pagina(self):
        """Obtiene el número de la última página procesada"""
        return self.ultima_pagina_procesada

    def _guardar_cache(self):
        """Guarda el caché en disco"""
        datos_cache = {
            "ultima_actualizacion": datetime.now().isoformat(),
            "ultima_pagina": self.ultima_pagina_procesada,
            "total_resultados": len(self.resultados),
            "resultados": self.resultados
        }

        try:
            with open(self.nombre_archivo, 'w', encoding='utf-8') as f:
                json.dump(datos_cache, f, ensure_ascii=False, indent=2)
        except Exception as e:
            logger.error(f"Error al guardar caché: {str(e)}")

    def cargar_cache_existente(self):
        """Carga un caché existente si está disponible"""
        if os.path.exists(self.nombre_archivo):
            try:
                with open(self.nombre_archivo, 'r', encoding='utf-8') as f:
                    datos_cache = json.load(f)
                    self.resultados = datos_cache.get("resultados", [])
                    self.ultima_pagina_procesada = datos_cache.get("ultima_pagina", 0)
                    logger.info(f"Caché cargado: {len(self.resultados)} resultados, última página: {self.ultima_pagina_procesada}")
                    return True
            except Exception as e:
                logger.error(f"Error al cargar caché: {str(e)}")
        return False

class ScraperTesisSCJN:
    """
    Clase para realizar scraping del buscador de tesis de la Suprema Corte
    de Justicia de la Nación de México, con optimización de rendimiento
    y manejo avanzado de errores.
    """

    def __init__(self, headless=False, modo_debug=False, timeout_default=10, cache_archivo=None):
        """
        Inicializa el scraper con opciones configurables.

        Args:
            headless (bool): Si se ejecuta en modo headless (sin interfaz gráfica)
            modo_debug (bool): Activa el modo de depuración con capturas de pantalla
            timeout_default (int): Tiempo de espera predeterminado en segundos
            cache_archivo (str): Nombre del archivo para persistir el caché de resultados
        """
        self.url_base = "https://sjf2.scjn.gob.mx/busqueda-principal-tesis"
        self.modo_debug = modo_debug
        self.timeout_default = timeout_default
        self.cache = ResultadosCache(cache_archivo)
        self.driver = self._configurar_driver(headless)

        # Contadores para estadísticas
        self.estadisticas = {
            "paginas_procesadas": 0,
            "resultados_obtenidos": 0,
            "reintentos_navegacion": 0,
            "tiempo_inicio": time.time()
        }

    def _configurar_driver(self, headless):
        """
        Configura el driver de Selenium con las opciones apropiadas

        Args:
            headless (bool): Si se ejecuta en modo sin interfaz gráfica

        Returns:
            WebDriver: Instancia configurada del driver
        """
        try:
            # Importar el helper para el driver
            import sys
            import os

            # Determinar si estamos en un ejecutable o script
            if getattr(sys, 'frozen', False):
                base_path = sys._MEIPASS
            else:
                base_path = os.path.dirname(os.path.abspath(__file__))

            # Importar dinámicamente el helper
            sys.path.insert(0, base_path)

            # Importar la función desde el módulo driver_helper
            from driver_helper import get_chrome_driver

            logger.info("Inicializando WebDriver Chrome con driver_helper...")
            driver = get_chrome_driver(headless=headless)
            driver.set_page_load_timeout(self.timeout_default * 2)  # Timeout para carga de página
            return driver
        except Exception as e:
            logger.error(f"Error al inicializar el driver: {str(e)}")
            raise

    def _captura_debug(self, nombre):
        """
        Guarda una captura de pantalla para depuración si el modo debug está activado

        Args:
            nombre (str): Nombre base para el archivo de captura
        """
        if not self.modo_debug:
            return

        try:
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            filename = f"debug_{nombre}_{timestamp}.png"
            self.driver.save_screenshot(filename)
            logger.debug(f"Captura de pantalla guardada: {filename}")
        except Exception as e:
            logger.warning(f"No se pudo guardar captura de depuración: {str(e)}")

    def _esperar_elemento(self, by, valor, timeout=None, condicion=EC.presence_of_element_located, mensaje=None):
        """
        Espera y retorna un elemento del DOM cuando está disponible

        Args:
            by (By): Tipo de selector (By.ID, By.CSS_SELECTOR, etc.)
            valor (str): Valor del selector
            timeout (int): Tiempo máximo de espera en segundos
            condicion (EC): Condición de espera de Selenium
            mensaje (str): Mensaje personalizado para el error

        Returns:
            WebElement: El elemento encontrado

        Raises:
            TimeoutException: Si el elemento no se encuentra en el tiempo especificado
        """
        if timeout is None:
            timeout = self.timeout_default

        mensaje = mensaje or f"Elemento no encontrado: {by}='{valor}' después de {timeout}s"

        try:
            return WebDriverWait(self.driver, timeout).until(
                condicion((by, valor))
            )
        except TimeoutException:
            logger.warning(mensaje)
            self._captura_debug(f"timeout_{by}_{valor.replace('/', '_')}")
            raise TimeoutException(mensaje)

    def _esperar_elementos(self, by, valor, timeout=None, condicion=EC.presence_of_all_elements_located):
        """
        Espera y retorna múltiples elementos del DOM cuando están disponibles

        Args:
            by (By): Tipo de selector
            valor (str): Valor del selector
            timeout (int): Tiempo máximo de espera en segundos
            condicion (EC): Condición de espera de Selenium

        Returns:
            list: Lista de elementos encontrados (puede estar vacía)
        """
        if timeout is None:
            timeout = self.timeout_default

        try:
            return WebDriverWait(self.driver, timeout).until(
                condicion((by, valor))
            )
        except TimeoutException:
            logger.debug(f"No se encontraron elementos con {by}='{valor}'")
            return []

    def _clic_seguro(self, elemento, timeout=2, usar_js=False):
        """
        Realiza un clic seguro en un elemento con manejo de excepciones

        Args:
            elemento (WebElement): Elemento en el que hacer clic
            timeout (int): Tiempo de espera antes de reintentar con JS
            usar_js (bool): Forzar el uso de JavaScript para el clic

        Returns:
            bool: True si el clic fue exitoso, False en caso contrario
        """
        try:
            # Asegurar que el elemento es visible
            self.driver.execute_script("arguments[0].scrollIntoView({block: 'center'});", elemento)
            time.sleep(0.5)  # Pequeña pausa para scroll

            if usar_js:
                self.driver.execute_script("arguments[0].click();", elemento)
                logger.debug("Clic realizado con JavaScript")
                return True

            try:
                elemento.click()
                logger.debug("Clic normal realizado con éxito")
                return True
            except (ElementClickInterceptedException, StaleElementReferenceException) as e:
                logger.debug(f"Error en clic normal: {str(e)}, intentando con JS")
                time.sleep(timeout)
                self.driver.execute_script("arguments[0].click();", elemento)
                logger.debug("Clic con JavaScript realizado después de error")
                return True

        except Exception as e:
            logger.warning(f"Error al hacer clic en elemento: {str(e)}")
            self._captura_debug("error_clic")
            return False

    @retry(max_attempts=3, delay=2, backoff=2,
            exceptions=(TimeoutException, WebDriverException, StaleElementReferenceException))
    def buscar_tesis(self, termino_busqueda, numero_resultados=50, continuar_cache=True):
        """
        Realiza una búsqueda de tesis con el término proporcionado,
        con manejo de errores y capacidad de recuperación.

        Args:
            termino_busqueda (str): Término a buscar en el sistema
            numero_resultados (int): Número de resultados a obtener en total
            continuar_cache (bool): Si se debe intentar continuar desde caché existente

        Returns:
            list: Lista de diccionarios con los resultados encontrados
        """
        # Reiniciar estadísticas
        self.estadisticas = {
            "paginas_procesadas": 0,
            "resultados_obtenidos": 0,
            "reintentos_navegacion": 0,
            "tiempo_inicio": time.time()
        }

        # Intentar cargar caché si está habilitado
        if continuar_cache and self.cache.cargar_cache_existente():
            logger.info(f"Continuando búsqueda desde caché. Ya tenemos {len(self.cache.obtener_resultados())} resultados.")

            # Si ya tenemos suficientes resultados, no es necesario hacer más scraping
            if len(self.cache.obtener_resultados()) >= numero_resultados:
                logger.info(f"El caché ya contiene {len(self.cache.obtener_resultados())} resultados, no es necesario obtener más.")
                return self.cache.obtener_resultados()[:numero_resultados]

            # IMPLEMENTACIÓN FUTURA: Continuar desde última página guardada en caché
            # Por ahora, reiniciamos la búsqueda
            logger.info("La continuación desde última página no está implementada. Reiniciando búsqueda.")

        logger.info(f"Iniciando búsqueda para: {termino_busqueda}")

        # Paso 1: Acceder a la página de búsqueda
        self.driver.get(self.url_base)
        logger.info("Accediendo a la página principal de búsqueda...")

        # Paso 2: Optimizado - Esperar y localizar el campo de búsqueda con múltiples selectores
        selectores_campo_busqueda = [
            "input[placeholder*='Escriba el tema']",
            "input[placeholder*='Buscar']",
            "input.busqueda",
            "input[type='search']",
            "input.search-input"
        ]

        campo_busqueda = None
        for selector in selectores_campo_busqueda:
            elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=5)
            if elementos:
                campo_busqueda = elementos[0]
                logger.info(f"Campo de búsqueda localizado con selector: {selector}")
                break

        # Si no encontramos con los selectores, intentar método alternativo
        if not campo_busqueda:
            logger.warning("Usando método alternativo para encontrar campo de búsqueda...")
            campos_input = self.driver.find_elements(By.TAG_NAME, "input")
            for campo in campos_input:
                placeholder = campo.get_attribute("placeholder") or ""
                if "tema" in placeholder.lower() or "busca" in placeholder.lower():
                    campo_busqueda = campo
                    logger.info(f"Campo encontrado por atributo placeholder: {placeholder}")
                    break

        if not campo_busqueda:
            raise NoSuchElementException("No se pudo localizar el campo de búsqueda por ningún método")

        # Paso 3: Preparar el término de búsqueda
        if not termino_busqueda.startswith('"') and not termino_busqueda.endswith('"'):
            termino_busqueda = f'"{termino_busqueda}"'

        # Paso 4: Ingresar el término de búsqueda
        campo_busqueda.clear()
        campo_busqueda.send_keys(termino_busqueda)
        logger.info(f"Término de búsqueda ingresado: {termino_busqueda}")

        # Paso 5: Optimizado - Hacer clic en el botón de búsqueda con estrategia mejorada
        selectores_boton = [
            "button.buscar", "button[type='submit']", "button.btn-search",
            "button.btn-primary", "button.search", "button[aria-label='Buscar']"
        ]

        boton_buscar = None
        # Intentar localizar usando selectores CSS
        for selector in selectores_boton:
            elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=2)
            if elementos:
                boton_buscar = elementos[0]
                logger.info(f"Botón encontrado con selector: {selector}")
                break

        # Si no lo encontramos, intentar por XPath
        if not boton_buscar:
            xpath_botones = [
                "//button[contains(text(), 'Buscar')]",
                "//button[contains(@aria-label, 'Buscar')]",
                "//button[.//i[contains(@class, 'search')]]"
            ]

            for xpath in xpath_botones:
                elementos = self._esperar_elementos(By.XPATH, xpath, timeout=2)
                if elementos:
                    boton_buscar = elementos[0]
                    logger.info(f"Botón encontrado con XPath: {xpath}")
                    break

        # Si aún no encontramos, usar JavaScript para identificar el botón de búsqueda
        if not boton_buscar:
            logger.warning("Intentando encontrar botón de búsqueda mediante análisis heurístico...")
            script_buscar_boton = """
            function encontrarBotonBusqueda() {
                // 1. Buscar por texto
                const botonesPorTexto = Array.from(document.querySelectorAll('button'))
                    .filter(b => {
                        const texto = (b.textContent || '').toLowerCase();
                        return texto.includes('buscar') || texto.includes('busca') || texto.includes('search');
                    });
                if (botonesPorTexto.length > 0) return botonesPorTexto[0];

                // 2. Buscar por iconos
                const botonesConIconos = Array.from(document.querySelectorAll('button'))
                    .filter(b => {
                        return b.querySelector('i[class*="search"]') ||
                               b.querySelector('svg[class*="search"]') ||
                               b.innerHTML.includes('search') ||
                               b.innerHTML.includes('lupa');
                    });
                if (botonesConIconos.length > 0) return botonesConIconos[0];

                // 3. Buscar por posición relativa al campo de búsqueda
                const campoInput = document.querySelector('input[placeholder*="tema"], input[placeholder*="Buscar"], input[type="search"]');
                if (campoInput) {
                    const rectInput = campoInput.getBoundingClientRect();
                    const botones = Array.from(document.querySelectorAll('button'));

                    // Ordenar por cercanía al campo
                    botones.sort((a, b) => {
                        const rectA = a.getBoundingClientRect();
                        const rectB = b.getBoundingClientRect();
                        const distA = Math.sqrt(Math.pow(rectA.left - rectInput.right, 2) + Math.pow(rectA.top - rectInput.top, 2));
                        const distB = Math.sqrt(Math.pow(rectB.left - rectInput.right, 2) + Math.pow(rectB.top - rectInput.top, 2));
                        return distA - distB;
                    });

                    // El primer botón a la derecha del campo
                    const botonCercano = botones.find(b => {
                        const rect = b.getBoundingClientRect();
                        return rect.left >= rectInput.right && Math.abs(rect.top - rectInput.top) < 50;
                    });

                    if (botonCercano) return botonCercano;
                }

                return null;
            }
            return encontrarBotonBusqueda();
            """

            boton_buscar = self.driver.execute_script(script_buscar_boton)
            if boton_buscar:
                logger.info("Botón de búsqueda encontrado mediante análisis heurístico con JavaScript")

        if not boton_buscar:
            raise NoSuchElementException("No se pudo localizar ningún botón de búsqueda")

        # Hacer clic en el botón con manejo de excepciones
        if not self._clic_seguro(boton_buscar):
            # Si falla, intentar enviar formulario directamente
            logger.warning("Fallaron los intentos de clic, intentando enviar formulario...")
            try:
                campo_busqueda.submit()
                logger.info("Formulario enviado directamente")
            except Exception as e:
                logger.error(f"Error al enviar formulario: {str(e)}")
                raise

        logger.info("Búsqueda iniciada. Esperando resultados...")

        # Esperar transición a página de resultados (optimizado)
        timeout_resultados = self.timeout_default * 1.5
        inicio_espera = time.time()
        url_cambiada = False
        url_inicial = self.driver.current_url

        while time.time() - inicio_espera < timeout_resultados:
            url_actual = self.driver.current_url
            if url_actual != url_inicial:
                url_cambiada = True
                logger.info(f"URL cambiada: {url_actual}")
                break
            time.sleep(0.5)  # Espera corta para verificar cambio

        if not url_cambiada:
            logger.warning(f"La URL no cambió después de hacer clic en búsqueda: {self.driver.current_url}")
            self._captura_debug("sin_cambio_url")

        # Recopilar resultados de múltiples páginas hasta alcanzar el número deseado
        resultados_completos = []
        pagina_actual = 1
        max_paginas = (numero_resultados + 19) // 20  # Aproximadamente 20 por página

        logger.info(f"Iniciando recopilación de hasta {numero_resultados} resultados (máx. {max_paginas} páginas)...")

        while len(resultados_completos) < numero_resultados and pagina_actual <= max_paginas:
            # Esperar un tiempo para asegurar carga completa (optimizado)
            try:
                # Esperar hasta que desaparezca el indicador de carga, si existe
                self._esperar_elementos(By.CSS_SELECTOR, ".loading, .spinner", timeout=3,
                                        condicion=EC.invisibility_of_element_located)
            except:
                # Si no hay indicador de carga, continuar
                pass

            # Extraer resultados con manejo de errores
            try:
                resultados_pagina = self._extraer_resultados_con_reintentos(pagina_actual)

                if not resultados_pagina:
                    logger.warning(f"No se encontraron resultados en la página {pagina_actual}")
                    # Si es la primera página y no hay resultados, podría ser un error crítico
                    if pagina_actual == 1:
                        self._captura_debug("sin_resultados_p1")
                        if "no encontr" in self.driver.page_source.lower() or "sin resultados" in self.driver.page_source.lower():
                            logger.info("El sistema indica explícitamente que no hay resultados para esta búsqueda")
                            break
                    else:
                        # Si no es la primera página, probablemente llegamos al final
                        logger.info("Parece que hemos llegado al final de los resultados disponibles")
                        break

                # Actualizar estadísticas
                self.estadisticas["paginas_procesadas"] += 1
                self.estadisticas["resultados_obtenidos"] += len(resultados_pagina)

                # Agregar los resultados de esta página al caché y a los resultados completos
                self.cache.agregar_resultados(resultados_pagina, pagina_actual)
                resultados_completos.extend(resultados_pagina)

                logger.info(f"Página {pagina_actual}: {len(resultados_pagina)} resultados. Total: {len(resultados_completos)}/{numero_resultados}")

                # Si ya tenemos suficientes resultados, salir
                if len(resultados_completos) >= numero_resultados:
                    break

                # Intentar navegar a la siguiente página con reintentos
                if not self._ir_siguiente_pagina_con_reintentos():
                    logger.info("No se encontró botón de siguiente página o ya estamos en la última página")
                    break

                pagina_actual += 1

                # Pequeña pausa aleatoria para evitar detección como bot
                time.sleep(0.5 + random.random())

            except RetryError as e:
                logger.error(f"Error crítico durante el procesamiento de la página {pagina_actual}: {str(e)}")
                self._captura_debug(f"error_pagina_{pagina_actual}")
                break
            except Exception as e:
                logger.error(f"Excepción no manejada en página {pagina_actual}: {str(e)}")
                self._captura_debug(f"excepcion_pagina_{pagina_actual}")
                break

        # Calcular estadísticas finales
        tiempo_total = time.time() - self.estadisticas["tiempo_inicio"]
        logger.info(f"Búsqueda completada en {tiempo_total:.2f} segundos")
        logger.info(f"Estadísticas: {self.estadisticas['paginas_procesadas']} páginas, "
                    f"{self.estadisticas['resultados_obtenidos']} resultados, "
                    f"{self.estadisticas['reintentos_navegacion']} reintentos de navegación")

        # Limitar los resultados al número solicitado
        if len(resultados_completos) > numero_resultados:
            resultados_completos = resultados_completos[:numero_resultados]
            logger.info(f"Limitando a los primeros {numero_resultados} resultados solicitados")

        return resultados_completos

    @retry(max_attempts=3, delay=1, backoff=2,
            exceptions=(TimeoutException, StaleElementReferenceException))
    def _extraer_resultados_con_reintentos(self, numero_pagina):
        """
        Extrae los resultados de la página actual con reintentos automáticos.

        Args:
            numero_pagina (int): Número de página actual para registro

        Returns:
            list: Lista de resultados de la página actual
        """
        return self._extraer_resultados(numero_pagina)

    def _extraer_resultados(self, numero_pagina):
        """
        Extrae los resultados de la página actual.

        Args:
            numero_pagina (int): Número de página actual para registro

        Returns:
            list: Lista de diccionarios con los resultados encontrados
        """
        resultados = []

        # Conjunto de selectores para items de resultado optimizados
        selectores_resultados = [
            ".registro-item",
            ".resultado-item",
            "div[id*='registro']",
            "div[class*='resultado']",
            "article.resultado",
            ".card:has(div[class*='tesis'])",
            ".listado div.mb-4"
        ]

        # Estrategia 1: Intentar selectores específicos
        elementos_tesis = []
        for selector in selectores_resultados:
            elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=3)
            if elementos:
                elementos_tesis = elementos
                logger.info(f"Encontrados {len(elementos_tesis)} resultados con selector: {selector}")
                break

        # Estrategia 2: Si no encontramos resultados, intentar análisis de DOM con JavaScript
        if not elementos_tesis:
            logger.warning("Usando técnica alternativa para identificar resultados...")
            script_detectar_resultados = """
            function encontrarResultados() {
                // Buscar elementos que parezcan tesis por su contenido
                const todosLosDivs = Array.from(document.querySelectorAll('div'));
                const posiblesResultados = todosLosDivs.filter(div => {
                    const texto = div.textContent || '';
                    // Buscar elementos que contengan texto que parezca tesis
                    return (
                        (texto.includes('Registro digital') || texto.includes('Tesis')) &&
                        div.querySelectorAll('div, p, span').length >= 3 &&  // Tiene estructura interna
                        div.offsetHeight > 100  // Tiene altura significativa
                    );
                });

                // Si encontramos posibles resultados, ver si están agrupados (misma clase o patrón)
                if (posiblesResultados.length > 0) {
                    // Ver si comparten clase
                    const clases = {};
                    posiblesResultados.forEach(div => {
                        const clasesArr = Array.from(div.classList);
                        clasesArr.forEach(clase => {
                            clases[clase] = (clases[clase] || 0) + 1;
                        });
                    });

                    // Encontrar la clase más común
                    let claseComun = null;
                    let maxConteo = 0;
                    for (const [clase, conteo] of Object.entries(clases)) {
                        if (conteo > maxConteo && clase.length > 0) {
                            maxConteo = conteo;
                            claseComun = clase;
                        }
                    }

                    // Si hay una clase común, usar elementos con esa clase
                    if (claseComun && maxConteo > 1) {
                        return Array.from(document.querySelectorAll('.' + claseComun));
                    }

                    // Si no hay clase común, devolver los elementos encontrados
                    return posiblesResultados;
                }

                // En último caso, buscar por texto
                const textosTesis = Array.from(document.querySelectorAll('div'))
                    .filter(div => {
                        const texto = div.textContent || '';
                        return texto.includes('AMPARO') ||
                               texto.includes('JUICIO') ||
                               texto.includes('TESIS') ||
                               (texto.length > 200 && /\\d{6,}/.test(texto)); // Textos largos con números de registro
                    });

                return textosTesis;
            }
            return encontrarResultados();
            """

            elementos_js = self.driver.execute_script(script_detectar_resultados)
            if elementos_js:
                elementos_tesis = elementos_js
                logger.info(f"Encontrados {len(elementos_tesis)} resultados mediante análisis heurístico")

        # Si aún no hay resultados, puede ser que no haya ninguno
        if not elementos_tesis:
            logger.warning("No se identificaron elementos de resultados")
            if "no se encontraron" in self.driver.page_source.lower() or "sin resultados" in self.driver.page_source.lower():
                logger.info("La página indica explícitamente que no hay resultados")
            self._captura_debug(f"sin_resultados_p{numero_pagina}")
            return []

        # Procesar cada elemento de tesis encontrado
        for i, tesis in enumerate(elementos_tesis):
            try:
                # Estrategia mejorada: extraer toda la información de una vez
                datos_tesis = self._extraer_datos_tesis(tesis, i+1)
                resultados.append(datos_tesis)
            except Exception as e:
                logger.error(f"Error al procesar resultado {i+1}: {str(e)}")
                # Agregar un resultado parcial para no perderlo completamente
                texto_completo = "Error al procesar: " + str(e)
                try:
                    # Intentar obtener al menos el texto
                    texto_completo = tesis.text
                except:
                    pass

                resultados.append({
                    "numero": i + 1,
                    "registro_digital": "Error",
                    "texto_completo": texto_completo,
                    "error_procesamiento": True
                })

        return resultados

    def _extraer_datos_tesis(self, elemento_tesis, numero):
        """
        Extrae los datos estructurados de un elemento de tesis.

        Args:
            elemento_tesis (WebElement): Elemento que contiene la tesis
            numero (int): Número de resultado para identificación

        Returns:
            dict: Diccionario con la información estructurada de la tesis
        """
        # Inicializar con valores por defecto
        datos = {
            "numero": numero,
            "registro_digital": "No identificado",
            "texto_completo": "",
            "rubro": "",
            "precedentes": "",
            "fecha": "",
            "votacion": "",
            "instancia": "",
            "fuente": ""
        }

        # Obtener el texto completo
        try:
            datos["texto_completo"] = elemento_tesis.text.strip()
        except:
            logger.warning(f"No se pudo obtener texto completo del resultado {numero}")

            # Buscar registro digital con expresiones regulares
        import re
        texto_completo = datos["texto_completo"]

        # Patrones para extraer información clave
        patrones = {
            "registro_digital": r'Registro digital:?\s*(\d+)',
            "rubro": r'Rubro:?\s*([^Preced]\S[^\n]+)',
            "precedentes": r'Precedentes:?\s*([^\n]+)',
            "instancia": r'Instancia:?\s*([^\n]+)',
            "fecha": r'(\d{1,2}\s+de\s+\w+\s+de\s+\d{4})',
            "votacion": r'Votación:?\s*([^\n]+)'
        }

        # Aplicar los patrones de extracción
        for campo, patron in patrones.items():
            match = re.search(patron, texto_completo, re.IGNORECASE)
            if match:
                datos[campo] = match.group(1).strip()

        # Intentar extraer más información por estrategias específicas
        try:
            # Buscar el registro digital directamente en elementos específicos
            registro_elementos = elemento_tesis.find_elements(By.XPATH, ".//*[contains(text(), 'Registro') or contains(text(), 'registro')]")
            for elem in registro_elementos:
                texto_registro = elem.text
                match = re.search(r'\d{6,}', texto_registro)  # Buscar secuencia de 6+ dígitos
                if match:
                    datos["registro_digital"] = match.group()
                    break
        except:
            pass

        return datos

    @retry(max_attempts=3, delay=1.5, backoff=1.5,
            exceptions=(TimeoutException, ElementClickInterceptedException, StaleElementReferenceException))
    def _ir_siguiente_pagina_con_reintentos(self):
        """
        Navega a la siguiente página de resultados con reintentos automáticos.

        Returns:
            bool: True si se pudo navegar a la siguiente página, False en caso contrario
        """
        resultado = self._ir_siguiente_pagina()
        if resultado:
            self.estadisticas["reintentos_navegacion"] += 1
        return resultado

    def _ir_siguiente_pagina(self):
        """
        Navega a la siguiente página de resultados.

        Returns:
            bool: True si se pudo navegar a la siguiente página, False en caso contrario
        """
        logger.info("Intentando navegar a la siguiente página...")

        # Estrategia 1: Selectores directos para botón siguiente
        selectores_siguiente = [
            # XPATH para texto y contenido
            ("xpath", "//a[contains(text(), 'Siguiente') or contains(text(), 'siguiente') or contains(text(), '>')]"),
            ("xpath", "//button[contains(text(), 'Siguiente') or contains(text(), 'siguiente') or contains(text(), '>')]"),
            ("xpath", "//a[.//i[contains(@class, 'arrow-right') or contains(@class, 'chevron-right')]]"),
            ("xpath", "//button[.//i[contains(@class, 'arrow-right') or contains(@class, 'chevron-right')]]"),

            # CSS para clases comunes
            ("css", "a.next, button.next, a.siguiente, button.siguiente"),
            ("css", ".pagination .next a, .pagination .siguiente a"),
            ("css", ".pagination-next, .next-page, .siguiente-pagina")
        ]

        # Intentar cada selector
        for tipo_selector, selector in selectores_siguiente:
            try:
                elementos = []
                if tipo_selector == "xpath":
                    elementos = self._esperar_elementos(By.XPATH, selector, timeout=2)
                else:
                    elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=2)

                if elementos:
                    # Filtrar solo elementos no deshabilitados
                    elementos_habilitados = [
                        elem for elem in elementos
                        if not (elem.get_attribute("disabled") == "true" or
                                "disabled" in (elem.get_attribute("class") or "") or
                                elem.get_attribute("aria-disabled") == "true")
                    ]

                    if elementos_habilitados:
                        boton_siguiente = elementos_habilitados[0]
                        logger.info(f"Botón siguiente encontrado con {tipo_selector}: {selector}")

                        # Hacer clic con manejo de errores
                        if self._clic_seguro(boton_siguiente):
                            logger.info("Navegación a siguiente página exitosa")
                            return True
            except Exception as e:
                logger.debug(f"Selector {selector} no encontró elementos válidos: {str(e)}")

        # Estrategia 2: Análisis de paginación actual
        try:
            # Intentar encontrar el elemento activo y su siguiente
            script_paginacion = """
            function encontrarSiguientePagina() {
                // Buscar elementos de paginación
                const elementosPaginacion = document.querySelectorAll('.pagination li, .paginacion li, nav[aria-label*="pagin"] li');

                if (elementosPaginacion.length === 0) return null;

                // Encontrar el elemento activo
                let elementoActivo = null;
                let indiceActivo = -1;

                for (let i = 0; i < elementosPaginacion.length; i++) {
                    const elem = elementosPaginacion[i];
                    if (elem.classList.contains('active') ||
                        elem.classList.contains('current') ||
                        elem.classList.contains('selected') ||
                        elem.getAttribute('aria-current') === 'page') {
                        elementoActivo = elem;
                        indiceActivo = i;
                        break;
                    }
                }

                // Si encontramos elemento activo, buscar el siguiente
                if (elementoActivo && indiceActivo >= 0 && indiceActivo + 1 < elementosPaginacion.length) {
                    const posibleSiguiente = elementosPaginacion[indiceActivo + 1];

                    // Verificar que no esté deshabilitado
                    if (!posibleSiguiente.classList.contains('disabled') &&
                        posibleSiguiente.getAttribute('aria-disabled') !== 'true') {

                        // Buscar el enlace dentro del elemento
                        const enlace = posibleSiguiente.querySelector('a, button');
                        return enlace || posibleSiguiente;
                    }
                }

                return null;
            }
            return encontrarSiguientePagina();
            """

            boton_paginacion = self.driver.execute_script(script_paginacion)
            if boton_paginacion:
                logger.info("Botón siguiente encontrado mediante análisis de paginación")
                if self._clic_seguro(boton_paginacion):
                    logger.info("Navegación a siguiente página exitosa (análisis de paginación)")
                    return True
        except Exception as e:
            logger.debug(f"Error en análisis de paginación: {str(e)}")

        # Estrategia 3: Modificar URL directamente basado en parámetros de página
        try:
            url_actual = self.driver.current_url

            # Buscar parámetros de paginación comunes
            import re
            patrones_pagina = [
                r'[?&]p(?:age|agina|g)=(\d+)',
                r'[?&]pa=(\d+)',
                r'[?&]page=(\d+)',
                r'[/]pagina[/](\d+)',
                r'[/]page[/](\d+)'
            ]

            pagina_actual = None
            for patron in patrones_pagina:
                match = re.search(patron, url_actual)
                if match:
                    pagina_actual = int(match.group(1))
                    logger.info(f"Página actual identificada en URL: {pagina_actual}")

                    # Incrementar el número de página
                    pagina_siguiente = pagina_actual + 1
                    nueva_url = re.sub(patron, lambda m: m.group(0).replace(m.group(1), str(pagina_siguiente)), url_actual)

                    if nueva_url != url_actual:
                        logger.info(f"Navegando a siguiente página mediante URL: {nueva_url}")
                        self.driver.get(nueva_url)
                        return True

            # Si no encontramos parámetro explícito pero vemos números de página en URL
            if not pagina_actual:
                # Buscar cualquier número que pudiera ser página
                match_num = re.search(r'(/|=)(\d+)(/|$|&)', url_actual)
                if match_num:
                    num_actual = int(match_num.group(2))
                    # Solo considerar como página si es un número razonable (1-100)
                    if 1 <= num_actual <= 100:
                        num_siguiente = num_actual + 1
                        nueva_url = url_actual.replace(match_num.group(0),
                                                      f"{match_num.group(1)}{num_siguiente}{match_num.group(3)}")
                        logger.info(f"Navegando a posible siguiente página mediante URL: {nueva_url}")
                        self.driver.get(nueva_url)
                        return True
        except Exception as e:
            logger.debug(f"Error al modificar URL para siguiente página: {str(e)}")

        # Estrategia 4: Último recurso - buscar cualquier elemento con texto o icono de "siguiente"
        try:
            script_ultimo_recurso = """
            function buscarCualquierSiguiente() {
                // Textos que podrían indicar "siguiente"
                const textosSiguiente = ['siguiente', 'next', 'sig', '>', '>>'];

                // Buscar elementos con estos textos
                const elementos = document.querySelectorAll('a, button, div, span');

                for (const elem of elementos) {
                    const textoElem = (elem.textContent || '').toLowerCase().trim();

                    // Si el texto coincide
                    if (textosSiguiente.some(t => textoElem.includes(t))) {
                        // Verificar que no esté deshabilitado
                        if (!elem.disabled &&
                            !elem.classList.contains('disabled') &&
                            elem.getAttribute('aria-disabled') !== 'true') {
                            return elem;
                        }
                    }

                    // Buscar también por íconos
                    if (elem.innerHTML.includes('arrow-right') ||
                        elem.innerHTML.includes('chevron-right') ||
                        elem.innerHTML.includes('fa-angle-right')) {
                        if (!elem.disabled &&
                            !elem.classList.contains('disabled') &&
                            elem.getAttribute('aria-disabled') !== 'true') {
                            return elem;
                        }
                    }
                }

                return null;
            }
            return buscarCualquierSiguiente();
            """

            elemento_ultimo_recurso = self.driver.execute_script(script_ultimo_recurso)
            if elemento_ultimo_recurso:
                logger.info("Encontrado posible elemento 'siguiente' mediante último recurso")
                if self._clic_seguro(elemento_ultimo_recurso, usar_js=True):
                    logger.info("Navegación a siguiente página exitosa (último recurso)")
                    return True
        except Exception as e:
            logger.debug(f"Error en estrategia de último recurso: {str(e)}")

        logger.warning("No se encontró ningún mecanismo para navegar a la siguiente página")
        return False

    def guardar_resultados(self, resultados, archivo_salida="tesis_scjn.txt", incluir_estadisticas=True):
        """
        Guarda los resultados en un archivo de texto.

        Args:
            resultados (list): Lista de diccionarios con los resultados
            archivo_salida (str): Nombre del archivo de salida
            incluir_estadisticas (bool): Si se deben incluir estadísticas de la ejecución

        Returns:
            bool: True si se guardó correctamente, False en caso contrario
        """
        try:
            with open(archivo_salida, 'w', encoding='utf-8') as f:
                f.write(f"RESULTADOS DE BÚSQUEDA - TESIS SCJN\n")
                f.write(f"Fecha y hora: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                f.write(f"Total de resultados: {len(resultados)}\n")

                # Incluir estadísticas si está habilitado
                if incluir_estadisticas:
                    tiempo_total = time.time() - self.estadisticas["tiempo_inicio"]
                    f.write(f"Tiempo de ejecución: {tiempo_total:.2f} segundos\n")
                    f.write(f"Páginas procesadas: {self.estadisticas['paginas_procesadas']}\n")
                    f.write(f"Reintentos de navegación: {self.estadisticas['reintentos_navegacion']}\n")

                f.write("\n" + "="*80 + "\n\n")

                for resultado in resultados:
                    f.write(f"RESULTADO #{resultado['numero']}\n")
                    f.write(f"Registro digital: {resultado['registro_digital']}\n")

                    # Incluir campos adicionales si están disponibles
                    campos_adicionales = ['rubro', 'instancia', 'fecha', 'votacion', 'precedentes', 'fuente']
                    for campo in campos_adicionales:
                        if campo in resultado and resultado[campo]:
                            f.write(f"{campo.capitalize()}: {resultado[campo]}\n")

                    f.write("\nTEXTO COMPLETO:\n")
                    f.write(f"{resultado['texto_completo']}\n\n")
                    f.write("-" * 80 + "\n\n")

            logger.info(f"Resultados guardados exitosamente en '{archivo_salida}'")
            return True
        except Exception as e:
            logger.error(f"ERROR al guardar resultados: {str(e)}")
            return False

    def guardar_resultados_json(self, resultados, archivo_salida="tesis_scjn.json"):
        """
        Guarda los resultados en formato JSON para procesamiento posterior.

        Args:
            resultados (list): Lista de diccionarios con los resultados
            archivo_salida (str): Nombre del archivo de salida

        Returns:
            bool: True si se guardó correctamente, False en caso contrario
        """
        try:
            datos_salida = {
                "metadata": {
                    "fecha": datetime.now().isoformat(),
                    "total_resultados": len(resultados),
                    "tiempo_ejecucion": time.time() - self.estadisticas["tiempo_inicio"],
                    "paginas_procesadas": self.estadisticas["paginas_procesadas"],
                    "reintentos_navegacion": self.estadisticas["reintentos_navegacion"]
                },
                "resultados": resultados
            }

            with open(archivo_salida, 'w', encoding='utf-8') as f:
                json.dump(datos_salida, f, ensure_ascii=False, indent=2)

            logger.info(f"Resultados guardados en formato JSON en '{archivo_salida}'")
            return True
        except Exception as e:
            logger.error(f"ERROR al guardar resultados en JSON: {str(e)}")
            return False

    def cerrar(self):
        """Cierra el driver y libera recursos"""
        if hasattr(self, 'driver') and self.driver:
            try:
                self.driver.quit()
                logger.info("Driver cerrado correctamente")
            except Exception as e:
                logger.error(f"Error al cerrar el driver: {str(e)}")


def ejecutar_scraper(termino_busqueda, headless=False, archivo_salida="tesis_scjn.txt",
                    numero_resultados=50, guardar_json=True, modo_debug=False,
                    continuar_cache=True, timeout=15):
    """
    Ejecuta el scraper con los parámetros especificados y manejo de errores mejorado.

    Args:
        termino_busqueda (str): Término a buscar
        headless (bool): Si se ejecuta en modo headless
        archivo_salida (str): Nombre del archivo de salida de texto
        numero_resultados (int): Número de resultados a obtener en total
        guardar_json (bool): Si se guardan también los resultados en formato JSON
        modo_debug (bool): Activa capturas de pantalla para depuración
        continuar_cache (bool): Intenta continuar desde un caché previo si existe
        timeout (int): Tiempo de espera máximo en segundos para operaciones

    Returns:
        bool: True si se completó correctamente, False en caso contrario
    """
    inicio_tiempo = time.time()
    resultados = []

    # Configurar archivo de caché basado en el término de búsqueda (normalizado)
    import hashlib
    cache_key = hashlib.md5(termino_busqueda.encode('utf-8')).hexdigest()[:10]
    cache_archivo = f"cache_scjn_{cache_key}.json"

    scraper = None
    exito = False

    try:
        logger.info(f"=== INICIANDO BÚSQUEDA DE TESIS JUDICIALES ===")
        logger.info(f"Término: {termino_busqueda}")
        logger.info(f"Configuración: {numero_resultados} resultados, headless={headless}, debug={modo_debug}")

        # Inicializar y ejecutar el scraper
        scraper = ScraperTesisSCJN(
            headless=headless,
            modo_debug=modo_debug,
            timeout_default=timeout,
            cache_archivo=cache_archivo
        )

        # Realizar la búsqueda con manejo de excepciones
        resultados = scraper.buscar_tesis(
            termino_busqueda,
            numero_resultados=numero_resultados,
            continuar_cache=continuar_cache
        )

        # Verificar si se obtuvieron resultados
        if not resultados:
            logger.warning("No se encontraron resultados para la búsqueda")
            return False

        logger.info(f"Se encontraron {len(resultados)} resultados")

        # Guardar resultados en formato texto
        exito_txt = scraper.guardar_resultados(resultados, archivo_salida)

        # Guardar en JSON si está habilitado
        exito_json = True
        if guardar_json:
            archivo_json = archivo_salida.rsplit('.', 1)[0] + '.json'
            exito_json = scraper.guardar_resultados_json(resultados, archivo_json)

        exito = exito_txt and exito_json

        # Mostrar tiempo total
        tiempo_total = time.time() - inicio_tiempo
        logger.info(f"Proceso completado en {tiempo_total:.2f} segundos")

        return exito
    except RetryError as e:
        logger.error(f"ERROR: Se agotaron los reintentos: {str(e)}")
        return False
    except Exception as e:
        logger.error(f"ERROR general en la ejecución: {str(e)}", exc_info=True)
        return False
    finally:
        # Asegurar que siempre se cierre el driver
        if scraper:
            scraper.cerrar()

        # Si obtuvimos resultados parciales pero hubo error, guardarlos de todos modos
        if not exito and resultados:
            logger.info(f"Guardando {len(resultados)} resultados parciales obtenidos antes del error...")
            try:
                archivo_parcial = "PARCIAL_" + archivo_salida
                with open(archivo_parcial, 'w', encoding='utf-8') as f:
                    f.write(f"RESULTADOS PARCIALES - Error durante ejecución\n")
                    f.write(f"Fecha y hora: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                    f.write(f"Total de resultados parciales: {len(resultados)}\n\n")

                    for resultado in resultados:
                        f.write(f"{resultado['numero']}. Registro digital: {resultado['registro_digital']}\n")
                        f.write(f"{resultado['texto_completo']}\n\n")
                        f.write("-" * 80 + "\n\n")

                logger.info(f"Resultados parciales guardados en '{archivo_parcial}'")
            except Exception as e:
                logger.error(f"No se pudieron guardar los resultados parciales: {str(e)}")

# Importar la funcionalidad de búsqueda compuesta
try:
    from scraper_compuesto import ejecutar_scraper_compuesto
    BUSQUEDA_COMPUESTA_DISPONIBLE = True
except ImportError:
    BUSQUEDA_COMPUESTA_DISPONIBLE = False
    print("AVISO: El módulo de búsqueda compuesta no está disponible.")
# Ejemplo de uso con manejo de opciones de línea de comandos
if __name__ == "__main__":
    import argparse
    import sys

    # Verificar si se proporcionaron argumentos
    if len(sys.argv) > 1:
        parser = argparse.ArgumentParser(description='Scraper de Tesis de la SCJN')
        parser.add_argument('--termino', '-t', required=True, help='Término de búsqueda')
        parser.add_argument('--refinamiento', '-r', help='Término para refinar la búsqueda (opcional)')
        parser.add_argument('--archivo', '-o', default='tesis_scjn.txt', help='Archivo de salida')
        parser.add_argument('--resultados', '-n', type=int, default=50, help='Número de resultados a obtener')
        parser.add_argument('--headless', '-H', action='store_true', help='Ejecutar en modo sin interfaz gráfica')
        parser.add_argument('--debug', '-d', action='store_true', help='Activar modo de depuración')
        parser.add_argument('--timeout', type=int, default=15, help='Tiempo máximo de espera en segundos')
        parser.add_argument('--no-json', action='store_true', help='No guardar en formato JSON')

        args = parser.parse_args()

        # Determinar si usar búsqueda compuesta o simple
        if args.refinamiento and BUSQUEDA_COMPUESTA_DISPONIBLE:
            print(f"Ejecutando búsqueda compuesta con refinamiento: '{args.refinamiento}'")
            exito = ejecutar_scraper_compuesto(
                termino_general=args.termino,
                termino_refinamiento=args.refinamiento,
                headless=args.headless,
                archivo_salida=args.archivo,
                numero_resultados=args.resultados,
                guardar_json=not args.no_json,
                modo_debug=args.debug,
                timeout=args.timeout
            )
        else:
            print("Ejecutando búsqueda simple")
            exito = ejecutar_scraper(
                termino_busqueda=args.termino,
                headless=args.headless,
                archivo_salida=args.archivo,
                numero_resultados=args.resultados,
                guardar_json=not args.no_json,
                modo_debug=args.debug,
                continuar_cache=True,
                timeout=args.timeout
            )

    # OPCIÓN 2: Modo interactivo (si no hay argumentos)
    else:
        print("=== SCRAPER DE TESIS JUDICIALES SCJN ===")
        print("Ejecutando en modo interactivo\n")

        # Solicitar valores al usuario o usar valores predeterminados
        try:
            termino_busqueda = input("Ingrese el término de búsqueda: ")
            if not termino_busqueda:
                print("ERROR: El término de búsqueda es obligatorio.")
                sys.exit(1)

            # Preguntar si quiere usar búsqueda compuesta
            usar_compuesta = False
            if BUSQUEDA_COMPUESTA_DISPONIBLE:
                compuesta_input = input("¿Desea utilizar búsqueda compuesta (refinamiento)? (s/n) [n]: ").lower()
                usar_compuesta = compuesta_input.startswith('s')

            termino_refinamiento = None
            if usar_compuesta:
                termino_refinamiento = input("Término de refinamiento: ")
                if not termino_refinamiento:
                    print("NOTA: No se especificó término de refinamiento, se realizará búsqueda simple")
                    usar_compuesta = False

            archivo_salida = input("Nombre del archivo de salida [tesis_scjn.txt]: ")
            if not archivo_salida:
                archivo_salida = "tesis_scjn.txt"

            num_input = input("Número de resultados a obtener [50]: ")
            numero_resultados = int(num_input) if num_input else 50

            headless_input = input("Ejecutar en modo sin interfaz gráfica (s/n) [n]: ").lower()
            headless = headless_input.startswith('s')

            debug_input = input("Activar modo de depuración (s/n) [n]: ").lower()
            modo_debug = debug_input.startswith('s')

            json_input = input("Guardar también en formato JSON (s/n) [s]: ").lower()
            guardar_json = not json_input.startswith('n')

            timeout_input = input("Tiempo máximo de espera en segundos [15]: ")
            timeout = int(timeout_input) if timeout_input else 15

            # Ejecutar según el tipo de búsqueda seleccionado
            if usar_compuesta:
                print(f"\nIniciando búsqueda COMPUESTA:")
                print(f"- Término general: '{termino_busqueda}'")
                print(f"- Refinamiento: '{termino_refinamiento}'")

                exito = ejecutar_scraper_compuesto(
                    termino_general=termino_busqueda,
                    termino_refinamiento=termino_refinamiento,
                    headless=headless,
                    archivo_salida=archivo_salida,
                    numero_resultados=numero_resultados,
                    guardar_json=guardar_json,
                    modo_debug=modo_debug,
                    timeout=timeout
                )
            else:
                print(f"\nIniciando búsqueda SIMPLE:")
                print(f"- Término: '{termino_busqueda}'")

                exito = ejecutar_scraper(
                    termino_busqueda=termino_busqueda,
                    headless=headless,
                    archivo_salida=archivo_salida,
                    numero_resultados=numero_resultados,
                    guardar_json=guardar_json,
                    modo_debug=modo_debug,
                    continuar_cache=True,
                    timeout=timeout
                )

        except KeyboardInterrupt:
            print("\nOperación cancelada por el usuario.")
            sys.exit(0)
        except Exception as e:
            print(f"\nError en la entrada de datos: {str(e)}")
            sys.exit(1)

    # Mostrar resultados
    if exito:
        print(f"\n¡Proceso completado con éxito!")
        print(f"Los resultados se han guardado en: {archivo_salida}")
        if guardar_json:
            print(f"Los resultados en formato JSON se han guardado en: {archivo_salida.rsplit('.', 1)[0] + '.json'}")
    else:
        print("\nEl proceso no se completó correctamente. Revise el archivo de log para más detalles.")
        print("Puede encontrar resultados parciales (si existen) en archivos con prefijo 'PARCIAL_'.")