"""
Scraper Compuesto para Tesis de la SCJN
Permite realizar búsquedas en dos etapas: general + refinamiento
Genera URLs para acceder directamente a cada tesis encontrada
Implementa descarga directa del listado de resultados oficial
"""

import os
import sys
import time
import random
import logging
import json
import traceback
import re
from datetime import datetime
import tempfile  # Para archivos temporales
import fitz  # PyMuPDF - Para procesar PDFs
import pandas as pd  # Para procesar Excel/CSV
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait, Select
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import (
    TimeoutException,
    NoSuchElementException,
    StaleElementReferenceException,
    ElementClickInterceptedException,
    WebDriverException
)

# Configuración de logging
logging.basicConfig(level=logging.INFO)

# Configurar loggers específicos
main_logger = logging.getLogger("MainScript")
main_logger.setLevel(logging.INFO)

scraper_logger = logging.getLogger("ScraperFunciones")
scraper_logger.setLevel(logging.INFO)

ejecutor_logger = logging.getLogger("EjecutorScraper")
ejecutor_logger.setLevel(logging.INFO)

# Configurar handlers si no existen
if not main_logger.handlers:
    formatter = logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s')

    # Handler para archivo
    file_handler = logging.FileHandler("scraper_compuesto.log")
    file_handler.setFormatter(formatter)

    # Handler para consola
    console_handler = logging.StreamHandler()
    console_handler.setFormatter(formatter)

    # Agregar handlers a los loggers
    for logger in [main_logger, scraper_logger, ejecutor_logger]:
        logger.addHandler(file_handler)
        logger.addHandler(console_handler)

# Intentar importar la clase base del scraper original
try:
    # Importar desde archivo original si existe
    from scraper_scjn import ScraperTesisSCJN
    USING_ORIGINAL_CLASS = True
    scraper_logger.info("Importada clase ScraperTesisSCJN original correctamente")
except ImportError:
    USING_ORIGINAL_CLASS = False
    print("ADVERTENCIA: Usando clase base mínima ScraperTesisSCJNBase.")
    print("Asegúrate de tener el scraper_scjn.py original para funcionalidad completa.")

    # Clase base mínima para cuando no se puede importar la original
    class ScraperTesisSCJNBase:
        """
        Implementación mínima de la clase base para cuando no está disponible la original.
        Incluye solo las funcionalidades esenciales para que funcione el scraper compuesto.
        """
        def __init__(self, headless=False, modo_debug=False, timeout_default=15):
            self.url_base = "https://sjf2.scjn.gob.mx/busqueda-principal-tesis"
            self.modo_debug = modo_debug
            self.timeout_default = timeout_default
            self.estadisticas = {
                "paginas_procesadas": 0,
                "resultados_obtenidos": 0,
                "reintentos_navegacion": 0,
                "tiempo_inicio": time.time()
            }
            self.driver = self._configurar_driver(headless)

        def _configurar_driver(self, headless):
            """Configura y devuelve una instancia de WebDriver"""
            try:
                # Importar el helper para el driver
                import sys
                import os

                # Determinar si estamos en un ejecutable o script
                if getattr(sys, 'frozen', False):
                    base_path = sys._MEIPASS
                else:
                    base_path = os.path.dirname(os.path.abspath(__file__))

                # Importar dinámicamente el helper
                sys.path.insert(0, base_path)

                # Importar la función desde el módulo driver_helper
                from driver_helper import get_chrome_driver

                scraper_logger.info("Inicializando WebDriver Chrome con driver_helper...")
                driver = get_chrome_driver(headless=headless)
                driver.set_page_load_timeout(self.timeout_default * 2)
                return driver
            except Exception as e:
                scraper_logger.error(f"Error al inicializar el WebDriver: {str(e)}")
                raise RuntimeError(f"No se pudo inicializar el WebDriver: {str(e)}")

        def _captura_debug(self, nombre):
            """Guarda una captura de pantalla para depuración si el modo debug está activado"""
            if not self.modo_debug:
                return

            try:
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                filename = f"debug_{nombre}_{timestamp}.png"
                self.driver.save_screenshot(filename)
                scraper_logger.debug(f"Captura de pantalla guardada: {filename}")
            except Exception as e:
                scraper_logger.warning(f"No se pudo guardar captura de depuración: {str(e)}")

        def _esperar_elemento(self, by, valor, timeout=None, condicion=EC.presence_of_element_located, mensaje=None):
            """Espera y retorna un elemento cuando está disponible"""
            if timeout is None:
                timeout = self.timeout_default

            mensaje = mensaje or f"Elemento no encontrado: {by}='{valor}' después de {timeout}s"

            try:
                return WebDriverWait(self.driver, timeout).until(
                    condicion((by, valor))
                )
            except TimeoutException:
                scraper_logger.warning(mensaje)
                self._captura_debug(f"timeout_{by}_{valor.replace('/', '_')}")
                raise TimeoutException(mensaje)

        def _esperar_elementos(self, by, valor, timeout=None, condicion=EC.presence_of_all_elements_located):
            """Espera y retorna múltiples elementos cuando están disponibles"""
            if timeout is None:
                timeout = self.timeout_default

            try:
                return WebDriverWait(self.driver, timeout).until(
                    condicion((by, valor))
                )
            except TimeoutException:
                scraper_logger.debug(f"No se encontraron elementos con {by}='{valor}'")
                return []

        def _clic_seguro(self, elemento, timeout=2, usar_js=False):
            """Realiza un clic seguro en un elemento con manejo de excepciones"""
            try:
                # Asegurar que el elemento es visible
                self.driver.execute_script("arguments[0].scrollIntoView({block: 'center'});", elemento)
                time.sleep(0.5)  # Pequeña pausa para scroll

                if usar_js:
                    self.driver.execute_script("arguments[0].click();", elemento)
                    scraper_logger.debug("Clic realizado con JavaScript")
                    return True

                try:
                    elemento.click()
                    scraper_logger.debug("Clic normal realizado con éxito")
                    return True
                except (ElementClickInterceptedException, StaleElementReferenceException) as e:
                    scraper_logger.debug(f"Error en clic normal: {str(e)}, intentando con JS")
                    time.sleep(timeout)
                    self.driver.execute_script("arguments[0].click();", elemento)
                    scraper_logger.debug("Clic con JavaScript realizado después de error")
                    return True

            except Exception as e:
                scraper_logger.warning(f"Error al hacer clic en elemento: {str(e)}")
                self._captura_debug("error_clic")
                return False

        def generar_url_registro(self, registro_digital):
            """
            Genera una URL para acceder a la información de una tesis basada en su registro digital.

            Args:
                registro_digital (str): Número de registro digital de la tesis

            Returns:
                str: URL completa para acceder a la tesis
            """
            plantilla_url = ("https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/reporte/"
                             "{registro}?nameDocto=Tesis&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020")
            return plantilla_url.format(registro=registro_digital)

        def _extraer_resultados(self, numero_pagina):
            """Método básico para extraer resultados de la página actual"""
            resultados = []
            elementos_tesis = self.driver.find_elements(
                By.CSS_SELECTOR,
                ".registro-item, .resultado-item, div[class*='resultado']"
            )

            scraper_logger.info(f"Encontrados {len(elementos_tesis)} elementos de resultado en página {numero_pagina}")

            for i, tesis in enumerate(elementos_tesis):
                # Procesamiento básico de cada resultado
                try:
                    texto_tesis = tesis.text

                    # Intentar extraer registro digital con regex básico
                    import re
                    registro = "No identificado"
                    match = re.search(r'Registro digital:?\s*(\d+)', texto_tesis)
                    if match:
                        registro = match.group(1)

                    # Agregar a resultados
                    resultados.append({
                        "numero": i + 1,
                        "registro_digital": registro,
                        "texto_completo": texto_tesis
                    })
                except Exception as e:
                    scraper_logger.error(f"Error procesando resultado {i+1}: {str(e)}")

            return resultados

        def _ir_siguiente_pagina(self):
            """Método básico para navegar a la siguiente página"""
            try:
                # Buscar botón de siguiente por texto
                botones = self.driver.find_elements(
                    By.XPATH,
                    "//a[contains(text(), 'Siguiente')] | //button[contains(text(), 'Siguiente')]"
                )

                if botones:
                    self._clic_seguro(botones[0])
                    scraper_logger.info("Navegado a siguiente página")
                    time.sleep(2)  # Esperar carga
                    return True

                # Intentar encontrar por ícono
                iconos = self.driver.find_elements(
                    By.CSS_SELECTOR, "i.fa-chevron-right, i.fa-arrow-right"
                )
                if iconos:
                    self._clic_seguro(iconos[0].find_element(By.XPATH, ".."))
                    scraper_logger.info("Navegado a siguiente página (por ícono)")
                    time.sleep(2)
                    return True

                scraper_logger.info("No se encontró elemento de paginación siguiente")
                return False
            except Exception as e:
                scraper_logger.error(f"Error navegando a siguiente página: {str(e)}")
                return False

        def guardar_resultados(self, resultados, archivo_salida="tesis_scjn.txt", incluir_estadisticas=True):
            """Guarda los resultados en un archivo de texto"""
            try:
                with open(archivo_salida, 'w', encoding='utf-8') as f:
                    f.write("RESULTADOS DE BÚSQUEDA - TESIS SCJN\n")
                    f.write(f"Fecha y hora: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
                    f.write(f"Total de resultados: {len(resultados)}\n\n")

                    for resultado in resultados:
                        f.write(f"{resultado['numero']}. Registro digital: {resultado['registro_digital']}\n")
                        f.write(f"URL: {resultado.get('url_tesis', '')}\n")
                        f.write(f"{resultado['texto_completo']}\n\n")
                        f.write("-" * 80 + "\n\n")

                scraper_logger.info(f"Resultados guardados exitosamente en '{archivo_salida}'")
                return True
            except Exception as e:
                scraper_logger.error(f"ERROR al guardar resultados: {str(e)}")
                return False

        def guardar_resultados_json(self, resultados, archivo_salida="tesis_scjn.json"):
            """Guarda los resultados en formato JSON"""
            try:
                datos_salida = {
                    "metadata": {
                        "fecha": datetime.now().isoformat(),
                        "total_resultados": len(resultados),
                        "tiempo_ejecucion": time.time() - self.estadisticas["tiempo_inicio"]
                    },
                    "resultados": resultados
                }

                with open(archivo_salida, 'w', encoding='utf-8') as f:
                    json.dump(datos_salida, f, ensure_ascii=False, indent=2)

                scraper_logger.info(f"Resultados guardados en formato JSON en '{archivo_salida}'")
                return True
            except Exception as e:
                scraper_logger.error(f"ERROR al guardar resultados en JSON: {str(e)}")
                return False

        def guardar_solo_urls(self, resultados, archivo_salida="tesis_scjn_urls.txt"):
            """
            Guarda únicamente las URLs generadas a partir de los registros digitales.

            Args:
                resultados (list): Lista de resultados procesados
                archivo_salida (str): Ruta del archivo de salida

            Returns:
                bool: True si se guardó correctamente, False en caso contrario
            """
            try:
                with open(archivo_salida, 'w', encoding='utf-8') as f:
                    for resultado in resultados:
                        if resultado.get('registro_digital') != "No identificado":
                            # Usar la URL ya generada si existe, o generarla ahora
                            url = resultado.get('url_tesis') or self.generar_url_registro(resultado['registro_digital'])
                            f.write(f"{url}\n")

                scraper_logger.info(f"Lista de URLs guardada en: {archivo_salida}")
                return True
            except Exception as e:
                scraper_logger.error(f"Error al guardar lista de URLs: {str(e)}")
                return False

        def cerrar(self):
            """Cierra el driver y libera recursos"""
            try:
                if hasattr(self, 'driver') and self.driver:
                    self.driver.quit()
                    scraper_logger.info("Driver cerrado correctamente")
            except Exception as e:
                scraper_logger.error(f"Error al cerrar el driver: {str(e)}")


# Determinar qué clase base usar
ScraperBase = ScraperTesisSCJN if USING_ORIGINAL_CLASS else ScraperTesisSCJNBase


class ScraperTesisCompuesto(ScraperBase):
    """
    Clase para realizar scraping de tesis de la SCJN con capacidad de
    búsqueda compuesta (primero término general, luego refinamiento).
    """

    def __init__(self, headless=False, modo_debug=False, timeout_default=15, tipo_tesis=0):
        # Llamar al constructor de la clase base
        super().__init__(headless=headless, modo_debug=modo_debug, timeout_default=timeout_default)
        # Configurar tipo de tesis
        self.tipo_tesis = tipo_tesis
        # Usar descarga directa por defecto (OPTIMIZACIÓN)
        self.usar_descarga_directa = True

    def generar_url_registro(self, registro_digital):
        """
        Genera una URL para acceder a la información de una tesis basada en su registro digital.

        Args:
            registro_digital (str): Número de registro digital de la tesis

        Returns:
            str: URL completa para acceder a la tesis
        """
        plantilla_url = ("https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/reporte/"
                         "{registro}?nameDocto=Tesis&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020")
        return plantilla_url.format(registro=registro_digital)

    def guardar_solo_urls(self, resultados, archivo_salida="tesis_scjn_urls.txt"):
        """
        Guarda únicamente las URLs generadas a partir de los registros digitales.

        Args:
            resultados (list): Lista de resultados procesados
            archivo_salida (str): Ruta del archivo de salida

        Returns:
            bool: True si se guardó correctamente, False en caso contrario
        """
        try:
            with open(archivo_salida, 'w', encoding='utf-8') as f:
                for resultado in resultados:
                    if resultado.get('registro_digital') != "No identificado":
                        url = resultado.get('url_tesis') or self.generar_url_registro(resultado['registro_digital'])
                        f.write(f"{url}\n")

            scraper_logger.info(f"Lista de URLs guardada en: {archivo_salida}")
            return True
        except Exception as e:
            scraper_logger.error(f"Error al guardar lista de URLs: {str(e)}")
            return False

    def descargar_listado_resultados(self):
        """
        Localiza y activa el botón 'Descargar listado de resultados' para obtener
        directamente la tabla de resultados en formato PDF/Excel/CSV.
        MEJORADO: Mejor manejo de la ventana modal y localización de botones.
        
        Returns:
            str: Ruta al archivo descargado, o None en caso de error
        """
        scraper_logger.info("Intentando descargar listado de resultados directamente...")
        inicio_tiempo = time.time()
        
        try:
            # Configurar directorio temporal para descargas
            directorio_temp = tempfile.mkdtemp()
            scraper_logger.info(f"Directorio temporal para descargas: {directorio_temp}")
            
            # Configurar comportamiento de descarga (para Chrome)
            try:
                # Para Selenium 4+
                self.driver.execute_cdp_cmd("Page.setDownloadBehavior", {
                    "behavior": "allow",
                    "downloadPath": directorio_temp
                })
            except Exception as e:
                scraper_logger.warning(f"No se pudo configurar CDP para descargas: {str(e)}")
                # Intentar configurar prefs en opciones
                try:
                    options = webdriver.ChromeOptions()
                    prefs = {"download.default_directory": directorio_temp}
                    self.driver.execute_cdp_cmd('Page.setDownloadBehavior', {
                        'behavior': 'allow',
                        'downloadPath': directorio_temp
                    })
                except Exception as e2:
                    scraper_logger.warning(f"Tampoco se pudo configurar opciones de descarga: {str(e2)}")
                    # Continuaremos de todas formas
            
            # MEJORA: Conjunto ampliado de selectores para identificar el botón de descarga principal
            selectores_boton = [
                # Selectores CSS específicos
                "button[ngbtooltip='Descargar listado de resultados']",
                "button.mat-focus-indicator.float-right",
                "button.mat-icon-button[ngbtooltip='Descargar listado de resultados']",
                "button.btn-scjn.mat-icon-button",
                "button.mat-button-base:has(fa-icon[icon='file-download'])",
                "button.mat-icon-button:has(span.mat-button-wrapper > fa-icon[icon='file-download'])",
                "button.mat-icon-button:has(fa-icon.ng-fa-icon[icon='file-download'])",
                # Selectores XPath específicos
                "//button[contains(@ngbtooltip, 'Descargar')]",
                "//button[contains(@aria-label, 'Descargar')]",
                "//button[.//fa-icon[@icon='file-download']]",
                "//button[contains(@class, 'mat-icon-button') and contains(@ngbtooltip, 'Descargar')]",
                "//button[contains(@class, 'download') or contains(@id, 'download')]",
                "//fa-icon[@icon='file-download']/ancestor::button",
                "//button[.//span[contains(text(), 'Descargar')]]"
            ]
            
            # Intentar localizar el botón con diferentes selectores
            boton_descarga = None
            for selector in selectores_boton:
                elementos = []
                try:
                    if selector.startswith("//"):
                        elementos = self.driver.find_elements(By.XPATH, selector)
                    else:
                        elementos = self.driver.find_elements(By.CSS_SELECTOR, selector)
                except Exception:
                    continue
                    
                if elementos:
                    boton_descarga = elementos[0]
                    scraper_logger.info(f"Botón de descarga encontrado con selector: {selector}")
                    break
                    
            # Si no encontramos con selectores específicos, usar JavaScript mejorado
            if not boton_descarga:
                scraper_logger.info("Intentando localizar botón mediante JavaScript...")
                script = """
                function encontrarBotonDescargar() {
                    // Buscar por texto en botones
                    const botones = Array.from(document.querySelectorAll('button'));
                    
                    // 1. Primero buscar por tooltip o aria-label
                    for (const btn of botones) {
                        const tooltip = btn.getAttribute('ngbtooltip') || btn.getAttribute('mattooltip') || btn.getAttribute('aria-label') || '';
                        if (tooltip.toLowerCase().includes('descargar listado')) {
                            return btn;
                        }
                    }
                    
                    // 2. Buscar por ícono de descarga
                    const botonesConIconoDescarga = botones.filter(btn => {
                        return btn.querySelector('fa-icon[icon="file-download"]') || 
                               btn.querySelector('i.fa-download') ||
                               btn.querySelector('mat-icon[svgicon="cloud_download"]') ||
                               btn.innerHTML.includes('file-download') ||
                               btn.innerHTML.includes('fa-download');
                    });
                    
                    if (botonesConIconoDescarga.length > 0) {
                        return botonesConIconoDescarga[0];
                    }
                    
                    // 3. Buscar cualquier botón con texto relacionado con descarga
                    for (const btn of botones) {
                        const texto = (btn.textContent || '').toLowerCase();
                        if (texto.includes('descargar') || texto.includes('listado') || texto.includes('exportar')) {
                            return btn;
                        }
                    }
                    
                    return null;
                }
                return encontrarBotonDescargar();
                """
                try:
                    boton_descarga = self.driver.execute_script(script)
                    if boton_descarga:
                        scraper_logger.info("Botón de descarga encontrado mediante análisis JavaScript")
                except Exception as e:
                    scraper_logger.warning(f"Error en script JavaScript para botón de descarga: {str(e)}")
                
            if not boton_descarga:
                scraper_logger.warning("No se encontró el botón de descarga de listado")
                if self.modo_debug:
                    self._captura_debug("no_boton_descarga")
                return None
            
            # MEJORA: Tomar captura para debug
            if self.modo_debug:
                self._captura_debug("antes_click_descarga")
            
            # Hacer clic en el botón con máximos intentos y usar JavaScript como respaldo
            scraper_logger.info("Haciendo clic en el botón de descarga...")
            if not self._clic_seguro(boton_descarga, usar_js=True):
                scraper_logger.warning("No se pudo hacer clic en el botón de descarga con métodos normales")
                # Último intento: clic forzado por JavaScript
                try:
                    self.driver.execute_script("arguments[0].click();", boton_descarga)
                    scraper_logger.info("Clic forzado mediante JavaScript")
                except Exception as e:
                    scraper_logger.error(f"Falló incluso el clic forzado: {str(e)}")
                    return None
            
            # MEJORA: Esperar a que aparezca la ventana modal con tiempo de espera adaptativo
            tiempo_espera_modal = 5  # Segundos máximos de espera para el modal
            scraper_logger.info(f"Esperando aparición de ventana modal ({tiempo_espera_modal}s máximo)...")
            
            # Detectar si la modal está presente
            modal_presente = False
            inicio_espera_modal = time.time()
            while time.time() - inicio_espera_modal < tiempo_espera_modal:
                # Intentar diferentes enfoques para detectar el modal
                try:
                    # 1. Buscar por clases comunes de modales
                    modales = self.driver.find_elements(By.CSS_SELECTOR, ".mat-dialog-container, .modal-content, [role='dialog']")
                    if modales:
                        modal_presente = True
                        scraper_logger.info(f"Modal detectado por clase después de {time.time() - inicio_espera_modal:.1f}s")
                        break
                    
                    # 2. Buscar por atributos de modal
                    modales_attr = self.driver.find_elements(By.CSS_SELECTOR, "[aria-modal='true']")
                    if modales_attr:
                        modal_presente = True
                        scraper_logger.info(f"Modal detectado por atributo después de {time.time() - inicio_espera_modal:.1f}s")
                        break
                    
                    # 3. Último recurso: verificar si hay elementos con z-index alto
                    try:
                        elementos_z = self.driver.execute_script("""
                            const elementos = Array.from(document.querySelectorAll('*'));
                            return elementos.filter(el => {
                                const estilo = window.getComputedStyle(el);
                                const zIndex = parseInt(estilo.zIndex);
                                return !isNaN(zIndex) && zIndex > 1000 && estilo.display !== 'none';
                            });
                        """)
                        
                        if elementos_z and len(elementos_z) > 0:
                            modal_presente = True
                            scraper_logger.info(f"Posible modal detectado por z-index después de {time.time() - inicio_espera_modal:.1f}s")
                            break
                    except:
                        pass
                except Exception as e:
                    scraper_logger.debug(f"Error durante detección de modal: {str(e)}")
                
                # Breve pausa antes de verificar nuevamente
                time.sleep(0.2)
            
            if not modal_presente:
                scraper_logger.warning("No se detectó la aparición de la ventana modal")
                if self.modo_debug:
                    self._captura_debug("no_modal_descarga")
                # No retornamos None aquí, intentaremos buscar el botón de descarga de todos modos
            
            # MEJORA: Capturar estado actual para debug
            if self.modo_debug:
                self._captura_debug("modal_descarga")
            
            # MEJORA: Buscar el botón "Descargar" dentro de la ventana modal con selectores más precisos
            selectores_boton_modal = [
                # Selectores CSS específicos para el botón dentro de la modal
                "button.mat-raised-button.mat-primary",
                "button.mat-button-base.mat-primary",
                "button.mat-dialog-close[color='primary']",
                "button.mat-button[color='primary']",
                "button.mat-raised-button:has(span:contains('Descargar'))",
                ".mat-dialog-container button.mat-primary",
                "[role='dialog'] button.mat-primary",
                ".modal-footer button.mat-primary, .mat-dialog-actions button.mat-primary",

                # Selectores XPath específicos
                "//button[contains(@class, 'mat-primary') and contains(., 'Descargar')]",
                "//button[contains(@class, 'mat-raised-button') and contains(., 'Descargar')]",
                "//span[contains(text(), 'Descargar')]/parent::button",
                "//*[@role='dialog']//button[contains(@class, 'mat-primary')]",
                "//*[@aria-modal='true']//button[contains(@class, 'mat-primary')]",
                "//mat-dialog-container//button[contains(@class, 'mat-primary')]",
                "//div[contains(@class, 'mat-dialog-actions') or contains(@class, 'modal-footer')]//button[contains(@class, 'mat-primary')]"
            ]
            
            boton_modal = None
            for selector in selectores_boton_modal:
                elementos = []
                try:
                    if selector.startswith("//"):
                        elementos = self.driver.find_elements(By.XPATH, selector)
                    else:
                        elementos = self.driver.find_elements(By.CSS_SELECTOR, selector)
                except Exception:
                    continue
                    
                if elementos:
                    # Filtrar elementos visibles
                    elementos_visibles = [elem for elem in elementos if elem.is_displayed()]
                    if elementos_visibles:
                        boton_modal = elementos_visibles[0]
                        scraper_logger.info(f"Botón de descarga en modal encontrado con selector: {selector}")
                        break
            
            # Si no encontramos el botón, intentar con JavaScript más avanzado
            if not boton_modal:
                scraper_logger.info("Intentando encontrar botón Descargar en modal mediante JavaScript...")
                
                script_modal = """
                function encontrarBotonDescargarEnModal() {
                    // 1. Buscar modales o diálogos
                    const modales = Array.from(document.querySelectorAll('.mat-dialog-container, .modal-content, [role="dialog"], [aria-modal="true"]'));
                    
                    if (modales.length === 0) {
                        // Si no encontramos modales explícitos, buscar contenedores con alto z-index
                        const elementosZ = Array.from(document.querySelectorAll('*')).filter(el => {
                            const estilo = window.getComputedStyle(el);
                            const zIndex = parseInt(estilo.zIndex);
                            return !isNaN(zIndex) && zIndex > 1000 && estilo.display !== 'none';
                        });
                        
                        if (elementosZ.length > 0) {
                            modales.push(...elementosZ);
                        }
                    }
                    
                    // 2. Buscar en cada modal
                    for (const modal of modales) {
                        // Buscar por texto "Descargar"
                        const botonesTexto = Array.from(modal.querySelectorAll('button')).filter(btn => {
                            return (btn.textContent || '').toLowerCase().includes('descargar');
                        });
                        
                        if (botonesTexto.length > 0) {
                            return botonesTexto[0];
                        }
                        
                        // Buscar botones primary (generalmente de acción principal)
                        const botonesPrimary = Array.from(modal.querySelectorAll('button.mat-primary, button.mat-raised-button.mat-primary, button[color="primary"]'));
                        
                        if (botonesPrimary.length > 0) {
                            // Preferir los que no sean de cancelar o cerrar
                            const botonesNoCancel = botonesPrimary.filter(b => 
                                !(b.textContent || '').toLowerCase().includes('cancel') && 
                                !(b.textContent || '').toLowerCase().includes('cerrar')
                            );
                            
                            if (botonesNoCancel.length > 0) {
                                return botonesNoCancel[0];
                            }
                            
                            return botonesPrimary[0];
                        }
                        
                        // Último recurso: cualquier botón en el footer
                        const botonesFooter = Array.from(modal.querySelectorAll('.mat-dialog-actions button, .modal-footer button'));
                        if (botonesFooter.length > 0) {
                            return botonesFooter[0]; // Generalmente el primero es el principal
                        }
                    }
                    
                    return null;
                }
                return encontrarBotonDescargarEnModal();
                """
                
                try:
                    boton_modal = self.driver.execute_script(script_modal)
                    if boton_modal:
                        scraper_logger.info("Botón de descarga en modal encontrado mediante análisis JavaScript")
                except Exception as e:
                    scraper_logger.warning(f"Error en script JavaScript para botón modal: {str(e)}")
            
            # Si encontramos el botón en la modal, hacer clic en él
            if boton_modal:
                if self.modo_debug:
                    self._captura_debug("boton_modal_descarga")
                
                scraper_logger.info("Haciendo clic en botón 'Descargar' en la ventana modal...")
                if not self._clic_seguro(boton_modal, usar_js=True):
                    scraper_logger.warning("No se pudo hacer clic en el botón de descarga de la modal")
                    # Último intento con JavaScript directo
                    try:
                        self.driver.execute_script("arguments[0].click();", boton_modal)
                        scraper_logger.info("Clic forzado en botón modal mediante JavaScript")
                    except Exception as e:
                        scraper_logger.error(f"Falló incluso el clic forzado en modal: {str(e)}")
                        
                        # Intentar cerrar la modal y abortar
                        try:
                            cerrar_botones = self.driver.find_elements(By.XPATH, "//button[contains(@class, 'mat-dialog-close')]")
                            if cerrar_botones:
                                self._clic_seguro(cerrar_botones[0])
                        except:
                            pass
                        return None
            else:
                scraper_logger.warning("No se encontró el botón 'Descargar' en la ventana modal")
                if self.modo_debug:
                    self._captura_debug("no_boton_descargar_modal")
                # No retornamos None, porque es posible que la descarga ya se haya iniciado automáticamente
            
            # Esperar a que aparezca el archivo descargado con tiempo de espera adaptativo
            scraper_logger.info("Esperando descarga del archivo...")
            tiempo_espera_descarga = 30  # Máximo 30 segundos para la descarga
            archivo_descargado = None
            
            inicio_espera = time.time()
            while time.time() - inicio_espera < tiempo_espera_descarga:
                archivos = os.listdir(directorio_temp)
                
                # Buscar cualquier tipo de archivo descargado
                pdfs = [f for f in archivos if f.endswith('.pdf') or f.endswith('.PDF')]
                xls = [f for f in archivos if f.endswith('.xls') or f.endswith('.xlsx') or f.endswith('.XLS') or f.endswith('.XLSX')]
                csv = [f for f in archivos if f.endswith('.csv') or f.endswith('.CSV')]
                
                # Priorizar PDF, pero aceptar Excel o CSV como alternativas
                archivos_validos = pdfs + xls + csv
                
                if archivos_validos:
                    archivo_descargado = os.path.join(directorio_temp, archivos_validos[0])
                    scraper_logger.info(f"Archivo descargado: {archivo_descargado} (después de {time.time() - inicio_espera:.1f}s)")
                    break
                    
                # MEJORA: También revisar archivos parciales de descarga
                archivos_parciales = [f for f in archivos if f.endswith('.crdownload') or f.endswith('.part') or f.endswith('.tmp')]
                if archivos_parciales:
                    scraper_logger.info(f"Descarga en progreso: {archivos_parciales[0]}, esperando finalización...")
                
                # Esperar un poco antes de volver a comprobar
                time.sleep(0.5)
            
            # MEJORA: Verificar también archivos parciales (en descarga)
            if not archivo_descargado:
                archivos_parciales = [f for f in os.listdir(directorio_temp) if f.endswith('.crdownload') or f.endswith('.part') or f.endswith('.tmp')]
                if archivos_parciales:
                    scraper_logger.info("La descarga está en progreso, esperando finalización...")
                    # Esperar un poco más para que se complete
                    tiempo_extra = 0
                    while tiempo_extra < 30 and not archivo_descargado:  # 30 segundos adicionales
                        time.sleep(1)
                        tiempo_extra += 1
                        
                        archivos = os.listdir(directorio_temp)
                        pdfs = [f for f in archivos if f.endswith('.pdf') or f.endswith('.PDF')]
                        xls = [f for f in archivos if f.endswith('.xls') or f.endswith('.xlsx') or f.endswith('.XLS') or f.endswith('.XLSX')]
                        csv = [f for f in archivos if f.endswith('.csv') or f.endswith('.CSV')]
                        
                        archivos_validos = pdfs + xls + csv
                        if archivos_validos:
                            archivo_descargado = os.path.join(directorio_temp, archivos_validos[0])
                            scraper_logger.info(f"Archivo descargado después de espera adicional: {archivo_descargado}")
                            break
            
            if not archivo_descargado:
                scraper_logger.warning("Tiempo de espera agotado, no se descargó ningún archivo")
                return None
            
            # MEJORA: Verificar que el archivo tenga contenido
            try:
                tamano_archivo = os.path.getsize(archivo_descargado)
                if tamano_archivo < 100:  # Menos de 100 bytes es sospechoso
                    scraper_logger.warning(f"El archivo descargado es demasiado pequeño: {tamano_archivo} bytes")
                    if tamano_archivo == 0:
                        return None
                else:
                    scraper_logger.info(f"Tamaño del archivo descargado: {tamano_archivo} bytes")
            except Exception as e:
                scraper_logger.warning(f"Error al verificar tamaño del archivo: {str(e)}")
            
            scraper_logger.info(f"Descarga exitosa completada en {time.time() - inicio_tiempo:.2f} segundos")
            return archivo_descargado
            
        except Exception as e:
            scraper_logger.error(f"Error al descargar listado de resultados: {str(e)}")
            if self.modo_debug:
                self._captura_debug("error_descarga_listado")
                traceback.print_exc()
            return None
            
    def extraer_datos_de_archivo(self, ruta_archivo):
        """
        Extrae datos estructurados del archivo de resultados descargado (PDF, Excel o CSV).
        MEJORADO: Procesamiento más robusto de diferentes formatos y extracción precisa 
        del campo Número de Tesis.
        
        Args:
            ruta_archivo (str): Ruta al archivo descargado
            
        Returns:
            list: Lista de diccionarios con datos de resultados estructurados
        """
        try:
            scraper_logger.info(f"Procesando archivo: {ruta_archivo}")
            resultados = []
            
            # Determinar tipo de archivo
            extension = os.path.splitext(ruta_archivo)[1].lower()
            
            # CASO 1: Archivo PDF
            if extension == '.pdf':
                scraper_logger.info("Procesando archivo PDF...")
                
                # Abrir PDF con PyMuPDF
                doc = fitz.open(ruta_archivo)
                
                # MEJORA: Extraer texto con mejor preservación de estructura
                texto_completo = ""
                texto_por_pagina = []
                
                # Extraer texto de todas las páginas
                for pagina in doc:
                    texto_pagina = pagina.get_text()
                    texto_por_pagina.append(texto_pagina)
                    texto_completo += texto_pagina
                
                doc.close()
                
                # MEJORA: Implementar métodos de extracción optimizados para el formato de PDF de la SCJN
                # 1. Intentar extracción mediante análisis tabular
                try:
                    # Intentar identificar la estructura del documento y sus columnas
                    resultados_tabulares = self._extraer_tabla_pdf_scjn(ruta_archivo)
                    
                    if resultados_tabulares and len(resultados_tabulares) > 0:
                        scraper_logger.info(f"Extracción tabular exitosa: {len(resultados_tabulares)} resultados")
                        
                        # Asegurarse de que todos los campos clave estén presentes
                        for i, resultado in enumerate(resultados_tabulares):
                            # Añadir metadatos para identificar origen de los datos
                            resultado["metodo_obtencion"] = "descarga_directa"
                            resultado["numero"] = i + 1
                            
                            # Asegurar campos mínimos
                            if "registro_digital" not in resultado:
                                registro = self._extraer_registro_digital(resultado.get("texto_completo", ""))
                                resultado["registro_digital"] = registro or "No identificado"
                            
                            # Generar URL
                            if resultado["registro_digital"] != "No identificado":
                                resultado["url_tesis"] = self.generar_url_registro(resultado["registro_digital"])
                            
                            # Asegurar campo de número de tesis (prioridad máxima)
                            if "numero_tesis" not in resultado:
                                # Intentar extraerlo del texto completo como último recurso
                                resultado["numero_tesis"] = self._extraer_numero_tesis_avanzado(resultado.get("texto_completo", ""))
                        
                        return resultados_tabulares
                except Exception as e:
                    scraper_logger.warning(f"Error en extracción tabular, probando método alternativo: {str(e)}")
                
                # 2. Enfoque avanzado: Analizar por bloques de texto para identificar cada tesis
                # Este enfoque es específico para el formato de PDF que genera la SCJN
                try:
                    resultados_bloques = self._extraer_bloques_tesis_pdf(texto_completo, texto_por_pagina)
                    
                    if resultados_bloques and len(resultados_bloques) > 0:
                        scraper_logger.info(f"Extracción por bloques exitosa: {len(resultados_bloques)} resultados")
                        return resultados_bloques
                except Exception as e:
                    scraper_logger.warning(f"Error en extracción por bloques: {str(e)}")
                
                # 3. Enfoque básico: Línea por línea para identificar patrones de tesis
                lineas = texto_completo.split('\n')
                
                # Variables para acumular información por resultado
                resultado_actual = None
                
                # MEJORA: Usar patrones específicos del PDF de la SCJN para detectar inicio de tesis
                patron_inicio_tesis = re.compile(r'^\s*(\d+)\.\s+', re.IGNORECASE)
                patron_registro = re.compile(r'Registro\s+digital:?\s*(\d+)', re.IGNORECASE)
                patron_numero_tesis = re.compile(r'(?:Tesis|Tesis No)?:?\s*([A-Z0-9\./]+\s*(?:\([^)]+\))?)', re.IGNORECASE)
                
                for i, linea in enumerate(lineas):
                    linea = linea.strip()
                    if not linea:
                        continue
                    
                    # Detectar inicio de una nueva tesis mediante su numeración
                    match_inicio = patron_inicio_tesis.match(linea)
                    
                    if match_inicio or (i > 0 and "Registro digital:" in linea and "Registro digital:" not in lineas[i-1]):
                        # Si ya teníamos un resultado abierto, guardarlo
                        if resultado_actual and "registro_digital" in resultado_actual:
                            # Intentar extraer número de tesis si no se ha encontrado aún
                            if "numero_tesis" not in resultado_actual or not resultado_actual["numero_tesis"]:
                                resultado_actual["numero_tesis"] = self._extraer_numero_tesis_avanzado(resultado_actual.get("texto_completo", ""))
                            
                            resultados.append(resultado_actual)
                        
                        # Crear nuevo resultado con método de obtención
                        resultado_actual = {
                            "numero": len(resultados) + 1,
                            "metodo_obtencion": "descarga_directa"
                        }
                        
                        # Si la línea tiene formato numerado (1. Registro digital: XXXXX),
                        # intentar extraer el registro directamente
                        if "Registro digital:" in linea:
                            match_reg = patron_registro.search(linea)
                            if match_reg:
                                resultado_actual["registro_digital"] = match_reg.group(1)
                        
                        resultado_actual["texto_completo"] = linea
                    elif resultado_actual:
                        # Continuar acumulando texto para el resultado actual
                        resultado_actual["texto_completo"] += "\n" + linea
                        
                        # Extraer registro digital
                        if "registro_digital" not in resultado_actual and "Registro digital:" in linea:
                            match_reg = patron_registro.search(linea)
                            if match_reg:
                                resultado_actual["registro_digital"] = match_reg.group(1)
                        
                        # Extraer número de tesis - CLAVE PARA NUESTRA MEJORA
                        if "numero_tesis" not in resultado_actual:
                            # Buscar líneas que contengan "Tesis:" o patrones similares
                            if "Tesis:" in linea or "Tesis No.:" in linea or re.search(r'\b[XVI]+\.[0-9]+[oa]?\.[A-Z]+\.[0-9]+', linea):
                                match_tesis = patron_numero_tesis.search(linea)
                                if match_tesis:
                                    resultado_actual["numero_tesis"] = match_tesis.group(1).strip()
                        
                        # Extraer rubro (generalmente en mayúsculas)
                        if "rubro" not in resultado_actual and len(linea) > 15 and linea.isupper():
                            resultado_actual["rubro"] = linea
                
                # No olvidar agregar el último resultado
                if resultado_actual and "registro_digital" in resultado_actual:
                    # Intento final para extraer número de tesis
                    if "numero_tesis" not in resultado_actual or not resultado_actual["numero_tesis"]:
                        resultado_actual["numero_tesis"] = self._extraer_numero_tesis_avanzado(resultado_actual.get("texto_completo", ""))
                    
                    resultados.append(resultado_actual)
            
            # CASO 2: Archivo Excel
            elif extension in ['.xls', '.xlsx', '.XLS', '.XLSX']:
                scraper_logger.info("Procesando archivo Excel...")
                
                # MEJORA: Manejo robusto de Excel
                try:
                    df = pd.read_excel(ruta_archivo)
                    
                    # MEJORA: Mejor detección de columnas relevantes
                    columnas = df.columns.tolist()
                    
                    # Mapeo de nombres de columnas esperados (con variaciones)
                    mapeo_columnas = {
                        'registro': [col for col in columnas if 'registro' in col.lower()],
                        'tesis': [col for col in columnas if any(term in col.lower() for term in ['tesis', 'número', 'clave', 'identificador'])],
                        'rubro': [col for col in columnas if any(term in col.lower() for term in ['rubro', 'título', 'tema', 'contenido'])],
                        'instancia': [col for col in columnas if any(term in col.lower() for term in ['instancia', 'órgano', 'emisor', 'tribunal'])],
                        'fecha': [col for col in columnas if any(term in col.lower() for term in ['fecha', 'publicación', 'emisión'])],
                        'tipo': [col for col in columnas if any(term in col.lower() for term in ['tipo', 'clase', 'categoría'])]
                    }
                    
                    # Obtener columnas efectivas
                    col_efectivas = {}
                    for campo, candidatos in mapeo_columnas.items():
                        if candidatos:
                            col_efectivas[campo] = candidatos[0]
                    
                    # Procesar filas
                    for i, fila in df.iterrows():
                        resultado = {
                            "numero": i + 1,
                            "metodo_obtencion": "descarga_directa"
                        }
                        
                        # Asignar valores de cada campo si existe la columna
                        for campo, columna in col_efectivas.items():
                            valor = str(fila[columna])
                            # Limpiar valores NaN/None
                            if valor.lower() == 'nan' or valor.lower() == 'none':
                                valor = "No identificado"
                            
                            # Campos especiales
                            if campo == 'registro':
                                resultado['registro_digital'] = valor
                            elif campo == 'tesis':
                                resultado['numero_tesis'] = valor
                            else:
                                resultado[campo] = valor
                        
                        # Generar texto completo combinando todos los campos
                        texto_completo = []
                        for campo, columna in col_efectivas.items():
                            valor = str(fila[columna])
                            if valor.lower() != 'nan' and valor.lower() != 'none':
                                texto_completo.append(f"{columna}: {valor}")
                        
                        resultado["texto_completo"] = "\n".join(texto_completo)
                        
                        # Asegurar que tenemos al menos registro y rubro
                        if 'registro_digital' not in resultado:
                            resultado['registro_digital'] = "No identificado"
                        if 'rubro' not in resultado:
                            resultado['rubro'] = "No identificado"
                        
                        # MEJORA: Verificar y limpiar el número de tesis
                        if 'numero_tesis' in resultado:
                            resultado['numero_tesis'] = self._limpiar_numero_tesis(resultado['numero_tesis'])
                        else:
                            # Último recurso: intentar extraer de texto completo
                            resultado['numero_tesis'] = self._extraer_numero_tesis_avanzado(resultado["texto_completo"])
                        
                        # Generar URL
                        if resultado["registro_digital"] != "No identificado":
                            resultado["url_tesis"] = self.generar_url_registro(resultado["registro_digital"])
                        
                        resultados.append(resultado)
                except Exception as e:
                    scraper_logger.error(f"Error procesando Excel: {str(e)}")
                    if self.modo_debug:
                        traceback.print_exc()
            
            # CASO 3: Archivo CSV
            elif extension in ['.csv', '.CSV']:
                scraper_logger.info("Procesando archivo CSV...")
                
                # MEJORA: Procesamiento más robusto de CSV
                # Probar diferentes encodings y delimitadores
                encodings = ['utf-8', 'latin1', 'cp1252', 'iso-8859-1']
                delimitadores = [',', ';', '\t', '|']
                
                df = None
                for encoding in encodings:
                    for delim in delimitadores:
                        try:
                            df = pd.read_csv(ruta_archivo, encoding=encoding, sep=delim)
                            if len(df.columns) > 1:  # Asegurarnos de que los datos se separaron correctamente
                                scraper_logger.info(f"CSV leído con éxito usando encoding {encoding} y delimitador '{delim}'")
                                break
                        except Exception:
                            continue
                    if df is not None and len(df.columns) > 1:
                        break
                
                if df is None or len(df.columns) <= 1:
                    scraper_logger.warning("No se pudo leer correctamente el archivo CSV. Intentando fallback simple.")
                    try:
                        # Intentar leer como texto plano
                        with open(ruta_archivo, 'r', encoding='utf-8', errors='replace') as f:
                            lines = f.readlines()
                        
                        # Intentar determinar delimitador
                        sample_line = lines[0] if lines else ""
                        delim = max(delimitadores, key=lambda d: sample_line.count(d))
                        
                        # Procesar líneas manualmente
                        header = lines[0].strip().split(delim)
                        for i, line in enumerate(lines[1:]):
                            valores = line.strip().split(delim)
                            result = {"numero": i + 1, "metodo_obtencion": "descarga_directa"}
                            
                            for j, val in enumerate(valores):
                                if j < len(header):
                                    campo = header[j].lower()
                                    if 'registro' in campo:
                                        result['registro_digital'] = val
                                    elif any(term in campo for term in ['tesis', 'número', 'clave', 'identificador']):
                                        result['numero_tesis'] = self._limpiar_numero_tesis(val)
                                    elif any(term in campo for term in ['rubro', 'título', 'tema']):
                                        result['rubro'] = val
                                    else:
                                        result[campo] = val
                            
                            if 'registro_digital' not in result:
                                result['registro_digital'] = "No identificado"
                            
                            result['texto_completo'] = line
                            
                            # Generar URL
                            if result["registro_digital"] != "No identificado":
                                result["url_tesis"] = self.generar_url_registro(result["registro_digital"])
                            
                            resultados.append(result)
                        
                        if resultados:
                            return resultados
                    except Exception as e:
                        scraper_logger.error(f"Error en fallback para CSV: {str(e)}")
                        return []
                
                # Continuar con el procesamiento normal si tenemos DataFrame
                columnas = df.columns.tolist()
                
                # MEJORA: Mapeo más extenso de posibles nombres de columnas
                mapeo_columnas = {
                    'registro': [col for col in columnas if 'registro' in col.lower()],
                    'tesis': [col for col in columnas if any(term in col.lower() for term in ['tesis', 'número', 'clave', 'identificador'])],
                    'rubro': [col for col in columnas if any(term in col.lower() for term in ['rubro', 'título', 'tema', 'contenido'])],
                    'instancia': [col for col in columnas if any(term in col.lower() for term in ['instancia', 'órgano', 'emisor', 'tribunal'])],
                    'fecha': [col for col in columnas if any(term in col.lower() for term in ['fecha', 'publicación', 'emisión'])],
                    'tipo': [col for col in columnas if any(term in col.lower() for term in ['tipo', 'clase', 'categoría'])]
                }
                
                # Obtener columnas efectivas
                col_efectivas = {}
                for campo, candidatos in mapeo_columnas.items():
                    if candidatos:
                        col_efectivas[campo] = candidatos[0]
                
                # Procesar filas
                for i, fila in df.iterrows():
                    resultado = {
                        "numero": i + 1,
                        "metodo_obtencion": "descarga_directa"
                    }
                    
                    # Asignar valores de cada campo si existe la columna
                    for campo, columna in col_efectivas.items():
                        valor = str(fila[columna])
                        # Limpiar valores NaN/None
                        if valor.lower() == 'nan' or valor.lower() == 'none':
                            valor = "No identificado"
                        
                        # Campos especiales
                        if campo == 'registro':
                            resultado['registro_digital'] = valor
                        elif campo == 'tesis':
                            resultado['numero_tesis'] = self._limpiar_numero_tesis(valor)
                        else:
                            resultado[campo] = valor
                    
                    # Generar texto completo combinando todos los campos
                    texto_completo = []
                    for campo, columna in col_efectivas.items():
                        valor = str(fila[columna])
                        if valor.lower() != 'nan' and valor.lower() != 'none':
                            texto_completo.append(f"{columna}: {valor}")
                    
                    resultado["texto_completo"] = "\n".join(texto_completo)
                    
                    # Asegurar que tenemos al menos registro y rubro
                    if 'registro_digital' not in resultado:
                        resultado['registro_digital'] = "No identificado"
                    if 'rubro' not in resultado:
                        resultado['rubro'] = "No identificado"
                    
                    # Generar URL
                    if resultado["registro_digital"] != "No identificado":
                        resultado["url_tesis"] = self.generar_url_registro(resultado["registro_digital"])
                    
                    resultados.append(resultado)
            
            scraper_logger.info(f"Se extrajeron {len(resultados)} resultados del archivo descargado")
            return resultados
            
        except Exception as e:
            scraper_logger.error(f"Error al extraer datos del archivo: {str(e)}")
            if self.modo_debug:
                traceback.print_exc()
            return []
    
    def _extraer_tabla_pdf_scjn(self, ruta_pdf):
        """
        Método especializado para extraer información tabular del PDF de listado
        de resultados de la SCJN.
        
        Args:
            ruta_pdf (str): Ruta al archivo PDF
            
        Returns:
            list: Lista de diccionarios con los datos extraídos
        """
        try:
            # Abrir el PDF
            doc = fitz.open(ruta_pdf)
            resultados = []
            
            # Iterar por cada página buscando tablas
            for pagina_num in range(len(doc)):
                pagina = doc[pagina_num]
                
                # Intentar extraer tablas con PyMuPDF
                tablas = pagina.find_tables()
                
                if tablas and tablas.tables:
                    for tabla in tablas.tables:
                        if not tabla:
                            continue
                        
                        # Obtener encabezados y filas
                        header = None
                        
                        for i, fila in enumerate(tabla.rows):
                            # La primera fila suele ser el encabezado
                            if i == 0:
                                header = [cell.text.strip() for cell in fila.cells]
                                continue
                            
                            # Procesar filas de datos
                            datos = {}
                            for j, celda in enumerate(fila.cells):
                                if j < len(header):
                                    # Normalizar nombres de columnas
                                    nombre_columna = header[j].strip()
                                    nombre_normalizado = nombre_columna.lower()
                                    
                                    valor = celda.text.strip()
                                    
                                    # Asignar a campos específicos según el nombre de columna
                                    if 'registro' in nombre_normalizado:
                                        datos['registro_digital'] = valor
                                    elif any(term in nombre_normalizado for term in ['tesis', 'número', 'clave', 'identificador']):
                                        datos['numero_tesis'] = self._limpiar_numero_tesis(valor)
                                    elif any(term in nombre_normalizado for term in ['rubro', 'título', 'tema']):
                                        datos['rubro'] = valor
                                    else:
                                        # Guardar con nombre original para otros campos
                                        datos[nombre_columna] = valor
                            
                            # Construir texto completo combinando todos los campos
                            texto_completo = []
                            for campo, valor in datos.items():
                                texto_completo.append(f"{campo}: {valor}")
                            
                            datos['texto_completo'] = "\n".join(texto_completo)
                            
                            # Añadir detalles de la extracción
                            datos['metodo_obtencion'] = 'descarga_directa'
                            
                            resultados.append(datos)
            
            # Cerrar el documento
            doc.close()
            
            # Normalizar los resultados
            for i, resultado in enumerate(resultados):
                resultado['numero'] = i + 1
                
                # Asegurar que tenemos todos los campos clave
                if 'registro_digital' not in resultado:
                    resultado['registro_digital'] = 'No identificado'
                if 'numero_tesis' not in resultado or not resultado['numero_tesis']:
                    resultado['numero_tesis'] = self._extraer_numero_tesis_avanzado(resultado.get('texto_completo', ''))
                if 'rubro' not in resultado:
                    resultado['rubro'] = 'No identificado'
                
                # Generar URL
                if resultado['registro_digital'] != 'No identificado':
                    resultado['url_tesis'] = self.generar_url_registro(resultado['registro_digital'])
            
            return resultados
            
        except Exception as e:
            scraper_logger.error(f"Error en extracción de tablas del PDF: {str(e)}")
            if self.modo_debug:
                traceback.print_exc()
            return []
    
    def _extraer_bloques_tesis_pdf(self, texto_completo, textos_pagina):
        """
        Extrae bloques correspondientes a cada tesis desde el texto del PDF.
        Método especializado para el formato específico del PDF de listado
        generado por la SCJN.
        
        Args:
            texto_completo (str): Texto completo del documento
            textos_pagina (list): Lista de textos por página
            
        Returns:
            list: Lista de resultados extraídos
        """
        resultados = []
        
        # Patrones para detectar inicio de tesis en el formato del PDF
        patron_inicio = re.compile(r'^\s*(\d+)\.(?:\s+Registro.+)?', re.MULTILINE)
        
        # Encontrar los inicios de cada bloque de tesis
        inicios = []
        for match in patron_inicio.finditer(texto_completo):
            inicios.append((match.start(), int(match.group(1))))
        
        # Si no encontramos inicios con este patrón, intentar uno alternativo
        if not inicios:
            # Patrón alternativo: Buscar líneas que comienzan con número seguido de punto
            patron_alternativo = re.compile(r'^\s*(\d+)\.\s+', re.MULTILINE)
            for match in patron_alternativo.finditer(texto_completo):
                inicios.append((match.start(), int(match.group(1))))
        
        # Ordenar por posición
        inicios.sort()
        
        # Extraer cada bloque
        for i, (inicio, numero) in enumerate(inicios):
            # Determinar el fin del bloque (inicio del siguiente o fin del texto)
            fin = inicios[i+1][0] if i < len(inicios) - 1 else len(texto_completo)
            
            # Extraer el bloque de texto
            bloque = texto_completo[inicio:fin].strip()
            
            # Extraer datos relevantes
            datos = {
                "numero": numero,
                "texto_completo": bloque,
                "metodo_obtencion": "descarga_directa"
            }
            
            # Extraer registro digital
            registro = self._extraer_registro_digital(bloque)
            datos["registro_digital"] = registro if registro else "No identificado"
            
            # Extraer número de tesis - CLAVE PARA NUESTRA MEJORA
            datos["numero_tesis"] = self._extraer_numero_tesis_avanzado(bloque)
            
            # Extraer rubro
            datos["rubro"] = self._extraer_rubro_avanzado(bloque)
            
            # Extraer campos adicionales si están presentes
            datos.update(self._extraer_campos_adicionales(bloque))
            
            # Generar URL
            if datos["registro_digital"] != "No identificado":
                datos["url_tesis"] = self.generar_url_registro(datos["registro_digital"])
            
            resultados.append(datos)
        
        return resultados
    
    def _extraer_registro_digital(self, texto):
        """
        Extrae el registro digital de un texto.
        
        Args:
            texto (str): Texto de donde extraer el registro
            
        Returns:
            str: Registro digital extraído o None si no se encuentra
        """
        patrones = [
            r'Registro\s+digital:?\s*(\d+)',
            r'Registro:?\s*(\d+)',
            r'(?<=\s|^)(\d{6,7})(?=\s|$)'  # Números sueltos de 6-7 dígitos
        ]
        
        for patron in patrones:
            match = re.search(patron, texto, re.IGNORECASE)
            if match:
                return match.group(1)
        
        return None
    
    def _extraer_numero_tesis_avanzado(self, texto):
        """
        Método especializado para extraer con precisión el número de tesis.
        MEJORA CLAVE: Implementa patrones específicos para los formatos
        de número de tesis que aparecen en los PDFs de la SCJN.
        
        Args:
            texto (str): Texto de donde extraer el número de tesis
            
        Returns:
            str: Número de tesis extraído o "No identificado" si no se encuentra
        """
        if not texto:
            return "No identificado"
        
        # MEJORA: Patrones específicos para los formatos de número de tesis de la SCJN
        patrones = [
            # NUEVO PATRÓN PARA EL FORMATO DE GACETA
            r'Gaceta del Semanario Judicial de la Federación;\s*([^;]+?);',
            r'Semanario Judicial[^;]*;\s*([^;]+?);',
            
            # Patrones de formato completo: PC.I.A. 15/2019 (10a.)
            r'(?:Tesis|Tesis No\.?:?|Número de Tesis:?)\s*([A-Z0-9\.]+\s*(?:[\/\.][A-Z0-9]+)+\s*(?:\([^)]+\))?)',
            # Patrones de formato completo: PC.I.A. 15/2019 (10a.)
            r'(?:Tesis|Tesis No\.?:?|Número de Tesis:?)\s*([A-Z0-9\.]+\s*(?:[\/\.][A-Z0-9]+)+\s*(?:\([^)]+\))?)',
            
            # Patrones de formato J/S (jurisprudencia o sentencia)
            r'([A-Z]{1,5}\.[A-Z]{1,3}\.(?:[A-Z]{1,3}\.)+[^;,\s]+?J\/\d+)',
            r'[^;]*?;\s*([^;,]+?J\/\d+)',
            r'([IVX]+\.\d+[oa]?\.[A-Z]+\.(?:\s*\d+)?(?:\s*J\/\d+)?)',
            
            # Patrones específicos de la SCJN
            r'(PC\.[IVX]+\.[A-Z]+\.\s*\d+\/\d+\s*[A-Z]?\s*(?:\([^)]+\))?)',
            r'([A-Z]+\.[IVX]+\.[A-Z]+\.\s*\d+\/\d+\s*[A-Z]?\s*(?:\([^)]+\))?)',
            r'([IVX]+\.\d+[oa]?\.[A-Z]+\.(?:\s*\d+)?(?:\s*(?:\/|\.)(?:\s*\d+)?)?(?:\s*\([^)]+\))?)',
            
            # Patrones para tesis aisladas y con época
            r'(\d+[ao]?\.[A-Z]+\.(?:\s*\d+)?(?:\s*\/\s*\d+)?(?:\s*\([^)]+\))?)',
            r'([A-Z]{1,5}\.[A-Z]{1,3}\.(?:[A-Z]{1,3}\.)+[^;,\s]+)',
            
            # Patrones para la busqueda "Principio de buena fe" en la SCJN
            r'([A-Z0-9]{1,4}\. ?[IVXLCDM]+ ?[A-Z]\. ?[A-Z0-9]{1,3}\.(?:\s*\d+)?(?:\s*\/\s*\d+)?(?:\s*\([^)]+\))?)',
            r'([IVXLCDM]+\. ?[A-Z]{1,3}\. ?[A-Z]{1,3}\.(?:\s*\d+)?(?:\s*\/\s*\d+)?(?:\s*\([^)]+\))?)',
            
            # Último recurso: cualquier sección que parezca un identificador alfanumérico con estructura
            r'(?:Tesis|Clave|Número):?\s*([A-Z0-9]{1,6}\.[A-Z0-9]{1,6}(?:\.[A-Z0-9]+)+)',
        ]
        
        for patron in patrones:
            match = re.search(patron, texto, re.IGNORECASE)
            if match:
                # Limpiar el resultado
                numero_tesis = match.group(1).strip()
                numero_tesis = self._limpiar_numero_tesis(numero_tesis)
                return numero_tesis
        
        # Si no se encontró con patrones específicos, intentar extraer de líneas clave
        lineas = texto.split('\n')
        for linea in lineas:
            if "Tesis:" in linea or "Número de Tesis:" in linea or "Clave:" in linea:
                # Extraer lo que sigue después de los dos puntos
                partes = linea.split(':', 1)
                if len(partes) > 1 and partes[1].strip():
                    # Limpiar y quedarse con la primera parte significativa
                    numero = partes[1].strip().split()[0]
                    if re.search(r'[A-Z0-9]', numero):  # Verificar que contenga alfanuméricos
                        return self._limpiar_numero_tesis(numero)
        
        return "No identificado"
    
    def _limpiar_numero_tesis(self, numero_tesis):
        """
        Limpia y normaliza el formato del número de tesis.
        
        Args:
            numero_tesis (str): Número de tesis a limpiar
            
        Returns:
            str: Número de tesis limpio y normalizado
        """
        if not numero_tesis or numero_tesis == "No identificado":
            return "No identificado"
        
        # Eliminar caracteres no deseados al inicio y final
        numero_tesis = numero_tesis.strip().rstrip('.,;:')
        
        # Eliminar texto descriptivo si está presente
        numero_tesis = re.sub(r'^(?:Tesis|Clave|Número):?\s*', '', numero_tesis)
        
        # Normalizar espacios
        numero_tesis = re.sub(r'\s+', ' ', numero_tesis)
        
        # Eliminar códigos HTML si existen
        numero_tesis = re.sub(r'<[^>]*>', '', numero_tesis)
        
        # Eliminar texto de seguimiento innecesario
        numero_tesis = re.sub(r'\s*==\s*\$\d+\s*$', '', numero_tesis)
        
        return numero_tesis
    
    def _extraer_rubro_avanzado(self, texto):
        """
        Extrae el rubro de la tesis con métodos avanzados.
        
        Args:
            texto (str): Texto de donde extraer el rubro
            
        Returns:
            str: Rubro extraído o "No identificado" si no se encuentra
        """
        if not texto:
            return "No identificado"
        
        # Patrones para localizar el rubro
        patrones = [
            r'Rubro:?\s*([^\n]+)',
            r'(?:CRITERIO SOSTENIDO|CRITERIO JURÍDICO):?\s*([^\n]+)',
            r'RUBRO:?\s*:?\s*([^\n]+)'
        ]
        
        for patron in patrones:
            match = re.search(patron, texto, re.IGNORECASE)
            if match:
                return match.group(1).strip()
        
        # Si no se encontró con patrones, analizar por estructura
        lineas = texto.strip().split('\n')
        for i, linea in enumerate(lineas):
            linea = linea.strip()
            if linea and len(linea) > 25:
                # El rubro suele aparecer después del registro o número de tesis
                if (i > 0 and any(term in lineas[i-1].upper() for term in ["REGISTRO", "TESIS", "DIGITAL"])):
                    return linea
                
                # También suele estar en mayúsculas
                if linea.isupper() and len(linea) > 30:
                    if not any(term in linea for term in ["REGISTRO", "ÉPOCA", "MATERIA"]):
                        return linea
        
        return "No identificado"
    
    def _extraer_campos_adicionales(self, texto):
        """
        Extrae campos adicionales del texto completo.
        
        Args:
            texto (str): Texto de donde extraer los campos
            
        Returns:
            dict: Diccionario con los campos adicionales extraídos
        """
        campos = {}
        
        # Patrones para campos comunes
        patrones = {
            'instancia': r'Instancia:?\s*([^\n]+)',
            'fecha': r'Fecha:?\s*([^\n]+)',
            'votacion': r'Votación:?\s*([^\n]+)',
            'precedentes': r'Precedentes?:?\s*([^\n]+)',
            'fuente': r'Fuente:?\s*([^\n]+)',
            'materia': r'Materia:?\s*([^\n]+)',
            'epoca': r'Época:?\s*([^\n]+)',
            'tipo_tesis': r'Tipo\s+de\s+tesis:?\s*([^\n]+)'
        }
        
        for campo, patron in patrones.items():
            match = re.search(patron, texto, re.IGNORECASE)
            if match:
                campos[campo] = match.group(1).strip()
        
        return campos

    def _enriquecer_resultado(self, resultado):
        """
        Enriquece un resultado con información adicional extraída
        mediante técnicas avanzadas de procesamiento de texto.
        
        Args:
            resultado (dict): Diccionario con la información del resultado
        """
        texto_completo = resultado.get('texto_completo', '')
        if not texto_completo:
            return
        
        # Extraer número de tesis con patrones optimizados
        if 'numero_tesis' not in resultado or not resultado['numero_tesis']:
            resultado['numero_tesis'] = self._extraer_numero_tesis_avanzado(texto_completo)
        
        # Extraer rubro si no está presente
        if 'rubro' not in resultado or not resultado['rubro']:
            resultado['rubro'] = self._extraer_rubro_avanzado(texto_completo)
        
        # Extraer otros metadatos relevantes
        metadatos = {
            'instancia': r'Instancia:?\s*([^\n]+)',
            'fecha': r'Fecha:?\s*([^\n]+)',
            'votacion': r'Votación:?\s*([^\n]+)',
            'precedentes': r'Precedentes?:?\s*([^\n]+)',
            'fuente': r'Fuente:?\s*([^\n]+)',
            'materia': r'Materia:?\s*([^\n]+)',
            'epoca': r'Época:?\s*([^\n]+)',
            'tipo_tesis': r'Tipo\s+de\s+tesis:?\s*([^\n]+)'
        }
        
        for campo, patron in metadatos.items():
            if campo not in resultado or not resultado[campo]:
                match = re.search(patron, texto_completo, re.IGNORECASE)
                if match:
                    resultado[campo] = match.group(1).strip()

    def buscar_tesis_compuesta(self, termino_general, termino_refinamiento=None, numero_resultados=50, 
                              incluir_8a_epoca=False, incluir_7a_epoca=False, 
                              incluir_6a_epoca=False, incluir_5a_epoca=False,
                              tipo_tesis=0):
        """
        Realiza una búsqueda compuesta: primero con un término general y luego refina los resultados.

        Args:
            termino_general (str): Término inicial de búsqueda
            termino_refinamiento (str): Término para refinar los resultados (opcional)
            numero_resultados (int): Número máximo de resultados a obtener
            incluir_8a_epoca (bool): Si se incluye la 8ª Época en la búsqueda
            incluir_7a_epoca (bool): Si se incluye la 7ª Época en la búsqueda
            incluir_6a_epoca (bool): Si se incluye la 6ª Época en la búsqueda
            incluir_5a_epoca (bool): Si se incluye la 5ª Época en la búsqueda
            tipo_tesis (int): Tipo de tesis a buscar (0: Todas, 1: Jurisprudencia, 2: Aislada)

        Returns:
            dict: Diccionario con resultados y metadatos de la búsqueda
        """
        # Si se pasa un valor de tipo_tesis, se actualiza el atributo
        self.tipo_tesis = tipo_tesis
        
        # Reiniciar estadísticas
        self.estadisticas = {
            "paginas_procesadas": 0,
            "resultados_obtenidos": 0,
            "reintentos_navegacion": 0,
            "tiempo_inicio": time.time(),
            "busqueda_general": termino_general,
            "busqueda_refinada": termino_refinamiento,
            "incluir_8a_epoca": incluir_8a_epoca,
            "incluir_7a_epoca": incluir_7a_epoca,
            "incluir_6a_epoca": incluir_6a_epoca,
            "incluir_5a_epoca": incluir_5a_epoca,
            "tipo_tesis": tipo_tesis,
            "total_general": 0,  # Se actualizará después de la búsqueda general
            "total_refinado": 0  # Se actualizará si se refina la búsqueda
        }

        scraper_logger.info(f"Iniciando búsqueda compuesta. Término general: '{termino_general}', "
                            f"Refinamiento: {termino_refinamiento or 'Ninguno'}, "
                            f"Épocas adicionales: 8ª={incluir_8a_epoca}, 7ª={incluir_7a_epoca}, "
                            f"6ª={incluir_6a_epoca}, 5ª={incluir_5a_epoca}, "
                            f"Tipo de tesis: {tipo_tesis}, "
                            f"Descarga directa: {self.usar_descarga_directa}")

        try:
            # Paso 1: Acceder a la página de búsqueda
            scraper_logger.info(f"Accediendo a la URL base {self.url_base}")
            self.driver.get(self.url_base)
        except Exception as e:
            scraper_logger.error(f"No se pudo acceder a la URL base {self.url_base}: {str(e)}")
            raise RuntimeError(f"No se pudo acceder a la URL: {str(e)}")

        # Paso 2: Realizar la búsqueda general
        exito_busqueda = self._realizar_busqueda_general(termino_general, incluir_8a_epoca, incluir_7a_epoca, incluir_6a_epoca, incluir_5a_epoca)
        if not exito_busqueda:
            scraper_logger.error(f"Falló la búsqueda general con término '{termino_general}'")
            return {
                "tipo_busqueda": "fallida",
                "termino_general": termino_general,
                "error": "Falló la búsqueda general",
                "resultados": []
            }

        # Obtener el número total de resultados de la búsqueda general
        total_general = self._obtener_total_resultados()
        self.estadisticas["total_general"] = total_general
        scraper_logger.info(f"Búsqueda general completada. Total de resultados: {total_general}")

        # Si no hay término de refinamiento, procesar resultados de la búsqueda general
        if not termino_refinamiento:
            scraper_logger.info("No se especificó término de refinamiento. Procesando resultados de búsqueda general.")
            
            # MEJORA: Intentar descarga directa primero (si está habilitado)
            if self.usar_descarga_directa:
                scraper_logger.info("Intentando obtener resultados mediante descarga directa...")
                archivo_descargado = self.descargar_listado_resultados()
                
                if archivo_descargado:
                    scraper_logger.info(f"Descarga directa exitosa: {archivo_descargado}")
                    # Extraer datos del archivo descargado
                    resultados_directos = self.extraer_datos_de_archivo(archivo_descargado)
                    
                    # Si encontramos resultados, usarlos
                    if resultados_directos and len(resultados_directos) > 0:
                        # Limitar al número solicitado
                        resultados_finales = resultados_directos[:numero_resultados]
                        scraper_logger.info(f"Usando {len(resultados_finales)} resultados obtenidos directamente del archivo descargado")
                        
                        return {
                            "tipo_busqueda": "general_optimizada",
                            "termino_general": termino_general,
                            "total_resultados": total_general,
                            "resultados": resultados_finales,
                            "estadisticas": self.estadisticas,
                            "metodo_obtencion": "descarga_directa"
                        }
                    else:
                        scraper_logger.warning("La descarga fue exitosa, pero no se pudieron extraer resultados. Usando método tradicional.")
            
            # Si la descarga directa falló o no está habilitada, usar el método tradicional
            scraper_logger.info("Usando método tradicional para obtener resultados")
            resultados = self._procesar_resultados_paginados(numero_resultados)

            return {
                "tipo_busqueda": "general",
                "termino_general": termino_general,
                "total_resultados": total_general,
                "resultados": resultados,
                "estadisticas": self.estadisticas
            }

        # Paso 3: Refinar la búsqueda con el segundo término
        scraper_logger.info(f"Refinando búsqueda con término: '{termino_refinamiento}'")
        exito_refinamiento = self._refinar_busqueda(termino_refinamiento)

        if not exito_refinamiento:
            scraper_logger.warning("No se pudo refinar la búsqueda. Procesando resultados de búsqueda general.")
            resultados = self._procesar_resultados_paginados(numero_resultados)

            return {
                "tipo_busqueda": "general",
                "termino_general": termino_general,
                "total_resultados": total_general,
                "resultados": resultados,
                "estadisticas": self.estadisticas,
                "error_refinamiento": True
            }

        # Obtener el número total de resultados refinados
        total_refinado = self._obtener_total_resultados()
        self.estadisticas["total_refinado"] = total_refinado
        scraper_logger.info(f"Búsqueda refinada completada. Total de resultados: {total_refinado}")

        # Paso 4: Procesar los resultados refinados
        # MEJORA: Intentar descarga directa primero (si está habilitado)
        if self.usar_descarga_directa:
            scraper_logger.info("Intentando obtener resultados refinados mediante descarga directa...")
            archivo_descargado = self.descargar_listado_resultados()
            
            if archivo_descargado:
                scraper_logger.info(f"Descarga directa exitosa para búsqueda refinada: {archivo_descargado}")
                # Extraer datos del archivo descargado
                resultados_directos = self.extraer_datos_de_archivo(archivo_descargado)
                
                # Si encontramos resultados, usarlos
                if resultados_directos and len(resultados_directos) > 0:
                    # Limitar al número solicitado
                    resultados_finales = resultados_directos[:numero_resultados]
                    scraper_logger.info(f"Usando {len(resultados_finales)} resultados refinados obtenidos del archivo descargado")
                    
                    return {
                        "tipo_busqueda": "refinada_optimizada",
                        "termino_general": termino_general,
                        "termino_refinamiento": termino_refinamiento,
                        "total_general": total_general,
                        "total_refinado": total_refinado,
                        "resultados": resultados_finales,
                        "estadisticas": self.estadisticas,
                        "metodo_obtencion": "descarga_directa"
                    }
                else:
                    scraper_logger.warning("La descarga de resultados refinados fue exitosa, pero no se pudieron extraer datos. Usando método tradicional.")

        # Si la descarga directa falló o no está habilitada, usar el método tradicional
        scraper_logger.info("Usando método tradicional para obtener resultados refinados")
        resultados_refinados = self._procesar_resultados_paginados(numero_resultados)

        return {
            "tipo_busqueda": "refinada",
            "termino_general": termino_general,
            "termino_refinamiento": termino_refinamiento,
            "total_general": total_general,
            "total_refinado": total_refinado,
            "resultados": resultados_refinados,
            "estadisticas": self.estadisticas
        }

    def _realizar_busqueda_general(self, termino_busqueda, incluir_8a_epoca=False, incluir_7a_epoca=False, 
                                  incluir_6a_epoca=False, incluir_5a_epoca=False):
        """
        Realiza la búsqueda general con el primer término.

        Args:
            termino_busqueda (str): Término a buscar
            incluir_8a_epoca (bool): Si se incluye la 8ª Época en la búsqueda
            incluir_7a_epoca (bool): Si se incluye la 7ª Época en la búsqueda
            incluir_6a_epoca (bool): Si se incluye la 6ª Época en la búsqueda
            incluir_5a_epoca (bool): Si se incluye la 5ª Época en la búsqueda

        Returns:
            bool: True si la búsqueda fue exitosa, False en caso contrario
        """
        scraper_logger.info(f"Realizando búsqueda general con término: '{termino_busqueda}'")
        scraper_logger.info(f"Configuración de épocas: 8ª Época={incluir_8a_epoca}, 7ª Época={incluir_7a_epoca}, " +
                           f"6ª Época={incluir_6a_epoca}, 5ª Época={incluir_5a_epoca}")

        try:
            # Identificar el campo de búsqueda principal
            selectores_campo_busqueda = [
                "input[placeholder*='Escriba el tema']",
                "input[placeholder*='Buscar']",
                "input.busqueda",
                "input[type='search']"
            ]

            campo_busqueda = None
            for selector in selectores_campo_busqueda:
                elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=5)
                if elementos:
                    campo_busqueda = elementos[0]
                    scraper_logger.info(f"Campo de búsqueda localizado con selector: {selector}")
                    break

            if not campo_busqueda:
                # Usar un método alternativo buscando por texto en placeholder
                scraper_logger.warning("Usando método alternativo para encontrar campo de búsqueda...")
                campos_input = self.driver.find_elements(By.TAG_NAME, "input")
                for campo in campos_input:
                    placeholder = campo.get_attribute("placeholder") or ""
                    if "tema" in placeholder.lower() or "busca" in placeholder.lower() or "palabra" in placeholder.lower():
                        campo_busqueda = campo
                        scraper_logger.info(f"Campo encontrado por atributo placeholder: {placeholder}")
                        break

            if not campo_busqueda:
                scraper_logger.error("No se pudo localizar el campo de búsqueda principal")
                self._captura_debug("no_campo_busqueda")
                return False

            # Configurar épocas antes de realizar la búsqueda
            try:
                # Intentar expandir el selector de filtros si está colapsado
                time.sleep(1)  # Pausa para estabilizar la página
                
                # Buscar botón para expandir filtros
                botones_filtros = self.driver.find_elements(
                    By.XPATH, 
                    "//button[contains(@class, 'filtro') or contains(text(), 'Filtro') or contains(text(), 'filtro')]"
                )
                if botones_filtros:
                    self._clic_seguro(botones_filtros[0])
                    scraper_logger.info("Expandidos filtros de búsqueda")
                    time.sleep(1)
                
                # Configurar el tipo de tesis
                try:
                    # Los IDs de los radio buttons en la página son check1, check2 y check3
                    check_id = f"check{self.tipo_tesis + 1}"  # +1 porque los IDs empiezan en 1, no en 0
                    
                    # Primero intentar expandir los filtros si es necesario
                    botones_filtros = self.driver.find_elements(
                        By.XPATH, 
                        "//button[contains(@class, 'filtro') or contains(text(), 'Filtro') or contains(text(), 'filtro')]"
                    )
                    if botones_filtros:
                        self._clic_seguro(botones_filtros[0])
                        scraper_logger.info("Expandidos filtros para tipo de tesis")
                        time.sleep(1)
                    
                    # Buscar el radio button correspondiente - intentar primero por ID
                    radio_button = None
                    try:
                        radio_button = self.driver.find_element(By.ID, check_id)
                    except NoSuchElementException:
                        # Si no se encuentra por ID, intentar por otros atributos
                        selectores_tipo = [
                            f"//input[@type='radio'][@id='{check_id}']",
                            f"//input[@type='radio'][@name='tipoTesis'][@value='{self.tipo_tesis}']",
                            f"//input[@type='radio'][contains(@class, 'tipo-tesis')][@value='{self.tipo_tesis}']"
                        ]
                        
                        for selector in selectores_tipo:
                            elementos = self.driver.find_elements(By.XPATH, selector)
                            if elementos:
                                radio_button = elementos[0]
                                break
                        
                        # Si aún no encontramos, buscar por labels
                        if not radio_button:
                            tipo_labels = ["Tesis jurisprudenciales y aisladas", "Jurisprudencia", "Aislada"]
                            label_xpath = f"//label[contains(text(), '{tipo_labels[self.tipo_tesis]}')]"
                            labels = self.driver.find_elements(By.XPATH, label_xpath)
                            if labels:
                                # Intentar encontrar el input asociado
                                try:
                                    radio_id = labels[0].get_attribute("for")
                                    if radio_id:
                                        radio_button = self.driver.find_element(By.ID, radio_id)
                                except:
                                    # Si falla, intentar hacer clic en el label directamente
                                    self._clic_seguro(labels[0])
                                    scraper_logger.info(f"Seleccionado tipo de tesis mediante label: {tipo_labels[self.tipo_tesis]}")
                    
                    # Si encontramos el radio button, verificar si está seleccionado y hacer clic si es necesario
                    if radio_button:
                        if not radio_button.is_selected():
                            self._clic_seguro(radio_button)
                            scraper_logger.info(f"Seleccionado tipo de tesis ID: {check_id}")
                except Exception as e:
                    scraper_logger.warning(f"No se pudo seleccionar el tipo de tesis: {str(e)}")
                
                # Buscar sección de épocas
                botones_epocas = self.driver.find_elements(
                    By.XPATH, 
                    "//button[contains(text(), 'Época') or contains(@aria-label, 'época') or contains(@id, 'epoca')]"
                )
                if botones_epocas:
                    self._clic_seguro(botones_epocas[0])
                    scraper_logger.info("Expandido selector de épocas")
                    time.sleep(1)

                # Intentar con un método alternativo si no funciona el anterior
                if not botones_epocas:
                    sections = self.driver.find_elements(
                        By.XPATH, 
                        "//div[contains(text(), 'Época')]/parent::*"
                    )
                    if sections:
                        self._clic_seguro(sections[0])
                        scraper_logger.info("Expandido selector de épocas (método alternativo)")
                        time.sleep(1)
                
                # Verificar y seleccionar las épocas adicionales
                if incluir_8a_epoca:
                    octava_elementos = self.driver.find_elements(
                        By.XPATH, 
                        "//label[contains(text(), '8') or contains(text(), 'Octava') or contains(text(), 'octava')]"
                    )
                    if octava_elementos:
                        self._clic_seguro(octava_elementos[0])
                        scraper_logger.info("8ª Época seleccionada correctamente")
                    else:
                        scraper_logger.warning("No se pudo encontrar el selector para 8ª Época")
                
                if incluir_7a_epoca:
                    septima_elementos = self.driver.find_elements(
                        By.XPATH, 
                        "//label[contains(text(), '7') or contains(text(), 'Séptima') or contains(text(), 'séptima')]"
                    )
                    if septima_elementos:
                        self._clic_seguro(septima_elementos[0])
                        scraper_logger.info("7ª Época seleccionada correctamente")
                    else:
                        scraper_logger.warning("No se pudo encontrar el selector para 7ª Época")

                # Código añadido para 6ª Época
                if incluir_6a_epoca:
                    sexta_elementos = self.driver.find_elements(
                        By.XPATH, 
                        "//label[contains(text(), '6') or contains(text(), 'Sexta') or contains(text(), 'sexta')]"
                    )
                    if sexta_elementos:
                        self._clic_seguro(sexta_elementos[0])
                        scraper_logger.info("6ª Época seleccionada correctamente")
                    else:
                        scraper_logger.warning("No se pudo encontrar el selector para 6ª Época")

                # Código añadido para 5ª Época
                if incluir_5a_epoca:
                    quinta_elementos = self.driver.find_elements(
                        By.XPATH, 
                        "//label[contains(text(), '5') or contains(text(), 'Quinta') or contains(text(), 'quinta')]"
                    )
                    if quinta_elementos:
                        self._clic_seguro(quinta_elementos[0])
                        scraper_logger.info("5ª Época seleccionada correctamente")
                    else:
                        scraper_logger.warning("No se pudo encontrar el selector para 5ª Época")
                
                # Cerrar el panel de filtros si es necesario
                botones_cerrar = self.driver.find_elements(
                    By.XPATH, 
                    "//button[contains(@aria-label, 'Cerrar')]"
                )
                if botones_cerrar:
                    self._clic_seguro(botones_cerrar[0])
                    scraper_logger.info("Cerrado panel de filtros")
                    time.sleep(0.5)
                
            except Exception as e:
                scraper_logger.warning(f"Error al configurar épocas: {str(e)}")
                # Continuamos aunque falle la configuración de épocas
                
            # Ingresar el término y realizar la búsqueda
            campo_busqueda.clear()
            campo_busqueda.send_keys(termino_busqueda)
            scraper_logger.info(f"Término de búsqueda ingresado: {termino_busqueda}")

            # Identificar y hacer clic en el botón de búsqueda
            selectores_boton = [
                "button.buscar", "button[type='submit']", "button.btn-search",
                "button.btn-primary", "button.search", "button[aria-label='Buscar']",
                "button:has(i.fa-search)"
            ]

            boton_buscar = None
            for selector in selectores_boton:
                elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=2)
                if elementos:
                    boton_buscar = elementos[0]
                    scraper_logger.info(f"Botón encontrado con selector: {selector}")
                    break

            # Si no encontramos el botón con selectores CSS, intentar con XPath
            if not boton_buscar:
                xpath_botones = [
                    "//button[contains(text(), 'Buscar')]",
                    "//button[.//i[contains(@class, 'search')]]",
                    "//button[.//span[contains(text(), 'Buscar')]]"
                ]
                for xpath in xpath_botones:
                    elementos = self._esperar_elementos(By.XPATH, xpath, timeout=2)
                    if elementos:
                        boton_buscar = elementos[0]
                        scraper_logger.info(f"Botón encontrado con XPath: {xpath}")
                        break

            # Si todavía no hay botón, buscar por JS
            if not boton_buscar:
                scraper_logger.warning("Intentando encontrar botón mediante JavaScript...")
                script_boton = """
                function encontrarBoton() {
                    // Buscar por texto
                    const botones = Array.from(document.querySelectorAll('button'));
                    for (const btn of botones) {
                        if ((btn.textContent || '').toLowerCase().includes('buscar')) {
                            return btn;
                        }
                    }
                    // Buscar por ícono
                    const iconosBusqueda = document.querySelectorAll('i.fa-search, i.fa-magnifying-glass');
                    for (const icono of iconosBusqueda) {
                        return icono.closest('button');
                    }
                    return null;
                }
                return encontrarBoton();
                """
                try:
                    boton_buscar = self.driver.execute_script(script_boton)
                    if boton_buscar:
                        scraper_logger.info("Botón encontrado mediante JavaScript")
                except Exception as e:
                    scraper_logger.error(f"Error al ejecutar script: {str(e)}")

            if not boton_buscar:
                # Último recurso: intentar presionar Enter en el campo
                scraper_logger.warning("No se encontró botón de búsqueda. Intentando con Enter...")
                try:
                    campo_busqueda.send_keys(Keys.ENTER)
                    scraper_logger.info("Búsqueda activada con tecla Enter")
                except Exception as e:
                    scraper_logger.error(f"Error al enviar tecla Enter: {str(e)}")
                    self._captura_debug("no_boton_buscar")
                    return False
            else:
                # Hacer clic en el botón
                if not self._clic_seguro(boton_buscar):
                    scraper_logger.error("No se pudo hacer clic en el botón de búsqueda")
                    self._captura_debug("error_clic_buscar")
                    return False

            # Esperar a que carguen los resultados (pausa para asegurar transición)
            scraper_logger.info("Esperando a que carguen los resultados...")
            time.sleep(3)

            # Verificar si estamos en la página de resultados
            try:
                # Esperar elementos de resultado o mensaje de "no resultados"
                elementos_resultado = self._esperar_elementos(
                    By.CSS_SELECTOR,
                    ".registro-item, .resultado-item, div[class*='resultado']",
                    timeout=5
                )

                if elementos_resultado:
                    scraper_logger.info(f"Se encontraron {len(elementos_resultado)} elementos de resultado")
                    return True

                # Verificar si hay mensaje de "no resultados"
                mensaje_no_resultados = self.driver.find_elements(
                    By.XPATH,
                    "//div[contains(text(), 'No se encontraron') or contains(text(), 'Sin resultados')]"
                )

                if mensaje_no_resultados:
                    scraper_logger.info("El sistema indica que no hay resultados para esta búsqueda")
                    return True  # Consideramos exitoso aunque no haya resultados

                scraper_logger.warning("No se detectaron resultados ni mensaje de 'no resultados'")
                self._captura_debug("sin_detectar_resultados")
                return False

            except Exception as e:
                scraper_logger.error(f"Error al verificar resultados: {str(e)}")
                self._captura_debug("error_verificar_resultados")
                return False

        except Exception as e:
            scraper_logger.error(f"Error en búsqueda general: {str(e)}")
            self._captura_debug("error_busqueda_general")
            return False

    def _refinar_busqueda(self, termino_refinamiento):
        """
        Refina la búsqueda actual con un término adicional.

        Args:
            termino_refinamiento (str): Término para refinar la búsqueda

        Returns:
            bool: True si el refinamiento fue exitoso, False en caso contrario
        """
        scraper_logger.info(f"Refinando búsqueda con término: '{termino_refinamiento}'")

        try:
            # Paso 1: Identificar el campo de refinamiento
            selectores_refinamiento = [
                "input[placeholder*='precisar']",
                "input[placeholder*='refinar']",
                "input[id*='refinar']",
                "input.refinamiento",
                "input[aria-label*='refinar']",
                "input[placeholder*='palabra']"
            ]

            campo_refinamiento = None
            for selector in selectores_refinamiento:
                elementos = self._esperar_elementos(By.CSS_SELECTOR, selector, timeout=2)
                if elementos:
                    campo_refinamiento = elementos[0]
                    scraper_logger.info(f"Campo de refinamiento encontrado con selector: {selector}")
                    break

            if not campo_refinamiento:
                scraper_logger.warning("Usando método alternativo para buscar campo de refinamiento...")
                campos_input = self.driver.find_elements(By.TAG_NAME, "input")
                for campo in campos_input:
                    if not campo.is_displayed():
                        continue
                    placeholder = campo.get_attribute("placeholder") or ""
                    if ("refin" in placeholder.lower() or "preci" in placeholder.lower() or 
                        "palabra" in placeholder.lower()):
                        campo_refinamiento = campo
                        scraper_logger.info(f"Campo de refinamiento encontrado por placeholder: {placeholder}")
                        break
                if not campo_refinamiento:
                    campos_visibles = [c for c in campos_input if c.is_displayed()]
                    if len(campos_visibles) > 1:
                        campo_refinamiento = campos_visibles[1]
                        scraper_logger.info("Usando segundo campo de entrada visible como campo de refinamiento")

            if not campo_refinamiento:
                scraper_logger.error("No se pudo encontrar el campo de refinamiento")
                self._captura_debug("no_campo_refinamiento")
                return False

            # Paso 2: Ingresar el término de refinamiento
            campo_refinamiento.clear()
            campo_refinamiento.send_keys(termino_refinamiento)
            scraper_logger.info(f"Término de refinamiento ingresado: {termino_refinamiento}")

            # Paso 3: Verificar que los filtros estén activados
            filtros = {
                "Localización": False,
                "Rubro": False,
                "Texto": False
            }

            try:
                for nombre_filtro in filtros.keys():
                    xpath_filtro = (f"//label[contains(text(), '{nombre_filtro}')]/preceding-sibling::input[@type='checkbox'] | "
                                    f"//label[contains(text(), '{nombre_filtro}')]/input[@type='checkbox']")
                    checkbox = self.driver.find_elements(By.XPATH, xpath_filtro)
                    if checkbox:
                        if not checkbox[0].is_selected():
                            scraper_logger.info(f"Activando filtro: {nombre_filtro}")
                            self._clic_seguro(checkbox[0])
                            filtros[nombre_filtro] = True
                        else:
                            filtros[nombre_filtro] = True
                            scraper_logger.info(f"Filtro ya estaba activado: {nombre_filtro}")
            except Exception as e:
                scraper_logger.warning(f"Error al verificar filtros: {str(e)} - Continuando con refinamiento")

            # Paso 4: Activar la búsqueda refinada
            try:
                campo_refinamiento.send_keys(Keys.ENTER)
                scraper_logger.info("Refinamiento activado con tecla Enter")
                time.sleep(2)
            except Exception as e:
                scraper_logger.warning(f"Error al enviar Enter: {str(e)} - Intentando con botón")
                try:
                    script_boton_cercano = """
                    function encontrarBotonCercano(campo) {
                        const rect = campo.getBoundingClientRect();
                        const botones = Array.from(document.querySelectorAll('button'));
                        return botones.sort((a, b) => {
                            const rectA = a.getBoundingClientRect();
                            const rectB = b.getBoundingClientRect();
                            const distA = Math.sqrt(Math.pow(rectA.left - rect.right, 2) +
                                                   Math.pow(rectA.top - rect.top, 2));
                            const distB = Math.sqrt(Math.pow(rectB.left - rect.right, 2) +
                                                   Math.pow(rectB.top - rect.top, 2));
                            return distA - distB;
                        })[0];
                    }
                    return encontrarBotonCercano(arguments[0]);
                    """
                    boton_cercano = self.driver.execute_script(script_boton_cercano, campo_refinamiento)
                    if boton_cercano:
                        self._clic_seguro(boton_cercano)
                        scraper_logger.info("Refinamiento activado con botón cercano")
                        time.sleep(2)
                except Exception as e:
                    scraper_logger.error(f"Error al buscar y activar botón de refinamiento: {str(e)}")
                    self._captura_debug("error_activar_refinamiento")
                    return False

            # Paso 5: Verificar que el refinamiento se aplicó
            try:
                time.sleep(2)
                filtro_aplicado = self.driver.find_elements(
                    By.XPATH,
                    f"//span[contains(text(), '{termino_refinamiento}')]"
                )
                if filtro_aplicado:
                    scraper_logger.info(f"Filtro aplicado encontrado: {filtro_aplicado[0].text}")
                    return True

                elementos_resultado = self._esperar_elementos(
                    By.CSS_SELECTOR,
                    ".registro-item, .resultado-item, div[class*='resultado']",
                    timeout=3
                )
                if elementos_resultado:
                    scraper_logger.info(f"Refinamiento exitoso. Se encontraron {len(elementos_resultado)} resultados")
                    return True

                mensaje_no_resultados = self.driver.find_elements(
                    By.XPATH,
                    "//div[contains(text(), 'No se encontraron') or contains(text(), 'Sin resultados')]"
                )
                if mensaje_no_resultados:
                    scraper_logger.info("El sistema indica que no hay resultados para la búsqueda refinada")
                    return True

                scraper_logger.warning("No se pudo verificar si el refinamiento se aplicó correctamente")
                return False

            except Exception as e:
                scraper_logger.error(f"Error al verificar aplicación de refinamiento: {str(e)}")
                self._captura_debug("error_verificar_refinamiento")
                return False

        except Exception as e:
            scraper_logger.error(f"Error en refinamiento de búsqueda: {str(e)}")
            self._captura_debug("error_refinamiento")
            return False

    def _obtener_total_resultados(self):
        """
        Obtiene el número total de resultados de la búsqueda actual utilizando
        múltiples estrategias para garantizar resultados precisos.

        Returns:
            int: Número total de resultados, o 0 si no se puede determinar
        """
        try:
            # Estrategia 1: Buscar elementos de información de resultados específicos
            selectores_info = [
                "div.info-resultados", 
                "span.info-resultados", 
                "div[class*='resultado-count']",
                ".pagination-info",
                "div[id*='info-resultados']"
            ]
            
            for selector in selectores_info:
                elementos = self.driver.find_elements(By.CSS_SELECTOR, selector)
                for elemento in elementos:
                    texto = elemento.text
                    match = re.search(r'(\d+)\s+(?:resultado|tesis)', texto, re.IGNORECASE)
                    if match:
                        total = int(match.group(1))
                        scraper_logger.info(f"Total de resultados encontrado en elemento info: {total}")
                        return total

            # Estrategia 2: Buscar patrones específicos en cualquier texto
            patrones_xpath = [
                "//div[contains(text(), 'Registros:')]",
                "//span[contains(text(), 'Registros:')]",
                "//div[contains(text(), 'al 50 de')]",
                "//span[contains(text(), 'al 50 de')]",
                "//div[contains(text(), 'resultado')]",
                "//span[contains(text(), 'resultado')]"
            ]

            for xpath in patrones_xpath:
                elementos = self.driver.find_elements(By.XPATH, xpath)
                for elemento in elementos:
                    texto = elemento.text
                    # Buscar "X resultados" o "Registros: X"
                    match = re.search(r'(\d+)\s+(?:resultado|tesis|registro)', texto, re.IGNORECASE)
                    if match:
                        total = int(match.group(1))
                        scraper_logger.info(f"Total de resultados encontrado en texto: {total}")
                        return total
                    
                    # Buscar patrón "1 al 20 de X"
                    match = re.search(r'al\s+\d+\s+de\s+(\d+)', texto, re.IGNORECASE)
                    if match:
                        total = int(match.group(1))
                        scraper_logger.info(f"Total de resultados encontrado en paginación: {total}")
                        return total

            # Estrategia 3: Analizar la paginación
            elementos_paginacion = self.driver.find_elements(By.CSS_SELECTOR, "ul.pagination li, nav[aria-label*='pagin'] li")
            if elementos_paginacion and len(elementos_paginacion) > 2:
                try:
                    # Intentar obtener el número de la última página
                    textos_paginas = [elem.text for elem in elementos_paginacion if elem.text.strip().isdigit()]
                    if textos_paginas:
                        ultima_pagina = max([int(texto) for texto in textos_paginas])
                        
                        # Contar resultados en la página actual
                        elementos_resultado = self.driver.find_elements(
                            By.CSS_SELECTOR, 
                            ".registro-item, .resultado-item, div[class*='resultado']"
                        )
                        resultados_por_pagina = len(elementos_resultado)
                        
                        # Estimar total
                        total_estimado = ultima_pagina * resultados_por_pagina
                        scraper_logger.info(f"Total estimado por paginación: {total_estimado} (páginas: {ultima_pagina}, resultados/pág: {resultados_por_pagina})")
                        return total_estimado
                except Exception as e:
                    scraper_logger.debug(f"Error al analizar paginación: {str(e)}")

            # Estrategia 4: Contar los resultados en la página actual
            elementos_resultado = self.driver.find_elements(
                By.CSS_SELECTOR,
                ".registro-item, .resultado-item, div[class*='resultado']"
            )
            if elementos_resultado:
                total_contado = len(elementos_resultado)
                scraper_logger.info(f"Total contado en la página actual: {total_contado}")
                return total_contado

            # Si ninguna estrategia funciona, intentar obtener mediante JavaScript
            try:
                script_total = """
                function encontrarTotalResultados() {
                    // Buscar texto que contenga números seguidos de "resultados"
                    const textos = [];
                    document.querySelectorAll('div, span, p').forEach(elem => {
                        const texto = elem.innerText || '';
                        if (texto.match(/\\d+\\s+(?:resultado|tesis|registro)/i)) {
                            textos.push({elem, texto});
                        }
                    });
                    
                    // Buscar patrones específicos en los textos encontrados
                    for (const {texto} of textos) {
                        const matchResultados = texto.match(/(\\d+)\\s+(?:resultado|tesis|registro)/i);
                        if (matchResultados) return parseInt(matchResultados[1]);
                        
                        const matchPaginacion = texto.match(/al\\s+\\d+\\s+de\\s+(\\d+)/i);
                        if (matchPaginacion) return parseInt(matchPaginacion[1]);
                    }
                    
                    // Contar resultados en la página
                    const resultados = document.querySelectorAll('.registro-item, .resultado-item, div[class*="resultado"]');
                    if (resultados.length > 0) return resultados.length;
                    
                    return 0;
                }
                return encontrarTotalResultados();
                """
                total_js = self.driver.execute_script(script_total)
                if total_js > 0:
                    scraper_logger.info(f"Total obtenido con JavaScript: {total_js}")
                    return total_js
            except Exception as e:
                scraper_logger.debug(f"Error al ejecutar script para total: {str(e)}")

            scraper_logger.warning("No se pudo determinar el total de resultados con ninguna estrategia")
            return 0

        except Exception as e:
            scraper_logger.error(f"Error general al obtener total de resultados: {str(e)}")
            return 0

    def _procesar_resultados_paginados(self, numero_resultados):
        """
        Procesa los resultados paginados hasta obtener el número deseado.
        Implementa detección y manejo de duplicados y asegura conteo preciso.

        Args:
            numero_resultados (int): Número máximo de resultados a obtener

        Returns:
            list: Lista de resultados procesados
        """
        resultados_completos = []
        pagina_actual = 1
        max_paginas = (numero_resultados + 19) // 20  # Aproximadamente 20 por página

        # Conjunto para mantener registro de IDs ya procesados y evitar duplicados
        registros_procesados = set()
        resultados_duplicados = 0

        scraper_logger.info(f"Iniciando recopilación de hasta {numero_resultados} resultados")

        while len(resultados_completos) < numero_resultados and pagina_actual <= max_paginas:
            try:
                # Esperar carga completa de la página
                time.sleep(1)  # Pausa breve para asegurar carga
                
                # Intentar identificar elementos de carga y esperar a que desaparezcan
                elementos_carga = self.driver.find_elements(By.CSS_SELECTOR, ".loading, .spinner, [class*='carga']")
                if elementos_carga:
                    scraper_logger.info("Esperando a que termine la carga...")
                    time.sleep(2)
                
                resultados_pagina = self._extraer_resultados(pagina_actual)
                if not resultados_pagina:
                    scraper_logger.warning(f"No se encontraron resultados en la página {pagina_actual}")
                    break

                self.estadisticas["paginas_procesadas"] += 1

                resultados_nuevos = 0
                for resultado in resultados_pagina:
                    registro = resultado.get('registro_digital')
                    if registro != "No identificado" and registro not in registros_procesados:
                        # Agregar URL de tesis al resultado
                        if 'url_tesis' not in resultado:
                            resultado['url_tesis'] = self.generar_url_registro(registro)
                        
                        # Intentar mejorar la extracción de datos con técnicas avanzadas
                        self._enriquecer_resultado(resultado)
                        
                        registros_procesados.add(registro)
                        resultados_completos.append(resultado)
                        resultados_nuevos += 1
                    else:
                        resultados_duplicados += 1

                self.estadisticas["resultados_obtenidos"] += resultados_nuevos

                scraper_logger.info(f"Página {pagina_actual}: {len(resultados_pagina)} resultados "
                                    f"({resultados_nuevos} nuevos, {len(resultados_pagina) - resultados_nuevos} duplicados). "
                                    f"Total: {len(resultados_completos)}/{numero_resultados}")

                if resultados_nuevos == 0:
                    scraper_logger.info("No se encontraron nuevos resultados en esta página. Terminando búsqueda.")
                    break

                if len(resultados_completos) >= numero_resultados:
                    break

                if not self._ir_siguiente_pagina():
                    scraper_logger.info("No se pudo navegar a la siguiente página o ya estamos en la última")
                    break

                pagina_actual += 1
                time.sleep(1)

            except Exception as e:
                scraper_logger.error(f"Error procesando resultados de la página {pagina_actual}: {str(e)}")
                break

        # Limitar al número exacto solicitado
        if len(resultados_completos) > numero_resultados:
            resultados_completos = resultados_completos[:numero_resultados]

        if resultados_duplicados > 0:
            scraper_logger.info(f"Total de resultados duplicados filtrados: {resultados_duplicados}")

        # Verificar coherencia con el total reportado
        total_reportado = self.estadisticas.get("total_general", 0)
        if total_reportado > 0 and len(resultados_completos) != total_reportado and len(resultados_completos) < numero_resultados:
            scraper_logger.info(f"Nota: Se obtuvieron {len(resultados_completos)} resultados de {total_reportado} reportados")

        # Actualizar estadísticas con el recuento real
        self.estadisticas["resultados_obtenidos"] = len(resultados_completos)
        
        return resultados_completos


def ejecutar_scraper_compuesto(termino_general, termino_refinamiento=None, headless=False,
                              archivo_salida="tesis_scjn", numero_resultados=50,
                              guardar_json=True, modo_debug=False, timeout=20,
                              incluir_8a_epoca=False, incluir_7a_epoca=False,
                              incluir_6a_epoca=False, incluir_5a_epoca=False,
                              tipo_tesis=0):
    """
    Ejecuta el scraper con búsqueda compuesta (general + refinamiento opcional).

    Args:
        termino_general (str): Término general para la búsqueda inicial
        termino_refinamiento (str): Término para refinar los resultados (opcional)
        headless (bool): Si se ejecuta en modo headless
        archivo_salida (str): Nombre base del archivo de salida (sin extensión)
        numero_resultados (int): Número de resultados a obtener
        guardar_json (bool): Si se guardan también resultados en formato JSON
        modo_debug (bool): Activa capturas de pantalla para depuración
        timeout (int): Tiempo máximo de espera en segundos
        incluir_8a_epoca (bool): Si se incluye la 8ª Época en la búsqueda
        incluir_7a_epoca (bool): Si se incluye la 7ª Época en la búsqueda
        incluir_6a_epoca (bool): Si se incluye la 6ª Época en la búsqueda
        incluir_5a_epoca (bool): Si se incluye la 5ª Época en la búsqueda
        tipo_tesis (int): Tipo de tesis a buscar (0: Todas, 1: Jurisprudencia, 2: Aislada)

    Returns:
        bool: True si se completó correctamente, False en caso contrario
    """
    ejecutor_logger.info("=== INICIANDO EJECUCIÓN SCRAPER COMPUESTO ===")
    ejecutor_logger.info(f"Término general: '{termino_general}'")
    if termino_refinamiento:
        ejecutor_logger.info(f"Término de refinamiento: '{termino_refinamiento}'")
    else:
        ejecutor_logger.info("No se especificó término de refinamiento")

    ejecutor_logger.info(f"Resultados a obtener: {numero_resultados}")
    ejecutor_logger.info(f"Salida base: {archivo_salida}")
    ejecutor_logger.info(f"Headless: {headless}, Debug: {modo_debug}, Timeout: {timeout}")
    ejecutor_logger.info(f"Incluir 8ª Época: {incluir_8a_epoca}, Incluir 7ª Época: {incluir_7a_epoca}")
    ejecutor_logger.info(f"Incluir 6ª Época: {incluir_6a_epoca}, Incluir 5ª Época: {incluir_5a_epoca}")
    ejecutor_logger.info(f"Tipo de tesis: {tipo_tesis}")

    inicio_tiempo = time.time()
    scraper = None

    try:
        scraper = ScraperTesisCompuesto(
            headless=headless,
            modo_debug=modo_debug,
            timeout_default=timeout,
            tipo_tesis=tipo_tesis  # Pasar tipo de tesis
        )
        
        # MEJORA: Asegurarse de que usar_descarga_directa esté activado (a menos que se desactive explícitamente)
        scraper.usar_descarga_directa = True

        resultado_busqueda = scraper.buscar_tesis_compuesta(
            termino_general=termino_general,
            termino_refinamiento=termino_refinamiento,
            numero_resultados=numero_resultados,
            incluir_8a_epoca=incluir_8a_epoca,
            incluir_7a_epoca=incluir_7a_epoca,
            incluir_6a_epoca=incluir_6a_epoca,
            incluir_5a_epoca=incluir_5a_epoca,
            tipo_tesis=tipo_tesis  # Pasar tipo de tesis
        )

        if resultado_busqueda.get("tipo_busqueda") == "fallida":
            ejecutor_logger.error(f"La búsqueda falló: {resultado_busqueda.get('error', 'Error desconocido')}")
            return False

        resultados = resultado_busqueda.get("resultados", [])
        if not resultados:
            ejecutor_logger.warning("No se obtuvieron resultados en la búsqueda")
            return False

        ejecutor_logger.info(f"Se obtuvieron {len(resultados)} resultados")

        sufijo = ""
        if termino_refinamiento:
            sufijo = "_" + termino_refinamiento.replace("*", "").replace(" ", "_").lower()

        archivo_txt = f"{archivo_salida}{sufijo}.txt"
        exito_txt = scraper.guardar_resultados(resultados, archivo_txt)

        exito_json = True
        if guardar_json:
            archivo_json = f"{archivo_salida}{sufijo}.json"
            datos_json = {
                "metadata": {
                    "fecha_busqueda": datetime.now().isoformat(),
                    "termino_general": termino_general,
                    "termino_refinamiento": termino_refinamiento,
                    "incluir_8a_epoca": incluir_8a_epoca,
                    "incluir_7a_epoca": incluir_7a_epoca,
                    "incluir_6a_epoca": incluir_6a_epoca,
                    "incluir_5a_epoca": incluir_5a_epoca,
                    "tipo_tesis": tipo_tesis,
                    "tipo_busqueda": "refinada" if termino_refinamiento else "general",
                    "total_resultados_general": resultado_busqueda.get("total_general", 0),
                    "total_resultados_refinados": resultado_busqueda.get("total_refinado", 0) if termino_refinamiento else 0,
                    "resultados_obtenidos": len(resultados),
                    "tiempo_ejecucion": time.time() - inicio_tiempo,
                    "paginas_procesadas": resultado_busqueda["estadisticas"]["paginas_procesadas"],
                    "metodo_obtencion": resultado_busqueda.get("metodo_obtencion", "tradicional")
                },
                "resultados": resultados
            }

            try:
                with open(archivo_json, 'w', encoding='utf-8') as f:
                    json.dump(datos_json, f, ensure_ascii=False, indent=2)
                ejecutor_logger.info(f"Resultados guardados en JSON: {archivo_json}")
                exito_json = True
            except Exception as e:
                ejecutor_logger.error(f"Error al guardar en JSON: {str(e)}")
                exito_json = False

        archivo_urls = f"{archivo_salida}{sufijo}_urls.txt"
        exito_urls = scraper.guardar_solo_urls(resultados, archivo_urls)

        tiempo_total = time.time() - inicio_tiempo
        ejecutor_logger.info(f"Proceso completado en {tiempo_total:.2f} segundos")

        exito_final = exito_txt and exito_json and exito_urls

        if exito_final:
            print("\n¡Proceso completado con éxito!")
            print(f"Resultados guardados en: {archivo_salida}{sufijo}.txt")
            print(f"Datos en formato JSON guardados en: {archivo_salida}{sufijo}.json")
            print(f"LISTA DE URLS guardada en: {archivo_urls}")

        return exito_final

    except Exception as e:
        ejecutor_logger.error(f"La búsqueda falló gravemente: {str(e)}")
        if modo_debug:
            traceback.print_exc()
        return False
    finally:
        ejecutor_logger.info("Cerrando el scraper...")
        if scraper:
            try:
                scraper.cerrar()
            except Exception as e:
                ejecutor_logger.error(f"Error al cerrar el scraper: {str(e)}")


if __name__ == "__main__":
    main_logger.info("\n" + "=" * 40)
    main_logger.info("=== SCRAPER COMPUESTO DE TESIS JUDICIALES (SCJN) ===")
    main_logger.info("=" * 40)

    try:
        termino_general = input("Ingrese término de búsqueda general: ")
        if not termino_general:
            main_logger.error("El término de búsqueda general es obligatorio")
            sys.exit(1)

        termino_refinamiento = input("Ingrese término de refinamiento (opcional): ")
        if termino_refinamiento.strip() == "":
            termino_refinamiento = None

        headless_input = input("¿Ejecutar en modo sin interfaz gráfica (headless)? (S/n): ").lower()
        headless = not headless_input.startswith('n')  # Por defecto sí

        archivo_salida = input("Nombre base del archivo de salida [tesis_scjn]: ")
        if not archivo_salida:
            archivo_salida = "tesis_scjn"

        numero_resultados_str = input("Número MÁXIMO de resultados a obtener [50]: ")
        numero_resultados = int(numero_resultados_str) if numero_resultados_str else 50

        # Seleccionar épocas adicionales
        incluir_8a_epoca_input = input("¿Incluir 8ª Época en la búsqueda? (s/N): ").lower()
        incluir_8a_epoca = incluir_8a_epoca_input.startswith('s')  # Por defecto no

        incluir_7a_epoca_input = input("¿Incluir 7ª Época en la búsqueda? (s/N): ").lower()
        incluir_7a_epoca = incluir_7a_epoca_input.startswith('s')  # Por defecto no

        # Nuevas entradas para 6ª y 5ª Épocas
        incluir_6a_epoca_input = input("¿Incluir 6ª Época en la búsqueda? (s/N): ").lower()
        incluir_6a_epoca = incluir_6a_epoca_input.startswith('s')  # Por defecto no

        incluir_5a_epoca_input = input("¿Incluir 5ª Época en la búsqueda? (s/N): ").lower()
        incluir_5a_epoca = incluir_5a_epoca_input.startswith('s')  # Por defecto no

        # NUEVO: Seleccionar tipo de tesis
        print("\nSeleccione el tipo de tesis:")
        print("1. Tesis jurisprudenciales y aisladas (default)")
        print("2. Jurisprudencia")
        print("3. Aislada")
        tipo_tesis_input = input("Opción [1]: ")

        # Convertir la entrada a un valor de 0 a 2
        tipo_tesis = 0  # Valor predeterminado
        if tipo_tesis_input:
            try:
                seleccion = int(tipo_tesis_input)
                if 1 <= seleccion <= 3:  # Validar rango
                    tipo_tesis = seleccion - 1  # Restar 1 para que sea 0-2
            except ValueError:
                pass  # Si hay error, usar el valor predeterminado

        # MEJORA: Preguntar si usar descarga directa (por defecto sí)
        usar_descarga_directa_input = input("¿Usar método optimizado de descarga directa? (S/n): ").lower()
        usar_descarga_directa = not usar_descarga_directa_input.startswith('n')  # Por defecto sí

        modo_debug_input = input("¿Activar modo debug con capturas de pantalla? (s/N): ").lower()
        modo_debug = modo_debug_input.startswith('s')  # Por defecto no

        exito_final = ejecutar_scraper_compuesto(
            termino_general=termino_general,
            termino_refinamiento=termino_refinamiento,
            headless=headless,
            archivo_salida=archivo_salida,
            numero_resultados=numero_resultados,
            guardar_json=True,
            modo_debug=modo_debug,
            timeout=20,
            incluir_8a_epoca=incluir_8a_epoca,
            incluir_7a_epoca=incluir_7a_epoca,
            incluir_6a_epoca=incluir_6a_epoca,
            incluir_5a_epoca=incluir_5a_epoca,
            tipo_tesis=tipo_tesis  # Pasar tipo de tesis
        )

        if exito_final:
            print("\n¡Proceso completado con éxito!")
            sufijo = "_" + termino_refinamiento.replace("*", "").replace(" ", "_").lower() if termino_refinamiento else ""
            print(f"Resultados guardados en: {archivo_salida}{sufijo}.txt")
            print(f"Datos en formato JSON guardados en: {archivo_salida}{sufijo}.json")
            print(f"LISTA DE URLS guardada en: {archivo_salida}{sufijo}_urls.txt")
        else:
            print("\nEl proceso no se completó correctamente.")
            print("Revise el archivo 'scraper_compuesto.log' para más detalles.")

    except KeyboardInterrupt:
        print("\nOperación cancelada por el usuario.")
        sys.exit(0)
    except Exception as e:
        main_logger.critical(f"Error fatal no esperado en **main**: {str(e)}")
        traceback.print_exc()
        print(f"ERROR FATAL: {str(e)}. Revise el log.")
        sys.exit(1)