"""
Integrador de Scraper SCJN y Extractor de PDFs
Este script integra la funcionalidad del scraper de tesis judiciales
con la extracción automática de texto de los documentos PDF asociados.

Desarrollado para uso en investigación jurídica.
"""

import os
import sys
import time
import requests
import fitz  # PyMuPDF
import json
import logging
import traceback
import re
import unicodedata
import subprocess
from datetime import datetime
from io import BytesIO
import PyPDF2  # Importación para la compilación de PDFs

# Determinar si estamos ejecutando como script o como ejecutable
if getattr(sys, 'frozen', False):
    # Si es ejecutable, usar sys._MEIPASS para encontrar recursos
    base_path = sys._MEIPASS
else:
    # Si es script, usar el directorio actual
    base_path = os.path.dirname(os.path.abspath(__file__))

# Agregar la ruta base al sys.path para que pueda encontrar los módulos
sys.path.insert(0, base_path)
# Intentar importar las funciones del scraper y el extractor de PDFs
try:
# Intentar importar con manejo de rutas para PyInstaller
    SCRAPER_IMPORTADO = False
    error_importacion_scraper = "No inicializado"
    try:
        # Importación directa
# Intentar importar con manejo de rutas para PyInstaller
        SCRAPER_IMPORTADO = False
        error_importacion_scraper = "No inicializado"
        try:
            # Importación directa
            from scraper_compuesto7 import ejecutar_scraper_compuesto, ScraperTesisCompuesto
            SCRAPER_IMPORTADO = True
            error_importacion_scraper = None
            print("Importación directa de 'scraper_compuesto7' exitosa.")
        except ImportError as e:
            error_importacion_scraper = f"Error importación directa scraper: {str(e)}"
            print(f"Fallo importación directa scraper: {error_importacion_scraper}")
            try:
                # Intentar carga dinámica desde la ruta del ejecutable/script
                import os
                import sys
                import importlib.util

                # Determinar directorio base
                if getattr(sys, 'frozen', False):
                    base_dir = os.path.dirname(sys.executable)
                else:
                    base_dir = os.path.dirname(os.path.abspath(__file__))

                # Intentar cargar el módulo dinámicamente
                scraper_path = os.path.join(base_dir, "scraper_compuesto7.py")
                print(f"Intentando carga dinámica de scraper desde: {scraper_path}")

                if os.path.exists(scraper_path):
                    spec = importlib.util.spec_from_file_location("scraper_compuesto7", scraper_path)
                    if spec and spec.loader:
                        scraper_module = importlib.util.module_from_spec(spec)
                        sys.modules["scraper_compuesto7"] = scraper_module # Registrar antes de ejecutar
                        spec.loader.exec_module(scraper_module)
                        # Verificar existencia de las importaciones necesarias
                        if hasattr(scraper_module, 'ejecutar_scraper_compuesto') and \
                           hasattr(scraper_module, 'ScraperTesisCompuesto'):
                            ejecutar_scraper_compuesto = scraper_module.ejecutar_scraper_compuesto
                            ScraperTesisCompuesto = scraper_module.ScraperTesisCompuesto
                            SCRAPER_IMPORTADO = True
                            error_importacion_scraper = None
                            print("Módulo 'scraper_compuesto7' cargado dinámicamente con éxito")
                        else:
                            error_importacion_scraper += " | Carga dinámica OK, pero faltan elementos importados"
                            print("Error: Módulo scraper cargado, pero faltan 'ejecutar_scraper_compuesto' o 'ScraperTesisCompuesto'")
                    else:
                        error_importacion_scraper += " | No se pudo crear especificación de módulo para scraper"
                        print("Error: No se pudo crear la especificación del módulo scraper.")
                else:
                    error_importacion_scraper += f" | Archivo scraper no encontrado: {scraper_path}"
                    print(f"ERROR: No se encontró scraper_compuesto7.py en {base_dir}")
            except Exception as e2:
                error_importacion_scraper += f" | Error fatal en carga alternativa scraper: {str(e2)}"
                print(f"ERROR crítico al importar scraper_compuesto7.py: {str(e2)}")
                traceback.print_exc()

        # Asegurarse de que las variables estén definidas
        if not SCRAPER_IMPORTADO:
            print("ADVERTENCIA: No se pudo importar 'scraper_compuesto7' de ninguna forma.")
            # Definir dummies para evitar errores
            def ejecutar_scraper_compuesto(*args, **kwargs):
                raise ImportError("Dependencia 'scraper_compuesto7.ejecutar_scraper_compuesto' no encontrada.")
            class ScraperTesisCompuesto:
                def __init__(self, *args, **kwargs):
                    raise ImportError("Dependencia 'scraper_compuesto7.ScraperTesisCompuesto' no encontrada.")
        SCRAPER_IMPORTADO = True
        error_importacion_scraper = None
        print("Importación directa de 'scraper_compuesto7' exitosa.")
    except ImportError as e:
        error_importacion_scraper = f"Error importación directa scraper: {str(e)}"
        print(f"Fallo importación directa scraper: {error_importacion_scraper}")
        try:
            # Intentar carga dinámica desde la ruta del ejecutable/script
            import os
            import sys
            import importlib.util

            # Determinar directorio base
            if getattr(sys, 'frozen', False):
                base_dir = os.path.dirname(sys.executable)
            else:
                base_dir = os.path.dirname(os.path.abspath(__file__))

            # Intentar cargar el módulo dinámicamente
            scraper_path = os.path.join(base_dir, "scraper_compuesto7.py")
            print(f"Intentando carga dinámica de scraper desde: {scraper_path}")

            if os.path.exists(scraper_path):
                spec = importlib.util.spec_from_file_location("scraper_compuesto7", scraper_path)
                if spec and spec.loader:
                    scraper_module = importlib.util.module_from_spec(spec)
                    sys.modules["scraper_compuesto7"] = scraper_module # Registrar antes de ejecutar
                    spec.loader.exec_module(scraper_module)
                    # Verificar existencia de las importaciones necesarias
                    if hasattr(scraper_module, 'ejecutar_scraper_compuesto') and \
                       hasattr(scraper_module, 'ScraperTesisCompuesto'):
                        ejecutar_scraper_compuesto = scraper_module.ejecutar_scraper_compuesto
                        ScraperTesisCompuesto = scraper_module.ScraperTesisCompuesto
                        SCRAPER_IMPORTADO = True
                        error_importacion_scraper = None
                        print("Módulo 'scraper_compuesto7' cargado dinámicamente con éxito")
                    else:
                        error_importacion_scraper += " | Carga dinámica OK, pero faltan elementos importados"
                        print("Error: Módulo scraper cargado, pero faltan 'ejecutar_scraper_compuesto' o 'ScraperTesisCompuesto'")
                else:
                    error_importacion_scraper += " | No se pudo crear especificación de módulo para scraper"
                    print("Error: No se pudo crear la especificación del módulo scraper.")
            else:
                error_importacion_scraper += f" | Archivo scraper no encontrado: {scraper_path}"
                print(f"ERROR: No se encontró scraper_compuesto7.py en {base_dir}")
        except Exception as e2:
            error_importacion_scraper += f" | Error fatal en carga alternativa scraper: {str(e2)}"
            print(f"ERROR crítico al importar scraper_compuesto7.py: {str(e2)}")
            traceback.print_exc()

    # Asegurarse de que las variables estén definidas
    if not SCRAPER_IMPORTADO:
        print("ADVERTENCIA: No se pudo importar 'scraper_compuesto7' de ninguna forma.")
        # Definir dummies para evitar errores
        def ejecutar_scraper_compuesto(*args, **kwargs):
            raise ImportError("Dependencia 'scraper_compuesto7.ejecutar_scraper_compuesto' no encontrada.")
        class ScraperTesisCompuesto:
            def __init__(self, *args, **kwargs):
                raise ImportError("Dependencia 'scraper_compuesto7.ScraperTesisCompuesto' no encontrada.")
    SCRAPER_IMPORTADO = True
except ImportError:
    SCRAPER_IMPORTADO = False
    print("ADVERTENCIA: No se pudo importar el scraper de tesis.")
    print("Coloque este script en la misma carpeta que scraper_compuesto7.py")

# Configuración de logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("integrador_tesis.log"),
        logging.StreamHandler()
    ]
)

# Crear logger para este script
logger = logging.getLogger("IntegradorTesis")

def verificar_coherencia_resultados(archivo_json):
    """
    Verifica y corrige la coherencia entre el conteo reportado y el número real
    de resultados en un archivo JSON.
    
    Args:
        archivo_json (str): Ruta al archivo JSON a verificar
        
    Returns:
        tuple: (conteo_real, conteo_reportado, corregido)
    """
    try:
        if not os.path.exists(archivo_json):
            return 0, 0, False
            
        with open(archivo_json, 'r', encoding='utf-8') as f:
            datos = json.load(f)
            
        resultados = datos.get('resultados', [])
        conteo_real = len(resultados)
        
        # Obtener conteo reportado de diferentes lugares posibles
        conteo_reportado = datos.get('metadata', {}).get('total_resultados', 0)
        if conteo_reportado == 0:
            conteo_reportado = datos.get('total', 0)
        
        # Si hay discrepancia, corregir
        if conteo_reportado != conteo_real and conteo_real > 0:
            logger.info(f"Corrigiendo discrepancia en conteo: reportado={conteo_reportado}, real={conteo_real}")
            
            # Actualizar metadata
            if 'metadata' in datos:
                datos['metadata']['total_resultados'] = conteo_real
            else:
                datos['metadata'] = {'total_resultados': conteo_real}
            
            # Añadir/actualizar total en la raíz
            datos['total'] = conteo_real
            
            # Reescribir archivo
            with open(archivo_json, 'w', encoding='utf-8') as f:
                json.dump(datos, f, ensure_ascii=False, indent=2)
                
            return conteo_real, conteo_reportado, True
        
        return conteo_real, conteo_reportado, False
        
    except Exception as e:
        logger.error(f"Error al verificar coherencia de resultados: {str(e)}")
        return 0, 0, False

def extraer_texto_pdf_desde_url(url, incluir_metadatos=True, guardar_pdf_original=False, directorio_pdfs=None):
    """
    Extrae el texto de un PDF desde una URL.
    
    Args:
        url (str): URL del archivo PDF
        incluir_metadatos (bool): Si se incluye información sobre el origen
        guardar_pdf_original (bool): Si se debe guardar el PDF original
        directorio_pdfs (str): Directorio donde guardar los PDFs originales
        
    Returns:
        dict: Diccionario con estado de la operación, texto extraído y ruta del PDF original
    """
    resultado = {
        "exito": False,
        "texto": "",
        "ruta_pdf": None,
        "error": None
    }
    
    try:
        logger.info(f"Procesando URL: {url}")
        
        # Descargar el PDF en memoria
        response = requests.get(url, timeout=20)
        response.raise_for_status()
        
        # Determinar si estamos tratando con un PDF
        content_type = response.headers.get('Content-Type', '').lower()
        es_pdf = 'application/pdf' in content_type or response.content.startswith(b'%PDF-')
        
        # Guardar el PDF original si se solicita
        ruta_pdf_original = None
        if guardar_pdf_original and es_pdf and directorio_pdfs:
            # Crear nombre de archivo para el PDF
            nombre_pdf = f"tesis_{datetime.now().strftime('%Y%m%d_%H%M%S')}_{os.urandom(4).hex()}.pdf"
            ruta_pdf_original = os.path.join(directorio_pdfs, nombre_pdf)
            
            # Guardar el PDF
            with open(ruta_pdf_original, 'wb') as f_pdf:
                f_pdf.write(response.content)
            
            logger.info(f"PDF original guardado en {ruta_pdf_original}")
            resultado["ruta_pdf"] = ruta_pdf_original
        
        # Si el Content-Type indica que no es un PDF, verificar contenido
        if not es_pdf:
            logger.warning(f"El contenido en {url} no parece ser un PDF válido.")
            # Si no es PDF, guardar el contenido como texto si es texto
            if 'text/' in content_type:
                texto_contenido = response.content.decode('utf-8', errors='replace')
                resultado["texto"] = texto_contenido
                resultado["exito"] = True
                return resultado
            
            resultado["error"] = "El contenido no es un PDF válido ni texto"
            return resultado
        
        # Procesar como PDF
        try:
            doc = fitz.open(stream=response.content, filetype="pdf")
        except Exception as e:
            logger.error(f"Error al abrir PDF desde {url}: {str(e)}")
            resultado["error"] = f"Error al abrir PDF: {str(e)}"
            return resultado
        
        # Extraer texto
        texto_completo = ""
        
        # Incluir metadatos si se solicita
        if incluir_metadatos:
            texto_completo += "=" * 80 + "\n"
            texto_completo += f"TEXTO EXTRAÍDO DE PDF\n"
            texto_completo += f"URL: {url}\n"
            texto_completo += f"Fecha de extracción: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n"
            texto_completo += "=" * 80 + "\n\n"
        
        # Extraer y concatenar el texto de cada página
        for i, pagina in enumerate(doc):
            texto_pagina = pagina.get_text()
            if incluir_metadatos:
                texto_completo += f"\n--- Página {i+1} ---\n\n"
            texto_completo += texto_pagina
            texto_completo += "\n"
        
        if incluir_metadatos:
            texto_completo += "\n" + "=" * 80 + "\n"
            texto_completo += "FIN DEL DOCUMENTO\n"
            texto_completo += "=" * 80 + "\n"
        
        doc.close()
        
        resultado["exito"] = True
        resultado["texto"] = texto_completo
        
        return resultado
    
    except Exception as e:
        logger.error(f"Error al procesar {url}: {str(e)}")
        resultado["error"] = str(e)
        return resultado

def compilar_pdfs_en_uno(lista_pdfs, archivo_salida):
    """
    Compila múltiples archivos PDF en uno solo.
    
    Args:
        lista_pdfs (list): Lista de rutas a los archivos PDF a compilar
        archivo_salida (str): Ruta donde guardar el PDF compilado
        
    Returns:
        bool: True si se completó con éxito, False en caso contrario
    """
    try:
        if not lista_pdfs:
            logger.warning("No hay PDFs para compilar")
            return False
        
        logger.info(f"Compilando {len(lista_pdfs)} PDFs en {archivo_salida}")
        
        # Crear un nuevo PDF
        pdf_merger = PyPDF2.PdfMerger()
        
        # Añadir cada PDF a la compilación
        for ruta_pdf in lista_pdfs:
            try:
                if os.path.exists(ruta_pdf) and os.path.getsize(ruta_pdf) > 0:
                    pdf_merger.append(ruta_pdf)
                    logger.debug(f"Añadido PDF: {ruta_pdf}")
                else:
                    logger.warning(f"PDF no encontrado o vacío: {ruta_pdf}")
            except Exception as e:
                logger.error(f"Error al añadir PDF {ruta_pdf}: {str(e)}")
                # Continuar con el siguiente PDF
        
        # Guardar el PDF compilado
        pdf_merger.write(archivo_salida)
        pdf_merger.close()
        
        logger.info(f"PDF compilado guardado en {archivo_salida}")
        return True
    
    except Exception as e:
        logger.error(f"Error al compilar PDFs: {str(e)}")
        return False

def procesar_lista_urls(archivo_urls, directorio_salida="textos_extraidos", prefijo="tesis_", 
                      archivo_compilado=None, guardar_pdfs=False, compilar_pdfs=False):
    """
    Procesa un archivo con lista de URLs y extrae el texto de cada PDF.
    OPTIMIZADO: Solo genera archivos compilados, no individuales.
    
    Args:
        archivo_urls (str): Ruta al archivo con las URLs
        directorio_salida (str): Directorio donde guardar los archivos compilados
        prefijo (str): Prefijo para los archivos de salida
        archivo_compilado (str): Ruta para el archivo compilado con todos los textos
        guardar_pdfs (bool): Si se deben guardar los PDFs temporales (para compilación PDF)
        compilar_pdfs (bool): Si se deben compilar todos los PDFs en uno solo
        
    Returns:
        tuple: (éxitos, fallos, total, ruta_compilado_texto, ruta_compilado_pdf)
    """
    # Verificar que el archivo existe
    if not os.path.exists(archivo_urls):
        logger.error(f"El archivo {archivo_urls} no existe.")
        return 0, 0, 0, None, None
    
    # Crear directorio de salida si no existe
    if not os.path.exists(directorio_salida):
        os.makedirs(directorio_salida)
        logger.info(f"Creado directorio de salida: {directorio_salida}")
    
    # Crear directorio TEMPORAL para PDFs solo si se va a compilar PDF
    directorio_pdfs = None
    if compilar_pdfs:
        directorio_pdfs = os.path.join(directorio_salida, "pdfs_temp")
        if not os.path.exists(directorio_pdfs):
            os.makedirs(directorio_pdfs)
            logger.info(f"Creado directorio temporal para PDFs: {directorio_pdfs}")
    
    # Cargar URLs desde el archivo
    with open(archivo_urls, 'r', encoding='utf-8') as f:
        urls = [url.strip() for url in f.readlines() if url.strip()]
    
    logger.info(f"Se encontraron {len(urls)} URLs para procesar.")
    
    # Procesar cada URL
    exitos = 0
    fallos = 0
    
    # Preparar archivo compilado (ahora siempre se crea)
    ruta_compilado_texto = archivo_compilado or os.path.join(directorio_salida, f"{prefijo}_COMPILADO.txt")
    # Crear encabezado del archivo compilado
    with open(ruta_compilado_texto, 'w', encoding='utf-8') as f_comp:
        f_comp.write("=" * 80 + "\n")
        f_comp.write("TEXTOS COMPILADOS DE TESIS JUDICIALES\n")
        f_comp.write(f"Fecha de extracción: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n")
        f_comp.write(f"Total de tesis procesadas: {len(urls)}\n")
        f_comp.write("=" * 80 + "\n\n")
    
    # Lista para guardar las rutas de los PDFs descargados temporalmente
    pdfs_descargados = []
    
    for i, url in enumerate(urls):
        try:
            print(f"[{i+1}/{len(urls)}] Procesando: {url}")
            
            # Extraer texto del PDF
            resultado = extraer_texto_pdf_desde_url(
                url, 
                guardar_pdf_original=compilar_pdfs, # Solo guardar si vamos a compilar PDF
                directorio_pdfs=directorio_pdfs
            )
            
            # Si se guardó el PDF, añadirlo a la lista
            if resultado.get("ruta_pdf"):
                pdfs_descargados.append(resultado["ruta_pdf"])
            
            # Si se obtuvo texto con éxito, añadir al archivo compilado
            if resultado["exito"]:
                try:
                    # Añadir al compilado con encabezado claro
                    with open(ruta_compilado_texto, 'a', encoding='utf-8') as f_comp:
                        f_comp.write("\n\n" + "=" * 80 + "\n")
                        f_comp.write(f"TESIS #{i+1}\n")
                        f_comp.write(f"URL: {url}\n")
                        f_comp.write("=" * 80 + "\n\n")
                        f_comp.write(resultado["texto"])
                        f_comp.write("\n\n" + "-" * 80 + "\n")
                except Exception as e:
                    logger.error(f"Error al compilar texto de {url}: {str(e)}")
            
            if resultado["exito"]:
                exitos += 1
                print(f"  ✅ Completado: Texto extraído correctamente")
            else:
                fallos += 1
                print(f"  ❌ Error al procesar URL: {resultado.get('error', 'Error desconocido')}")
            
        except Exception as e:
            logger.error(f"Error inesperado procesando {url}: {str(e)}")
            fallos += 1
    
    # Finalizar archivo compilado de texto
    try:
        # Verificar el total real procesado
        exitos_real = sum(1 for pdf in pdfs_descargados if os.path.exists(pdf) and os.path.getsize(pdf) > 0)
        
        # Añadir información exacta al archivo compilado
        with open(ruta_compilado_texto, 'a', encoding='utf-8') as f_comp:
            f_comp.write("\n\n" + "=" * 80 + "\n")
            f_comp.write(f"FIN DEL DOCUMENTO COMPILADO\n")
            f_comp.write(f"Total de tesis exitosas: {exitos} de {len(urls)}\n")
            f_comp.write("=" * 80 + "\n")
    except Exception as e:
        logger.error(f"Error al finalizar archivo compilado de texto: {str(e)}")
    
    # Compilar PDFs si se solicitó
    ruta_compilado_pdf = None
    if compilar_pdfs and pdfs_descargados:
        nombre_archivo_base = os.path.splitext(os.path.basename(archivo_compilado))[0] if archivo_compilado else prefijo
        ruta_compilado_pdf = os.path.join(directorio_salida, f"{nombre_archivo_base}_COMPILADO.pdf")
        
        if compilar_pdfs_en_uno(pdfs_descargados, ruta_compilado_pdf):
            logger.info(f"PDFs compilados exitosamente en {ruta_compilado_pdf}")
        else:
            logger.error("Falló la compilación de PDFs")
            ruta_compilado_pdf = None
    
    # SIEMPRE eliminar directorio temporal de PDFs después de compilar
    if directorio_pdfs and os.path.exists(directorio_pdfs):
        try:
            for pdf_file in pdfs_descargados:
                if os.path.exists(pdf_file):
                    os.remove(pdf_file)
            
            # Intentar eliminar el directorio
            if os.path.exists(directorio_pdfs):
                os.rmdir(directorio_pdfs)
                logger.info(f"Directorio temporal de PDFs eliminado: {directorio_pdfs}")
        except Exception as e:
            logger.warning(f"No se pudo eliminar completamente el directorio temporal: {str(e)}")
    
    return exitos, fallos, len(urls), ruta_compilado_texto, ruta_compilado_pdf

def sanitizar_nombre_archivo(nombre):
    """
    Sanitiza un string para usarlo como nombre de archivo válido.
    
    Args:
        nombre (str): Nombre original posiblemente con caracteres inválidos
        
    Returns:
        str: Nombre sanitizado seguro para usar como nombre de archivo
    """
    # 1. Eliminar caracteres no permitidos en nombres de archivo
    # Caracteres prohibidos en Windows: \ / : * ? " < > |
    nombre = re.sub(r'[\\/*?:"<>|]', '', nombre)
    
    # 2. Eliminar comillas (ya incluidas arriba, pero por si acaso)
    nombre = nombre.replace('"', '').replace("'", "")
    
    # 3. Reemplazar espacios por guiones bajos
    nombre = re.sub(r'\s+', '_', nombre)
    
    # 4. Normalizar acentos (opcional, Windows permite acentos)
    # nombre = unicodedata.normalize('NFKD', nombre).encode('ASCII', 'ignore').decode('ASCII')
    
    # 5. Limitar longitud (opcional)
    max_longitud = 100  # Ajustar según sea necesario
    if len(nombre) > max_longitud:
        nombre = nombre[:max_longitud]
    
    # 6. Asegurarse de que no termine con punto o espacio
    nombre = nombre.rstrip('. ')
    
    # 7. Si quedó vacío, usar un nombre predeterminado
    if not nombre:
        nombre = "archivo"
    
    return nombre

def obtener_urls_desde_json(archivo_json):
    """
    Extrae las URLs de un archivo JSON generado por el scraper.
    
    Args:
        archivo_json (str): Ruta al archivo JSON
        
    Returns:
        list: Lista de URLs
    """
    try:
        with open(archivo_json, 'r', encoding='utf-8') as f:
            datos = json.load(f)
        
        urls = []
        for resultado in datos.get('resultados', []):
            registro = resultado.get('registro_digital')
            if registro and registro != "No identificado":
                # Verificar si ya tiene URL generada
                if 'url_tesis' in resultado and resultado['url_tesis']:
                    urls.append(resultado['url_tesis'])
                else:
                    # Generar URL basada en el registro
                    scraper = ScraperTesisCompuesto(headless=True)
                    url = scraper.generar_url_registro(registro)
                    scraper.cerrar()
                    urls.append(url)
        
        return urls
    except Exception as e:
        logger.error(f"Error al extraer URLs desde JSON: {str(e)}")
        return []

def guardar_resultados_json(resultados, archivo_salida, estadisticas):
    """
    Guarda los resultados en formato JSON con metadatos precisos.
    
    Args:
        resultados (list): Lista de resultados a guardar
        archivo_salida (str): Ruta del archivo donde guardar los resultados
        estadisticas (dict): Diccionario con estadísticas del proceso
        
    Returns:
        bool: True si se completó con éxito, False en caso contrario
    """
    try:
        # Verificar conteo real de resultados
        conteo_real = len(resultados)
        
        # NUEVO: Detectar método de obtención
        metodo_obtencion = "tradicional"
        if any(r.get("metodo_obtencion") == "descarga_directa" for r in resultados if isinstance(r, dict)):
            metodo_obtencion = "descarga_directa"
        
        datos_salida = {
            "metadata": {
                "fecha": datetime.now().isoformat(),
                "total_resultados": conteo_real,  # Usar el conteo REAL, no estimaciones
                "tiempo_ejecucion": time.time() - estadisticas.get("tiempo_inicio", 0),
                "paginas_procesadas": estadisticas.get("paginas_procesadas", 0),
                "metodo_obtencion": metodo_obtencion  # Nuevo campo
            },
            "resultados": resultados,
            "total": conteo_real  # Añadir también en la raíz para compatibilidad
        }

        with open(archivo_salida, 'w', encoding='utf-8') as f:
            json.dump(datos_salida, f, ensure_ascii=False, indent=2)

        logger.info(f"Resultados guardados en formato JSON en '{archivo_salida}' (Total: {conteo_real}, Método: {metodo_obtencion})")
        return True
    except Exception as e:
        logger.error(f"ERROR al guardar resultados en JSON: {str(e)}")
        return False

def ejecutar_flujo_completo(termino_general, termino_adicional1="", termino_adicional2="", termino_adicional3="", 
                         max_resultados=50, directorio_salida="textos_extraidos",
                         headless=True, compilar_textos=True, abrir_archivo=True,
                         guardar_pdfs=False, compilar_pdfs=False,
                         incluir_8a_epoca=False, incluir_7a_epoca=False,
                         incluir_6a_epoca=False, incluir_5a_epoca=False,
                         tipo_tesis=0, usar_descarga_directa=True):  # Nuevo parámetro
    """
    Ejecuta el flujo completo: búsqueda de tesis + extracción de texto.
    Modificada para generar solo archivos compilados.
    
    Args:
        termino_general (str): Término principal para la búsqueda
        termino_adicional1 (str): Primer término adicional opcional
        termino_adicional2 (str): Segundo término adicional opcional
        termino_adicional3 (str): Tercer término adicional opcional
        max_resultados (int): Número máximo de resultados a obtener
        directorio_salida (str): Directorio donde guardar los textos extraídos
        headless (bool): Si el navegador se ejecuta en modo headless
        compilar_textos (bool): Si se deben compilar todos los textos en un solo archivo
        abrir_archivo (bool): Si se debe abrir el archivo compilado al finalizar
        guardar_pdfs (bool): Si se deben guardar los PDFs originales
        compilar_pdfs (bool): Si se deben compilar todos los PDFs en un solo archivo
        incluir_8a_epoca (bool): Si se incluye la 8ª Época en la búsqueda
        incluir_7a_epoca (bool): Si se incluye la 7ª Época en la búsqueda
        incluir_6a_epoca (bool): Si se incluye la 6ª Época en la búsqueda
        incluir_5a_epoca (bool): Si se incluye la 5ª Época en la búsqueda
        tipo_tesis (int): Tipo de tesis a buscar (0: Todas, 1: Jurisprudencia, 2: Aislada)
        usar_descarga_directa (bool): Si se debe usar el método de descarga directa optimizado
        
    Returns:
        dict: Estadísticas del proceso
    """
    inicio = time.time()
    timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
    
    estadisticas = {
        "timestamp": timestamp,
        "termino_general": termino_general,
        "termino_adicional1": termino_adicional1,
        "termino_adicional2": termino_adicional2,
        "termino_adicional3": termino_adicional3,
        "max_resultados": max_resultados,
        "incluir_8a_epoca": incluir_8a_epoca,
        "incluir_7a_epoca": incluir_7a_epoca,
        "incluir_6a_epoca": incluir_6a_epoca,
        "incluir_5a_epoca": incluir_5a_epoca,
        "tipo_tesis": tipo_tesis,
        "fase_scraper": {
            "completado": False,
            "tiempo_ejecucion": 0,
            "resultados_obtenidos": 0,
            "ruta_archivo_urls": ""
        },
        "fase_extraccion": {
            "completado": False,
            "tiempo_ejecucion": 0,
            "exitos": 0,
            "fallos": 0,
            "total": 0,
            "directorio_salida": directorio_salida,
            "archivo_compilado_texto": None,
            "archivo_compilado_pdf": None
        },
        "tiempo_total": 0
    }
    
    # Verificar que el scraper esté disponible
    if not SCRAPER_IMPORTADO:
        print("ERROR: No se puede continuar sin el scraper de tesis.")
        return estadisticas
    
    # Crear directorio de salida si no existe
    if not os.path.exists(directorio_salida):
        os.makedirs(directorio_salida)
    
    # Construir la consulta completa combinando los términos no vacíos
    terminos = [termino_general]
    if termino_adicional1.strip():
        terminos.append(termino_adicional1.strip())
    if termino_adicional2.strip():
        terminos.append(termino_adicional2.strip())
    if termino_adicional3.strip():
        terminos.append(termino_adicional3.strip())
    
    # Combinar todos los términos en una sola cadena de búsqueda
    termino_busqueda_completo = " ".join(terminos)
    
    # Generar nombre base para los archivos de salida
    # Sanitizar los términos para evitar caracteres inválidos en nombres de archivo
    nombre_base_sanitizado = sanitizar_nombre_archivo(termino_busqueda_completo)
    nombre_base = f"tesis_{nombre_base_sanitizado}"
    
    # Especificar la ruta completa de los archivos dentro del directorio
    ruta_base = os.path.join(directorio_salida, nombre_base)
    
    print("\n" + "=" * 70)
    print(f"INICIANDO FLUJO COMPLETO DE BÚSQUEDA Y EXTRACCIÓN")
    print("=" * 70)
    print(f"Términos de búsqueda: '{termino_busqueda_completo}'")
    print(f"Resultados máximos: {max_resultados}")
    print(f"Directorio de salida: {directorio_salida}")
    
    # Mostrar tipo de tesis seleccionado
    tipos_tesis_nombres = ["Tesis jurisprudenciales y aisladas", "Jurisprudencia", "Aislada"]
    print(f"Tipo de tesis: {tipos_tesis_nombres[tipo_tesis]}")
    
    # Modificación: Mejorar visualización de épocas seleccionadas
    epocas_adicionales = []
    if incluir_8a_epoca:
        epocas_adicionales.append("8ª")
    if incluir_7a_epoca:
        epocas_adicionales.append("7ª")
    if incluir_6a_epoca:
        epocas_adicionales.append("6ª")
    if incluir_5a_epoca:
        epocas_adicionales.append("5ª")
        
    if epocas_adicionales:
        print(f"Épocas adicionales: {', '.join(epocas_adicionales)}")
    
    if compilar_pdfs:
        print(f"Compilar PDFs originales: Sí")
        
    # NUEVO: Mostrar modo de descarga
    print(f"Modo de descarga: {'Optimizado' if usar_descarga_directa else 'Tradicional'}")
    print("=" * 70 + "\n")
    
    try:
        # FASE 1: Ejecutar el scraper
        print("\n--- FASE 1: BÚSQUEDA DE TESIS ---\n")
        tiempo_scraper_inicio = time.time()
        
        # MODIFICACIÓN: Inicializar el scraper directamente con la bandera
        scraper = ScraperTesisCompuesto(
            headless=headless,
            modo_debug=False,
            timeout_default=20,
            tipo_tesis=tipo_tesis
        )
        scraper.usar_descarga_directa = usar_descarga_directa  # Nueva propiedad
        
        # Llamar al scraper con la combinación de términos
        resultado_scraper = ejecutar_scraper_compuesto(
            termino_general=termino_busqueda_completo,
            termino_refinamiento=None,  # No usamos refinamiento
            headless=headless,
            archivo_salida=ruta_base,
            numero_resultados=max_resultados,
            guardar_json=True,
            modo_debug=False,
            incluir_8a_epoca=incluir_8a_epoca,
            incluir_7a_epoca=incluir_7a_epoca,
            incluir_6a_epoca=incluir_6a_epoca,  # Nuevo parámetro
            incluir_5a_epoca=incluir_5a_epoca,  # Nuevo parámetro
            tipo_tesis=tipo_tesis
        )
        
        tiempo_scraper = time.time() - tiempo_scraper_inicio
        estadisticas["fase_scraper"]["tiempo_ejecucion"] = tiempo_scraper
        
        if not resultado_scraper:
            print("\n❌ Error en la fase de búsqueda. No se puede continuar.")
            return estadisticas
        
        estadisticas["fase_scraper"]["completado"] = True
        
        # Archivo de URLs generado por el scraper
        archivo_urls = f"{ruta_base}_urls.txt"
        estadisticas["fase_scraper"]["ruta_archivo_urls"] = archivo_urls
        
        # Verificar resultados con el archivo JSON
        archivo_json = f"{ruta_base}.json"

        # Añadir verificación de coherencia después de obtener los resultados
        if resultado_scraper and os.path.exists(archivo_json):
            conteo_real, conteo_reportado, fue_corregido = verificar_coherencia_resultados(archivo_json)
            if fue_corregido:
                logger.info(f"Se corrigió el conteo de resultados: {conteo_reportado} → {conteo_real}")
            
            resultados_obtenidos = conteo_real
            estadisticas["fase_scraper"]["resultados_obtenidos"] = resultados_obtenidos
            print(f"\n✅ Búsqueda completada: {resultados_obtenidos} resultados")
            
            # NUEVO: Añadir referencia al archivo JSON
            estadisticas["archivo_json"] = archivo_json
        elif os.path.exists(archivo_json):
            try:
                with open(archivo_json, 'r', encoding='utf-8') as f:
                    datos_json = json.load(f)
                
                # Verificar conteo real vs. reportado
                resultados_obtenidos = len(datos_json.get('resultados', []))
                total_reportado = datos_json.get('metadata', {}).get('total_resultados', 0)
                
                # Si hay discrepancia, actualizar el JSON con el conteo correcto
                if total_reportado != resultados_obtenidos and resultados_obtenidos > 0:
                    logger.info(f"Corrigiendo conteo de resultados: reportado={total_reportado}, real={resultados_obtenidos}")
                    
                    # Actualizar metadata
                    if 'metadata' in datos_json:
                        datos_json['metadata']['total_resultados'] = resultados_obtenidos
                    
                    # Añadir/actualizar total en la raíz
                    datos_json['total'] = resultados_obtenidos
                    
                    # Reescribir archivo JSON con datos corregidos
                    with open(archivo_json, 'w', encoding='utf-8') as f:
                        json.dump(datos_json, f, ensure_ascii=False, indent=2)
                    
                    logger.info(f"Archivo JSON actualizado con conteo corregido: {resultados_obtenidos}")
                
                estadisticas["fase_scraper"]["resultados_obtenidos"] = resultados_obtenidos
                print(f"\n✅ Búsqueda completada: {resultados_obtenidos} resultados")
                
                # NUEVO: Añadir referencia al archivo JSON
                estadisticas["archivo_json"] = archivo_json
            except Exception as e:
                logger.warning(f"No se pudo procesar completamente el archivo JSON: {str(e)}")
                print("\n⚠️ No se pudo leer el archivo JSON de resultados")
        
        # FASE 2: Extracción de texto de PDFs
        print("\n--- FASE 2: EXTRACCIÓN DE TEXTO DE PDFs ---\n")
        if compilar_pdfs:
            print("(Incluyendo compilación de PDFs originales)")
        
        tiempo_extraccion_inicio = time.time()
        
        # Comprobar que el archivo de URLs existe
        if not os.path.exists(archivo_urls):
            print(f"\n❌ No se encontró el archivo de URLs: {archivo_urls}")
            return estadisticas
        
        # Archivo de texto compilado (siempre se crea)
        archivo_compilado_texto = f"{ruta_base}_COMPILADO.txt"
        print(f"Se compilarán todos los textos en: {archivo_compilado_texto}")
        
        # Procesar cada URL y extraer texto directamente al archivo compilado
        exitos, fallos, total, ruta_compilado_texto, ruta_compilado_pdf = procesar_lista_urls(
            archivo_urls=archivo_urls,
            directorio_salida=directorio_salida,
            prefijo=f"{nombre_base}_",
            archivo_compilado=archivo_compilado_texto,
            guardar_pdfs=compilar_pdfs,  # Solo guardamos PDFs si vamos a compilarlos
            compilar_pdfs=compilar_pdfs
        )
        
        tiempo_extraccion = time.time() - tiempo_extraccion_inicio
        estadisticas["fase_extraccion"] = {
            "completado": True,
            "tiempo_ejecucion": tiempo_extraccion,
            "exitos": exitos,
            "fallos": fallos,
            "total": total,
            "directorio_salida": directorio_salida,
            "archivo_compilado_texto": ruta_compilado_texto,
            "archivo_compilado_pdf": ruta_compilado_pdf,
            "archivo_json": archivo_json if os.path.exists(archivo_json) else None  # NUEVO: Referencia al JSON en fase_extraccion
        }
        
        # Completado
        tiempo_total = time.time() - inicio
        estadisticas["tiempo_total"] = tiempo_total
        
        # Mostrar resumen
        print("\n" + "=" * 70)
        print("PROCESO COMPLETADO")
        print("=" * 70)
        print(f"Tiempo total: {tiempo_total:.2f} segundos")
        print(f"Resultados obtenidos: {resultados_obtenidos}")
        print(f"Textos extraídos: {exitos} exitosos, {fallos} fallidos, {total} total")
        print(f"Archivos guardados en: {directorio_salida}")
        
        # Información sobre archivo compilado de texto
        if ruta_compilado_texto and os.path.exists(ruta_compilado_texto):
            print(f"Archivo de texto compilado: {ruta_compilado_texto}")
            
            # Abrir el archivo compilado si está activada la opción
            if abrir_archivo:
                try:
                    print("\nAbriendo archivo compilado de texto...")
                    # Determinar el comando según el sistema operativo
                    if sys.platform.startswith('win'):
                        # Windows
                        subprocess.Popen(["notepad.exe", ruta_compilado_texto])
                    elif sys.platform.startswith('darwin'):
                        # macOS
                        subprocess.Popen(["open", ruta_compilado_texto])
                    else:
                        # Linux y otros
                        subprocess.Popen(["xdg-open", ruta_compilado_texto])
                except Exception as e:
                    print(f"No se pudo abrir el archivo compilado de texto: {str(e)}")
                    # Mostrar tamaño del archivo generado para referencia del usuario
            try:
                tamanio_kb = os.path.getsize(ruta_compilado_texto) / 1024
                tamanio_mb = tamanio_kb / 1024
                if tamanio_mb >= 1:
                    print(f"Tamaño del archivo texto: {tamanio_mb:.2f} MB")
                else:
                    print(f"Tamaño del archivo texto: {tamanio_kb:.2f} KB")
            except Exception as e:
                print(f"No se pudo determinar el tamaño del archivo: {str(e)}")
        
        # Información sobre archivo compilado de PDF
        if ruta_compilado_pdf and os.path.exists(ruta_compilado_pdf):
            print(f"Archivo PDF compilado: {ruta_compilado_pdf}")
            
            # Abrir el archivo compilado de PDF si está activada la opción
            if abrir_archivo:
                try:
                    print("\nAbriendo archivo compilado de PDF...")
                    # Determinar el comando según el sistema operativo
                    if sys.platform.startswith('win'):
                        # Windows
                        os.startfile(ruta_compilado_pdf)
                    elif sys.platform.startswith('darwin'):
                        # macOS
                        subprocess.Popen(["open", ruta_compilado_pdf])
                    else:
                        # Linux y otros
                        subprocess.Popen(["xdg-open", ruta_compilado_pdf])
                except Exception as e:
                    print(f"No se pudo abrir el archivo compilado de PDF: {str(e)}")
        
        print("=" * 70 + "\n")
        
        return estadisticas
    
    except Exception as e:
        logger.error(f"Error en flujo completo: {str(e)}")
        traceback.print_exc()
        estadisticas["error"] = str(e)
        return estadisticas

if __name__ == "__main__":
    print("\n" + "=" * 70)
    print("INTEGRADOR DE BÚSQUEDA DE TESIS JUDICIAL Y EXTRACCIÓN DE TEXTO")
    print("=" * 70 + "\n")
    
    # Solicitar parámetros básicos
    termino_general = input("Ingrese término de búsqueda general: ")
    if not termino_general:
        print("ERROR: El término de búsqueda general es obligatorio")
        sys.exit(1)
    
    # Obtener términos adicionales opcionales
    termino_adicional1 = input("Término adicional 1 (opcional): ")
    termino_adicional2 = input("Término adicional 2 (opcional): ")
    termino_adicional3 = input("Término adicional 3 (opcional): ")
    
    # Informar al usuario sobre la consulta formada
    terminos = [termino_general]
    if termino_adicional1.strip():
        terminos.append(termino_adicional1.strip())
    if termino_adicional2.strip():
        terminos.append(termino_adicional2.strip())
    if termino_adicional3.strip():
        terminos.append(termino_adicional3.strip())
    consulta_completa = " ".join(terminos)
    print(f"\nConsulta a ejecutar: '{consulta_completa}'")
    
    # Opcional: número de resultados
    max_resultados_str = input("Número máximo de resultados a obtener [50]: ")
    max_resultados = int(max_resultados_str) if max_resultados_str else 50
    
    # Opcional: directorio personalizado
    directorio_salida = input("Directorio para guardar resultados [textos_extraidos]: ")
    if not directorio_salida:
        directorio_salida = "textos_extraidos"
    
    # Seleccionar épocas adicionales
    incluir_8a_epoca_input = input("¿Incluir 8ª Época en la búsqueda? (s/N): ").lower()
    incluir_8a_epoca = incluir_8a_epoca_input.startswith('s')  # Por defecto no
    
    incluir_7a_epoca_input = input("¿Incluir 7ª Época en la búsqueda? (s/N): ").lower()
    incluir_7a_epoca = incluir_7a_epoca_input.startswith('s')  # Por defecto no
    
    # Nuevas opciones para 6ª y 5ª Épocas
    incluir_6a_epoca_input = input("¿Incluir 6ª Época en la búsqueda? (s/N): ").lower()
    incluir_6a_epoca = incluir_6a_epoca_input.startswith('s')  # Por defecto no
    
    incluir_5a_epoca_input = input("¿Incluir 5ª Época en la búsqueda? (s/N): ").lower()
    incluir_5a_epoca = incluir_5a_epoca_input.startswith('s')  # Por defecto no
    
    # Seleccionar tipo de tesis
    print("\nSeleccione el tipo de tesis:")
    print("1. Tesis jurisprudenciales y aisladas (default)")
    print("2. Jurisprudencia")
    print("3. Aislada")
    tipo_tesis_input = input("Opción [1]: ")
    
    # Convertir la entrada a un valor de 0 a 2
    tipo_tesis = 0  # Valor predeterminado
    if tipo_tesis_input:
        try:
            seleccion = int(tipo_tesis_input)
            if 1 <= seleccion <= 3:  # Validar rango
                tipo_tesis = seleccion - 1  # Restar 1 para que sea 0-2
        except ValueError:
            pass  # Si hay error, usar el valor predeterminado
    
    # Compilar textos (siempre activado)
    compilar_textos = True
    
    # Compilar PDFs (opcional)
    compilar_pdfs = input("¿Compilar todos los textos en un PDF? (s/N): ").lower()
    compilar_pdfs = compilar_pdfs.startswith('s')  # Por defecto no
    
    # Abrir automáticamente (por defecto sí)
    abrir = input("¿Abrir los archivos compilados al finalizar? (S/n): ").lower()
    abrir_archivo = not abrir.startswith('n')  # Por defecto sí
    
    # NUEVO: Modo de descarga optimizado (por defecto sí)
    usar_descarga_directa_input = input("¿Usar modo de descarga optimizado? (S/n): ").lower()
    usar_descarga_directa = not usar_descarga_directa_input.startswith('n')  # Por defecto sí
    
    # Ejecutar el flujo completo
    try:
        ejecutar_flujo_completo(
            termino_general=termino_general,
            termino_adicional1=termino_adicional1,
            termino_adicional2=termino_adicional2,
            termino_adicional3=termino_adicional3,
            max_resultados=max_resultados,
            directorio_salida=directorio_salida,
            compilar_textos=compilar_textos,
            abrir_archivo=abrir_archivo,
            guardar_pdfs=False,  # No guardamos PDFs individuales
            compilar_pdfs=compilar_pdfs,
            incluir_8a_epoca=incluir_8a_epoca,
            incluir_7a_epoca=incluir_7a_epoca,
            incluir_6a_epoca=incluir_6a_epoca,  # Nuevo parámetro
            incluir_5a_epoca=incluir_5a_epoca,  # Nuevo parámetro
            tipo_tesis=tipo_tesis,
            usar_descarga_directa=usar_descarga_directa  # Nuevo parámetro
        )
    except KeyboardInterrupt:
        print("\nOperación cancelada por el usuario.")
        sys.exit(0)
    except Exception as e:
        print(f"\nERROR FATAL: {str(e)}")
        traceback.print_exc()
        sys.exit(1)