# -*- coding: utf-8 -*-
"""
Utilidades compartidas para las interfaces gráficas del integrador de jurisprudencia.
Centraliza funciones comunes para evitar redundancia de código entre las distintas
clases de GUI.
"""

import os
import sys
import threading
import time
import tkinter as tk
from tkinter import messagebox
import logging
import importlib.util

def configurar_logging(nombre_log="integrador_gui.log", nivel=logging.INFO):
    """
    Configura el sistema de logging si no está ya configurado.
    
    Args:
        nombre_log: Nombre del archivo de log
        nivel: Nivel de logging (por defecto INFO)
    
    Returns:
        Logger configurado
    """
    if not logging.getLogger().hasHandlers():
        logging.basicConfig(
            level=nivel,
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            handlers=[
                logging.FileHandler(nombre_log),
                logging.StreamHandler()
            ]
        )
    return logging.getLogger(nombre_log.split('.')[0])

def obtener_base_path():
    """
    Determina la ruta base dependiendo de si es ejecutable o script.
    
    Returns:
        Ruta base del programa
    """
    if getattr(sys, 'frozen', False):
        base_path = sys._MEIPASS
        logging.info(f"Ejecutando como aplicación congelada. Base path: {base_path}")
    else:
        base_path = os.path.dirname(os.path.abspath(__file__))
        logging.info(f"Ejecutando como script. Base path: {base_path}")
    
    return base_path

def verificar_importaciones(modulos_requeridos, logger=None, mostrar_error=True):
    """
    Verifica si los módulos especificados se pueden importar.
    
    Args:
        modulos_requeridos: Lista de diccionarios con información de los módulos
            [{'nombre': 'modulo', 'desde': 'paquete', 'como': 'alias', 'critico': True}]
        logger: Logger para registrar resultados
        mostrar_error: Si se debe mostrar mensaje de error en caso de fallo
    
    Returns:
        Diccionario con estado de cada módulo, errores y variables globales importadas
    """
    resultados = {}
    errores = []
    globales_importados = {}
    
    if logger is None:
        logger = logging.getLogger()
    
    for modulo in modulos_requeridos:
        nombre = modulo['nombre']
        desde = modulo.get('desde')
        como = modulo.get('como', nombre)
        critico = modulo.get('critico', False)
        
        try:
            if desde:
                # Importación del tipo: from X import Y
                modulo_obj = importlib.import_module(desde)
                obj = getattr(modulo_obj, nombre)
                globales_importados[como] = obj
                resultados[como] = True
                logger.info(f"Importación de '{nombre}' desde '{desde}' exitosa.")
            else:
                # Importación directa: import X
                modulo_obj = importlib.import_module(nombre)
                globales_importados[como] = modulo_obj
                resultados[como] = True
                logger.info(f"Importación directa de '{nombre}' exitosa.")
        except ImportError as e:
            resultados[como] = False
            error_msg = f"Fallo importación de '{nombre}': {str(e)}"
            errores.append(error_msg)
            logger.warning(error_msg)
        except Exception as e:
            resultados[como] = False
            error_msg = f"Error inesperado importando '{nombre}': {str(e)}"
            errores.append(error_msg)
            logger.error(error_msg, exc_info=True)
    
    # Mostrar mensaje de error si es necesario
    if mostrar_error and errores and any(modulo.get('critico', False) for modulo in modulos_requeridos):
        try:
            root_temp = tk.Tk()
            root_temp.withdraw()
            mensaje = "Ocurrieron errores al cargar componentes críticos:\n\n"
            mensaje += "\n".join(f"- {err}" for err in errores)
            mensaje += "\n\nLa aplicación podría no funcionar correctamente."
            messagebox.showerror("Error de Carga de Módulos", mensaje, parent=None)
            root_temp.destroy()
        except Exception as tk_error:
            logger.error(f"Error al mostrar mensaje de error: {str(tk_error)}")
    
    return {
        'resultados': resultados,
        'errores': errores,
        'todos_ok': all(resultados.values()),
        'globales': globales_importados
    }

def centrar_ventana(ventana, ancho=None, alto=None, logger=None):
    """
    Centra una ventana en la pantalla.
    
    Args:
        ventana: Ventana Tkinter a centrar
        ancho: Ancho deseado (opcional)
        alto: Alto deseado (opcional)
        logger: Logger para registrar resultados
    
    Returns:
        Tupla (x, y) con las coordenadas de la ventana
    """
    if logger is None:
        logger = logging.getLogger()
        
    ventana.update_idletasks()
    
    # Si no se especifican dimensiones, usar las actuales
    width = ancho if ancho is not None else ventana.winfo_width()
    height = alto if alto is not None else ventana.winfo_height()
    
    # Calcular posición para centrar
    screen_width = ventana.winfo_screenwidth()
    screen_height = ventana.winfo_screenheight()
    x = max(0, (screen_width // 2) - (width // 2))
    y = max(0, (screen_height // 2) - (height // 2))
    
    # Aplicar geometría
    ventana.geometry(f'{width}x{height}+{x}+{y}')
    
    logger.debug(f"Ventana centrada en {x},{y} con tamaño {width}x{height}")
    
    return x, y

def abrir_archivo_os(ruta_archivo, log_func=None):
    """
    Abre un archivo usando el visor predeterminado del sistema operativo.
    
    Args:
        ruta_archivo: Ruta completa al archivo
        log_func: Función opcional para registrar mensajes
    
    Returns:
        True si el archivo se abrió correctamente, False en caso contrario
    """
    import subprocess
    
    logger = logging.getLogger()
    logger.info(f"Intentando abrir archivo: {ruta_archivo}")
    
    try:
        if sys.platform.startswith('win'):
            os.startfile(ruta_archivo)
        elif sys.platform.startswith('darwin'):
            subprocess.run(['open', ruta_archivo], check=True)
        else:
            subprocess.run(['xdg-open', ruta_archivo], check=True)
        
        # Registrar en el log si hay una función disponible
        if log_func:
            log_func(f"Archivo abierto: {os.path.basename(ruta_archivo)}", "info")
        
        return True
    except FileNotFoundError:
        mensaje = f"Error: Archivo no encontrado para abrir: {ruta_archivo}"
        logger.error(mensaje)
        if log_func:
            log_func(mensaje, "error")
        messagebox.showerror("Error", f"No se encontró el archivo:\n{ruta_archivo}")
        return False
    except Exception as e:
        mensaje = f"Error al abrir archivo '{ruta_archivo}': {str(e)}"
        logger.error(mensaje, exc_info=True)
        if log_func:
            log_func(mensaje, "error")
        messagebox.showerror("Error", f"No se pudo abrir el archivo:\n{os.path.basename(ruta_archivo)}\n\nError: {str(e)}")
        return False 