# -*- coding: utf-8 -*-
"""
Interfaz gráfica rediseñada para el Integrador de Tesis SCJN v1.2
Proporciona acceso fácil a las funcionalidades principales mediante una GUI moderna
con un aspecto técnico, organizado, tipografía unificada y tema con contraste
intercalado (claro/oscuro) optimizado para mejor visibilidad.
"""

import os
import sys
import threading
import time
import tkinter as tk
from tkinter import ttk, filedialog, messagebox, font
from tkinter.scrolledtext import ScrolledText
import json
import re
import traceback
import logging
from datetime import datetime
import subprocess
import importlib.util # Añadido para carga dinámica robusta

# Importar utilidades compartidas para interfaces gráficas
try:
    from gui_utils import configurar_logging, obtener_base_path, verificar_importaciones, centrar_ventana, abrir_archivo_os
    utils_importados = True
except ImportError:
    utils_importados = False
    print("No se pudo importar gui_utils. Se usará configuración local.")

# Configuración de logging usando función central si está disponible
if utils_importados:
    # Usar la función centralizada
    gui_logger = configurar_logging("integrador_gui_rediseñado.log")
else:
    # Configuración fallback
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[
            logging.FileHandler("integrador_gui_rediseñado.log"), # Log específico
            logging.StreamHandler()
        ]
    )
    gui_logger = logging.getLogger("IntegradorGUIRediseñado")

# -------------------------------------
# Manejo de Rutas y Dependencias
# -------------------------------------

# Determinar la ruta base usando la función centralizada si está disponible
if utils_importados:
    base_path = obtener_base_path()
else:
    # Determinar si estamos ejecutando como script o como ejecutable
    if getattr(sys, 'frozen', False):
        # Si es ejecutable, usar sys._MEIPASS para encontrar recursos
        base_path = sys._MEIPASS
        gui_logger.info(f"Ejecutando como aplicación congelada. Base path: {base_path}")
    else:
        # Si es script, usar el directorio actual
        base_path = os.path.dirname(os.path.abspath(__file__))
        gui_logger.info(f"Ejecutando como script. Base path: {base_path}")

# Agregar la ruta base al sys.path para que pueda encontrar los módulos
sys.path.insert(0, base_path)

# --- Carga Robusta de Módulos Dependientes ---

# Intentar usar la función centralizada para verificar importaciones
if utils_importados:
    modulos_requeridos = [
        {'nombre': 'ejecutar_flujo_completo', 'desde': 'integrador_tesis_scjn', 'como': 'ejecutar_flujo_completo', 'critico': True},
        {'nombre': 'ScraperTesisCompuesto', 'desde': 'scraper_compuesto7', 'como': 'ScraperTesisCompuesto', 'critico': False}
    ]
    importaciones = verificar_importaciones(modulos_requeridos, gui_logger)
    
    # Obtener las referencias de los módulos importados
    INTEGRADOR_IMPORTADO = importaciones['resultados'].get('ejecutar_flujo_completo', False)
    SCRAPER_IMPORTADO = importaciones['resultados'].get('ScraperTesisCompuesto', False)
    error_importacion = importaciones['errores']
    
    # Si se importó ejecutar_flujo_completo, también obtener la referencia
    if INTEGRADOR_IMPORTADO:
        ejecutar_flujo_completo = importaciones['globales']['ejecutar_flujo_completo']
    else:
        # Definir función dummy si la importación falló
        gui_logger.error("ADVERTENCIA CRÍTICA: No se pudo importar 'ejecutar_flujo_completo'.")
        def ejecutar_flujo_completo(*args, **kwargs):
            gui_logger.error("Error: La función 'ejecutar_flujo_completo' no está disponible.")
            messagebox.showerror("Error Crítico",
                                "No se pudo cargar la funcionalidad principal.\n"
                                "Asegúrese de que 'integrador_tesis_scjn.py' está presente.",
                                parent=tk._default_root) # Usar root si está disponible
            raise ImportError("Dependencia 'integrador_tesis_scjn' no encontrada.")
else:
    INTEGRADOR_IMPORTADO = False
    SCRAPER_IMPORTADO = False
    error_importacion = [] # Lista para acumular errores

    # Intentar importar 'integrador_tesis_scjn'
    try:
        from integrador_tesis_scjn import ejecutar_flujo_completo
        INTEGRADOR_IMPORTADO = True
        gui_logger.info("Importación directa de 'integrador_tesis_scjn' exitosa.")
    except ImportError as e:
        error_msg = f"Fallo importación directa integrador: {str(e)}"
        gui_logger.warning(error_msg)
        error_importacion.append(error_msg)
    except Exception as e:
        error_msg = f"Error inesperado importando integrador: {str(e)}"
        gui_logger.error(error_msg, exc_info=True)
        error_importacion.append(error_msg)


    # Intentar importar 'scraper_compuesto7'
    try:
        from scraper_compuesto7 import ScraperTesisCompuesto
        SCRAPER_IMPORTADO = True
        gui_logger.info("Importación directa de 'ScraperTesisCompuesto' exitosa.")
    except ImportError as e:
        error_msg = f"Fallo importación directa scraper: {str(e)}"
        gui_logger.warning(error_msg)
        error_importacion.append(error_msg)
    except Exception as e:
        error_msg = f"Error inesperado importando scraper: {str(e)}"
        gui_logger.error(error_msg, exc_info=True)
        error_importacion.append(error_msg)


    # Definir función dummy si la importación falló
    if not INTEGRADOR_IMPORTADO:
        gui_logger.error("ADVERTENCIA CRÍTICA: No se pudo importar 'ejecutar_flujo_completo'.")
        def ejecutar_flujo_completo(*args, **kwargs):
            gui_logger.error("Error: La función 'ejecutar_flujo_completo' no está disponible.")
            messagebox.showerror("Error Crítico",
                                "No se pudo cargar la funcionalidad principal.\n"
                                "Asegúrese de que 'integrador_tesis_scjn.py' está presente.",
                                parent=tk._default_root) # Usar root si está disponible
            raise ImportError("Dependencia 'integrador_tesis_scjn' no encontrada.")

# -------------------------------------
# Clase Principal de la GUI Rediseñada
# -------------------------------------

class IntegradorTesisGUIRediseñado:
    """
    Interfaz gráfica rediseñada con aspecto técnico, contraste mejorado y tema intercalado.
    """
    def __init__(self, root):
        self.root = root
        self.root.title("🤖 Robot Integrador de Jurisprudencia SCJN") # Título unificado
        self.root.geometry("1250x850")
        self.root.minsize(1100, 750)

        # Aplicar la configuración de estilo rediseñada
        self.configurar_estilo_tecnico()

        # Configurar canvas con scrollbar
        self.configurar_canvas_scroll()

        # Variables de control
        self.inicializar_variables()

        # Crear la estructura de la interfaz rediseñada
        self.crear_interfaz_rediseñada()

        # Mostrar errores de importación si existen
        if error_importacion:
            self.mostrar_errores_importacion()

    def configurar_estilo_tecnico(self):
        """Configura estilos TTK para un aspecto técnico intercalado (claro/oscuro)."""
        gui_logger.info("Configurando estilo técnico para la GUI...")

        # --- Paleta de Colores ---
        self.color_fondo_claro = "#ECEFF1"      # Gris muy claro (casi blanco) - Fondo principal claro
        self.color_fondo_oscuro = "#465B65"     # Gris azulado oscuro - Fondo principal oscuro
        self.color_texto_claro = "#FFFFFF"      # Blanco - Texto sobre fondo oscuro
        self.color_texto_oscuro = "#37474F"     # Gris oscuro azulado - Texto sobre fondo claro
        self.color_texto_sec_claro = "#B0BEC5"  # Gris claro - Texto secundario sobre fondo oscuro
        self.color_texto_sec_oscuro = "#546E7A" # Gris azulado medio - Texto secundario sobre fondo claro
        self.color_primario = "#455A64"         # Gris azulado oscuro (elementos principales sobre claro)
        self.color_secundario = "#607D8B"       # Gris azulado (elementos secundarios sobre claro)
        self.color_acento = "#1976D2"           # Azul (para botones de acción, bordes activos)
        self.color_borde_claro = "#B0BEC5"      # Gris claro (bordes inactivos sobre claro)
        self.color_borde_oscuro = "#78909C"     # Gris medio (bordes inactivos sobre oscuro)
        self.color_fondo_widget = "#FFFFFF"     # Blanco para campos de entrada/display (Entry, Treeview, etc.)
        self.color_error = "#E57373"            # Rojo claro (visible sobre oscuro y claro)
        self.color_success = "#81C784"          # Verde claro (visible sobre oscuro y claro)
        self.color_warning = "#FFB74D"          # Naranja claro (visible sobre oscuro y claro)
        self.color_debug = "#64B5F6"            # Azul claro (visible sobre oscuro y claro)
        
        # --- Nuevos colores para la barra de progreso vanguardista ---
        self.color_progreso_gradiente_1 = "#2196F3"  # Azul claro para gradiente
        self.color_progreso_gradiente_2 = "#0D47A1"  # Azul oscuro para gradiente
        self.color_progreso_fondo = "#F0F4F7"        # Gris muy claro para el fondo de la barra
        self.color_progreso_borde = "#FFFFFF"        # Borde blanco para efecto elevación

        # --- Fuentes ---
        self.main_font_family = "Consolas" # O "Courier New", "Lucida Console"
        # Usar tamaño 12 como base
        self.main_font_size = 12
        self.main_font_bold_size = 12
        self.title_font_size = 19       # Ligeramente mayor
        self.subtitle_font_size = 15    # Ligeramente mayor
        self.section_title_font_size = 13 # Ligeramente mayor

        available_fonts = list(font.families())
        if self.main_font_family not in available_fonts:
            gui_logger.warning(f"Fuente '{self.main_font_family}' no encontrada. Usando fallback 'Courier New'.")
            self.main_font_family = "Courier New" # Fallback común

        self.font_normal = (self.main_font_family, self.main_font_size)
        self.font_bold = (self.main_font_family, self.main_font_bold_size, "bold")
        self.font_titulo = (self.main_font_family, self.title_font_size, "bold")
        self.font_subtitulo = (self.main_font_family, self.subtitle_font_size)
        self.font_seccion = (self.main_font_family, self.section_title_font_size, "bold")

        # --- Configurar Estilos TTK ---
        estilo = ttk.Style()
        self.estilo = estilo  # Guardar referencia al estilo para usar en otras partes
        estilo.theme_use('clam')  # Usar clam como base es mejor para personalización

        # Configuración General (para fondo claro por defecto)
        estilo.configure(".",
                         background=self.color_fondo_claro,
                         foreground=self.color_texto_oscuro,
                         font=self.font_normal,
                         borderwidth=1,
                         focusthickness=1,
                         focuscolor=self.color_acento)

        # --- Estilos para Fondo Claro ---
        estilo.configure("Light.TFrame", background=self.color_fondo_claro)
        estilo.configure("Card.TFrame", background=self.color_fondo_claro, relief="solid", bordercolor=self.color_borde_claro, borderwidth=1) # Para LabelFrames claros
        estilo.configure("TLabel", background=self.color_fondo_claro, foreground=self.color_texto_oscuro, font=self.font_normal)
        estilo.configure("Info.TLabel", foreground=self.color_texto_sec_oscuro, background=self.color_fondo_claro) # Texto secundario sobre claro
        estilo.configure("Campo.TLabel", font=self.font_bold, foreground=self.color_primario, background=self.color_fondo_claro) # Label de campo sobre claro
        estilo.configure("TCheckbutton", background=self.color_fondo_claro, foreground=self.color_texto_oscuro, font=self.font_normal)
        estilo.map("TCheckbutton", 
                   indicatorcolor=[('selected', self.color_acento)], 
                   foreground=[('active', self.color_primario)],
                   background=[('active', self.color_fondo_claro)],
                   indicatorsize=[('', 20)])  # Indicador más grande (2x)
        estilo.configure("TRadiobutton", 
                         background=self.color_fondo_claro, 
                         foreground=self.color_texto_oscuro, 
                         font=(self.main_font_family, self.main_font_size + 1, "bold"))  # Fuente más grande y en negrita
        estilo.map("TRadiobutton", 
                   indicatorcolor=[('selected', self.color_acento)], 
                   foreground=[('active', self.color_primario)],
                   background=[('active', self.color_fondo_claro)],
                   indicatorsize=[('', 20)])  # Indicador más grande (2x)

        # --- Estilos para Fondo Oscuro ---
        estilo.configure("Dark.TFrame", background=self.color_fondo_oscuro)
        estilo.configure("Dark.Card.TFrame", background=self.color_fondo_oscuro, relief="solid", bordercolor=self.color_borde_oscuro, borderwidth=1) # Para LabelFrames oscuros
        estilo.configure("Dark.TLabel", background=self.color_fondo_oscuro, foreground=self.color_texto_claro, font=self.font_normal)
        estilo.configure("Dark.Info.TLabel", foreground=self.color_texto_sec_claro, background=self.color_fondo_oscuro) # Texto secundario sobre oscuro
        estilo.configure("Dark.Campo.TLabel", font=self.font_bold, foreground=self.color_texto_claro, background=self.color_fondo_oscuro) # Label de campo sobre oscuro
        
        # FASE 1: Crear nuevo estilo para texto en negrita y blanco sobre fondo oscuro
        estilo.configure("Dark.Bold.TLabel", 
                        background=self.color_fondo_oscuro,   # Fondo oscuro 
                        foreground=self.color_texto_claro,    # Texto blanco (no grisáceo)
                        font=(self.main_font_family, self.main_font_size, "bold"))  # Fuente en negrita
        
        # Checkbutton oscuro mejorado con configuración explícita
        estilo.configure("Dark.TCheckbutton", 
                         background=self.color_fondo_oscuro,  # Fondo oscuro siempre
                         foreground=self.color_texto_claro,   # Texto blanco
                         font=(self.main_font_family, self.main_font_size + 1, "bold"))  # Fuente más grande y en negrita
        estilo.map("Dark.TCheckbutton",
                   indicatorcolor=[('selected', self.color_acento), ('active', self.color_acento)],
                   foreground=[('active', "#FFFFFF"), ('disabled', self.color_texto_sec_claro)],
                   background=[('active', self.color_fondo_oscuro), ('disabled', self.color_fondo_oscuro)],
                   selectcolor=[('', self.color_fondo_oscuro)],  # Color de fondo del indicador siempre oscuro
                   indicatorsize=[('', 20)])  # Indicador más grande (2x)

        # Radiobutton oscuro (agregado nuevo)
        estilo.configure("Dark.TRadiobutton", 
                         background=self.color_fondo_oscuro,  # Fondo oscuro siempre
                         foreground=self.color_texto_claro,   # Texto blanco
                         font=(self.main_font_family, self.main_font_size + 1, "bold"))  # Fuente más grande y en negrita
        estilo.map("Dark.TRadiobutton",
                   indicatorcolor=[('selected', self.color_acento)],
                   foreground=[('active', self.color_texto_claro), ('disabled', self.color_texto_sec_claro)],
                   background=[('active', self.color_fondo_oscuro), ('disabled', self.color_fondo_oscuro)],
                   selectcolor=[('', self.color_fondo_oscuro)],  # Color de fondo del indicador siempre oscuro
                   indicatorsize=[('', 20)])  # Indicador más grande (2x)

        # --- Estilos Comunes o Específicos ---
        # Header (oscuro)
        estilo.configure("Header.TFrame", background=self.color_fondo_oscuro)
        estilo.configure("Titulo.TLabel", 
                         font=self.font_titulo, 
                         foreground=self.color_texto_claro, 
                         background=self.color_fondo_oscuro)
        estilo.configure("Subtitulo.TLabel", 
                         font=self.font_subtitulo, 
                         foreground=self.color_texto_claro, 
                         background=self.color_fondo_oscuro)

        # LabelFrame títulos (depende del fondo)
        # Asegurar que el fondo del título coincida con el LabelFrame
        estilo.configure("Card.TLabelframe", 
                         background=self.color_fondo_claro, 
                         bordercolor=self.color_borde_claro, 
                         padding=15)
        estilo.configure("Card.TLabelframe.Label", 
                         background=self.color_fondo_claro, 
                         foreground=self.color_primario, 
                         font=self.font_seccion, 
                         padding=(0, 5))
        
        # Asegurar que los títulos en fondos oscuros tengan el fondo adecuado
        estilo.configure("Dark.Card.TLabelframe", 
                         background=self.color_fondo_oscuro, 
                         bordercolor=self.color_borde_oscuro, 
                         padding=15)
        estilo.configure("Dark.Card.TLabelframe.Label", 
                         background=self.color_fondo_oscuro, 
                         foreground=self.color_texto_claro, 
                         font=self.font_seccion, 
                         padding=(0, 5))

        # Botones (sobre fondo claro por defecto)
        estilo.configure("TButton", font=self.font_bold, padding=(10, 5), relief="raised", background=self.color_secundario, foreground=self.color_fondo_widget, bordercolor=self.color_primario)
        estilo.map("TButton", background=[('active', self.color_primario), ('pressed', self.color_primario)], foreground=[('active', self.color_fondo_widget)])
        estilo.configure("Accion.TButton", background=self.color_acento, foreground=self.color_fondo_widget, font=self.font_bold)
        estilo.map("Accion.TButton", background=[('active', '#1565C0'), ('pressed', '#1565C0')])
        estilo.configure("Primario.TButton", background=self.color_fondo_claro, foreground=self.color_acento, bordercolor=self.color_acento, relief="solid")
        estilo.map("Primario.TButton", background=[('active', '#E3F2FD'), ('pressed', '#E3F2FD')], foreground=[('active', self.color_acento)])

        # Entry y Spinbox (fondo blanco, texto oscuro)
        estilo.configure("TEntry", fieldbackground=self.color_fondo_widget, foreground=self.color_texto_oscuro, insertcolor=self.color_texto_oscuro, bordercolor=self.color_borde_claro, lightcolor=self.color_borde_claro, darkcolor=self.color_borde_claro, font=self.font_normal)
        estilo.map("TEntry", bordercolor=[('focus', self.color_acento)], lightcolor=[('focus', self.color_acento)], darkcolor=[('focus', self.color_acento)])
        estilo.configure("TSpinbox", fieldbackground=self.color_fondo_widget, foreground=self.color_texto_oscuro, insertcolor=self.color_texto_oscuro, bordercolor=self.color_borde_claro, arrowcolor=self.color_primario, arrowsize=15, relief="flat", font=self.font_normal)
        estilo.map("TSpinbox", bordercolor=[('focus', self.color_acento)])

        # Treeview (fondo claro)
        estilo.configure("Treeview", background=self.color_fondo_widget, fieldbackground=self.color_fondo_widget, foreground=self.color_texto_oscuro, font=self.font_normal, rowheight=int(self.main_font_size * 2.2))
        estilo.configure("Treeview.Heading", font=self.font_bold, background=self.color_primario, foreground=self.color_fondo_widget, relief="raised", padding=(5, 5))
        estilo.map("Treeview.Heading", background=[('active', self.color_secundario)])
        # Tags para filas alternas
        estilo.configure('Treeview.Oddrow', background='#FFFFFF')
        estilo.configure('Treeview.Evenrow', background='#F7F9FA')

        # ScrolledText (para Log - necesita fondo oscuro)
        # El widget ScrolledText base de Tk no usa estilos ttk directamente para el área de texto.
        # Configuraremos sus colores al crearlo.

        # Scrollbar (se adapta mejor al fondo claro por defecto)
        estilo.configure("Vertical.TScrollbar", troughcolor=self.color_fondo_claro, bordercolor=self.color_borde_claro, background=self.color_secundario, arrowcolor=self.color_primario)
        estilo.configure("Horizontal.TScrollbar", troughcolor=self.color_fondo_claro, bordercolor=self.color_borde_claro, background=self.color_secundario, arrowcolor=self.color_primario)
        
        # Scrollbar para fondos oscuros
        estilo.configure("Dark.Vertical.TScrollbar", 
                         troughcolor=self.color_fondo_oscuro, 
                         bordercolor=self.color_borde_oscuro, 
                         background=self.color_secundario, 
                         arrowcolor=self.color_texto_claro)
        estilo.configure("Dark.Horizontal.TScrollbar", 
                         troughcolor=self.color_fondo_oscuro, 
                         bordercolor=self.color_borde_oscuro, 
                         background=self.color_secundario, 
                         arrowcolor=self.color_texto_claro)

        # Progressbar (sobre fondo claro)
        estilo.configure("Horizontal.TProgressbar", troughcolor=self.color_borde_claro, background=self.color_acento, thickness=10)
        
        # Estilo vanguardista para la barra de progreso
        estilo.configure("Vanguardista.Horizontal.TProgressbar",
                        troughcolor=self.color_progreso_fondo,
                        background=self.color_progreso_gradiente_1,
                        bordercolor=self.color_progreso_borde,
                        lightcolor=self.color_progreso_gradiente_1,
                        darkcolor=self.color_progreso_gradiente_2,
                        thickness=14,
                        pbarrelief="flat")

        # Estilos para mensajes de log (se usarán como tags)
        # Definidos aquí para referencia, se aplicarán en el método log
        self.log_tags = {
            "error": {"foreground": self.color_error, "font": self.font_bold},
            "success": {"foreground": self.color_success, "font": self.font_bold},
            "warning": {"foreground": self.color_warning, "font": self.font_bold},
            "info": {"foreground": self.color_texto_claro}, # Texto blanco sobre oscuro para mejor contraste
            "debug": {"foreground": self.color_debug} # Azul claro sobre oscuro
        }

        # Crear un estilo Success.TLabel para mensajes de éxito en fondo claro
        estilo.configure("Success.TLabel", 
                         foreground=self.color_success, 
                         font=self.font_bold, 
                         background=self.color_fondo_claro)

        gui_logger.info("Estilos técnicos configurados.")


    def configurar_canvas_scroll(self):
        """Configura el canvas con scroll vertical"""
        self.main_container = ttk.Frame(self.root, style="Light.TFrame") # Asegurar fondo claro base
        self.main_container.pack(fill=tk.BOTH, expand=True)

        self.canvas = tk.Canvas(self.main_container,
                                background=self.color_fondo_claro, # Fondo claro para el canvas
                                highlightthickness=0)
        self.scrollbar = ttk.Scrollbar(self.main_container, orient="vertical", command=self.canvas.yview, style="Vertical.TScrollbar")

        # Frame que contendrá todo el contenido desplazable, fondo claro base
        self.scrollable_frame = ttk.Frame(self.canvas, style="Light.TFrame")

        self.scrollable_frame.bind(
            "<Configure>",
            lambda e: self.canvas.configure(scrollregion=self.canvas.bbox("all"))
        )

        self.canvas_window = self.canvas.create_window((0, 0), window=self.scrollable_frame, anchor="nw")

        def configure_scroll_frame(event):
            canvas_width = event.width
            self.canvas.itemconfig(self.canvas_window, width=canvas_width)

        self.canvas.bind("<Configure>", configure_scroll_frame)
        self.canvas.configure(yscrollcommand=self.scrollbar.set)

        self.scrollbar.pack(side="right", fill="y")
        self.canvas.pack(side="left", fill="both", expand=True)

        def _on_mousewheel(event):
            if event.num == 5 or event.delta < 0: delta = 1
            elif event.num == 4 or event.delta > 0: delta = -1
            else: delta = event.delta // 120
            self.canvas.yview_scroll(delta, "units")

        self.root.bind_all("<MouseWheel>", _on_mousewheel) # Windows/Mac
        self.root.bind_all("<Button-4>", _on_mousewheel)   # Linux scroll up
        self.root.bind_all("<Button-5>", _on_mousewheel)   # Linux scroll down

        gui_logger.debug("Canvas y Scrollbar configurados.")

    def inicializar_variables(self):
        """Inicializa las variables de control Tkinter"""
        self.termino_general = tk.StringVar()
        self.termino_adicional1 = tk.StringVar()
        self.termino_adicional2 = tk.StringVar()
        self.termino_adicional3 = tk.StringVar()
        # Nuevas variables para operadores lógicos de términos adicionales
        self.operador_logico1 = tk.StringVar(value="AND")
        self.operador_logico2 = tk.StringVar(value="AND")
        self.operador_logico3 = tk.StringVar(value="AND")
        # Variable para la vista previa de la consulta
        self.vista_previa_consulta = tk.StringVar(value="")
        
        default_output_dir = os.path.join(os.getcwd(), "SCJN_Resultados")
        self.directorio_salida = tk.StringVar(value=default_output_dir)
        self.max_resultados = tk.IntVar(value=50)
        # Opciones ocultas ahora son fijas
        self.headless = True # tk.BooleanVar(value=True)
        self.usar_descarga_directa = True # tk.BooleanVar(value=True)
        # Opciones visibles
        # Cambiado para compatibilidad con clases derivadas
        self.compilar_textos = tk.BooleanVar(value=True) # Convertido de booleano simple a BooleanVar
        self.compilar_pdfs = tk.BooleanVar(value=True)
        self.abrir_archivo = tk.BooleanVar(value=True) # Cambiado a True (estaba en False)
        self.incluir_8a_epoca = tk.BooleanVar(value=False)
        self.incluir_7a_epoca = tk.BooleanVar(value=False)
        self.incluir_6a_epoca = tk.BooleanVar(value=False)
        self.incluir_5a_epoca = tk.BooleanVar(value=False)
        self.tipo_tesis = tk.IntVar(value=0)
        self.status_var = tk.StringVar(value="Estado: Listo")
        self.busqueda_en_progreso = False
        self.progreso_actual = 0
        self.etapas_progreso = [
            "Inicializando...", "Configurando búsqueda", "Conectando a SCJN",
            "Ejecutando consulta", "Recopilando resultados", "Procesando datos",
            "Generando documentos", "Finalizando..."
        ]
        self.resultados_completos = []
        self.total_resultados_real = 0
        gui_logger.debug("Variables de control inicializadas.")

    def crear_interfaz_rediseñada(self):
        """Crea la interfaz gráfica con el nuevo diseño técnico intercalado."""
        gui_logger.info("Creando interfaz gráfica rediseñada...")

        # Frame principal dentro del scrollable_frame (fondo claro base)
        main_frame = ttk.Frame(self.scrollable_frame, padding=10, style="Light.TFrame")
        main_frame.pack(fill=tk.BOTH, expand=True, padx=0, pady=0) # Sin padding extra aquí

        # --- Encabezado (Fondo Oscuro) ---
        self.crear_encabezado_rediseñado(main_frame)
        # No añadir separador gráfico aquí, la separación es por color

        # --- Sección de Parámetros de Búsqueda (Fondo Claro) ---
        self.crear_seccion_busqueda_rediseñada(main_frame)
        ttk.Separator(main_frame, orient='horizontal').pack(fill=tk.X, pady=10, padx=10)

        # --- Sección de Instrucciones (Fondo Oscuro) ---
        self.crear_seccion_instrucciones_rediseñada(main_frame)
        # No añadir separador gráfico aquí

        # --- Sección de Filtros (Tipo y Épocas) (Fondo Claro) ---
        self.crear_seccion_filtros_rediseñada(main_frame)
        
        # --- Sección de botones de acción con la barra de progreso vanguardista ---
        self.crear_botones_con_barra_progreso(main_frame)  # Esta es la nueva función
        ttk.Separator(main_frame, orient='horizontal').pack(fill=tk.X, pady=10, padx=10)

        # --- Sección de Opciones Adicionales (Fondo Oscuro) ---
        self.crear_seccion_opciones_rediseñada(main_frame)
        # No añadir separador gráfico aquí

        # --- Sección de Resultados Preliminares (Tabla) (Fondo Claro) ---
        self.crear_seccion_resultados_rediseñada(main_frame)
        ttk.Separator(main_frame, orient='horizontal').pack(fill=tk.X, pady=10, padx=10)

        # --- Sección de Log de Actividad (Fondo Oscuro) ---
        self.crear_seccion_log_rediseñada(main_frame)
        # No añadir separador gráfico aquí

        # --- Barra de Estado inferior (Fondo Claro) ---
        status_bar = ttk.Frame(self.root, style="Card.TFrame", relief="sunken", borderwidth=1) # Estilo claro
        status_bar.pack(side=tk.BOTTOM, fill=tk.X)
        ttk.Label(status_bar, textvariable=self.status_var, anchor=tk.W, padding=(5, 2)).pack(fill=tk.X)

        gui_logger.info("Interfaz gráfica rediseñada creada.")

    def crear_botones_con_barra_progreso(self, parent):
        """
        Crea una sección específica para los botones de acción y la barra de progreso vanguardista
        debajo de ellos, como se solicitó específicamente.
        """
        # Frame principal para toda la sección (fondo claro)
        principal_frame = ttk.Frame(parent, style="Light.TFrame")
        principal_frame.pack(fill=tk.X, pady=10, padx=10)
        
        # ==== PARTE 1: BOTONES ====
        # Frame específico para los botones
        botones_frame = ttk.Frame(principal_frame, style="Light.TFrame")
        botones_frame.pack(fill=tk.X, pady=(10, 15))
        
        # Centrar los botones horizontalmente
        botones_center_frame = ttk.Frame(botones_frame, style="Light.TFrame")
        botones_center_frame.pack(anchor=tk.CENTER)
        
        # Crear los botones alineados horizontalmente
        self.btn_extraer = ttk.Button(botones_center_frame, 
                                    text="▶ Ejecutar Búsqueda", 
                                    style="Accion.TButton", 
                                    command=self.iniciar_busqueda, 
                                    width=22)
        self.btn_extraer.pack(side=tk.LEFT, padx=5)
        
        self.btn_limpiar = ttk.Button(botones_center_frame, 
                                    text="Limpiar Campos", 
                                    style="Primario.TButton", 
                                    command=self.limpiar_campos, 
                                    width=15)
        self.btn_limpiar.pack(side=tk.LEFT, padx=5)
        
        self.btn_ayuda = ttk.Button(botones_center_frame, 
                                text="Ayuda", 
                                style="Primario.TButton", 
                                command=self.mostrar_ayuda, 
                                width=10)
        self.btn_ayuda.pack(side=tk.LEFT, padx=5)
        
        self.btn_info = ttk.Button(botones_center_frame, 
                                text="Info Extractor", 
                                style="Primario.TButton", 
                                command=self.mostrar_info_extractor, 
                                width=15)
        self.btn_info.pack(side=tk.LEFT, padx=5)
        
        self.btn_aviso_legal = ttk.Button(botones_center_frame, 
                                text="Aviso Legal", 
                                style="Primario.TButton", 
                                command=self.mostrar_aviso_legal, 
                                width=15)
        self.btn_aviso_legal.pack(side=tk.LEFT, padx=5)
        
        self.btn_salir = ttk.Button(botones_center_frame, 
                                text="Salir", 
                                command=self.root.destroy, 
                                width=10)
        self.btn_salir.pack(side=tk.LEFT, padx=5)
        
        # ==== PARTE 2: BARRA DE PROGRESO VANGUARDISTA ====
        # Frame contenedor para la barra de progreso vanguardista (con borde para hacerla más visible)
        progress_container = ttk.Frame(principal_frame, style="Card.TFrame", padding=(10, 8, 10, 8))
        progress_container.pack(fill=tk.X, pady=(0, 10))
        
        # Panel superior con información del estado actual y porcentaje
        status_panel = ttk.Frame(progress_container, style="Light.TFrame")
        status_panel.pack(fill=tk.X, pady=(0, 5))
        
        # Etiqueta izquierda - Estado de procesamiento
        self.etiqueta_estado = ttk.Label(status_panel,
                                    text="Estado: Listo para iniciar búsqueda",
                                    style="Campo.TLabel",
                                    font=(self.main_font_family, self.main_font_size, "bold"))
        self.etiqueta_estado.pack(side=tk.LEFT)
        
        # Etiqueta derecha - Porcentaje
        self.etiqueta_porcentaje = ttk.Label(status_panel,
                                        text="0%",
                                        style="Campo.TLabel",
                                        font=(self.main_font_family, self.main_font_size, "bold"))
        self.etiqueta_porcentaje.pack(side=tk.RIGHT)
        
        # Configurar estilo vanguardista para la barra de progreso
        self.estilo.configure("Vanguardista.Horizontal.TProgressbar",
                             thickness=16,                     # Barra más gruesa
                             background=self.color_acento,     # Color base
                             troughcolor="#F0F4F7",            # Color de fondo más claro y moderno
                             bordercolor="#FFFFFF",            # Borde blanco para efecto de elevación
                             lightcolor="#4FC3F7",             # Color claro para gradiente
                             darkcolor="#0277BD",              # Color oscuro para gradiente
                             pbarrelief="flat")                # Sin relieve para aspecto moderno
        
        # Marco para la barra de progreso con efecto visual
        barra_frame = ttk.Frame(progress_container, style="Light.TFrame")
        barra_frame.pack(fill=tk.X, pady=(0, 3))
        
        # Barra de progreso con estilo vanguardista
        self.barra_progreso = ttk.Progressbar(barra_frame,
                                            style="Vanguardista.Horizontal.TProgressbar",
                                            orient="horizontal",
                                            length=100,
                                            mode="determinate")
        self.barra_progreso.pack(fill=tk.X, ipady=2)
        
        # Etiqueta inferior para descripción de la fase actual
        self.etiqueta_fase = ttk.Label(progress_container,
                                    text="Listo para iniciar el proceso de extracción de jurisprudencia",
                                    anchor=tk.W,
                                    style="Info.TLabel")
        self.etiqueta_fase.pack(fill=tk.X, pady=(3, 0))
        
        # Configuramos los botones adicionales que no están en la imagen pero existían en el código original
        self.btn_detener = ttk.Button(principal_frame, text="■ Detener", style="TButton", 
                                    command=self.detener_busqueda, width=12)
        self.btn_abrir = ttk.Button(principal_frame, text="Abrir Compilado", style="Primario.TButton", 
                                command=self.abrir_compilado, width=15)
        
        # Los ocultamos pero mantenemos las referencias para que el código funcione
        self.btn_detener.pack_forget()
        self.btn_abrir.pack_forget()
        
        # Hacemos referencia a la barra de progreso en la variable 'progress' usada por otros métodos
        self.progress = self.barra_progreso
        # Referencias para los métodos existentes que actualizan el progreso
        self.progress_status = self.etiqueta_estado
        self.progress_percent = self.etiqueta_porcentaje
        self.progress_label = self.etiqueta_fase

    def crear_encabezado_rediseñado(self, parent):
        """Crea la sección de encabezado con imagen (Fondo Oscuro)."""
        # Usar el estilo Header.TFrame definido con fondo oscuro
        header_frame = ttk.Frame(parent, style="Header.TFrame")
        # Pack con fill=X y padding interno vertical - aumentar el ipady para dar más espacio a la imagen
        header_frame.pack(fill=tk.X, pady=0, ipady=10)

        # Centrar el contenido en una columna
        header_frame.columnconfigure(0, weight=1)
        
        # Hacer accesible la función de redimensionamiento
        redimensionar_imagen = self.redimensionar_imagen
        
        # Función para cargar imagen con soporte para múltiples métodos
        def cargar_imagen(ruta, max_width=None, max_height=None):
            if not os.path.exists(ruta):
                return None, f"Archivo no existe: {ruta}"
                
            # Intentar con PhotoImage nativo
            try:
                return tk.PhotoImage(file=ruta), None
            except Exception as e:
                # Intentar importar PIL para alternativa
                try:
                    from PIL import Image, ImageTk
                    try:
                        img_pil = Image.open(ruta)
                        
                        # Redimensionar la imagen si se especificaron dimensiones máximas
                        if max_width is not None or max_height is not None:
                            # Verificar si el método redimensionar_imagen existe en la clase
                            if hasattr(self, 'redimensionar_imagen'):
                                img_pil = self.redimensionar_imagen(img_pil, max_width, max_height)
                            else:
                                # Implementación básica de redimensionamiento si no existe el método
                                orig_width, orig_height = img_pil.size
                                if max_width and orig_width > max_width:
                                    ratio = max_width / orig_width
                                    new_height = int(orig_height * ratio)
                                    img_pil = img_pil.resize((max_width, new_height), Image.LANCZOS if hasattr(Image, 'LANCZOS') else Image.ANTIALIAS)
                                if max_height and img_pil.size[1] > max_height:
                                    ratio = max_height / img_pil.size[1]
                                    new_width = int(img_pil.size[0] * ratio)
                                    img_pil = img_pil.resize((new_width, max_height), Image.LANCZOS if hasattr(Image, 'LANCZOS') else Image.ANTIALIAS)
                        
                        return ImageTk.PhotoImage(img_pil), None
                    except Exception as e2:
                        return None, f"Error PIL: {str(e2)}"
                except ImportError:
                    return None, f"Error PhotoImage: {str(e)}"
        
        # Rutas a intentar para el título
        rutas_titulo = [
            # Ruta original
            r"C:\Users\hhugo\Dropbox\SCRIPTS DE PYTHON\ROBOT INTEGRADOR TESIS SCJN V5 CURSOR}\titulo robot.png",
            # Ruta corregida sin el carácter }
            r"C:\Users\hhugo\Dropbox\SCRIPTS DE PYTHON\ROBOT INTEGRADOR TESIS SCJN V5 CURSOR\titulo robot.png",
            # Otras ubicaciones posibles
            os.path.join(os.getcwd(), "titulo robot.png"),
            os.path.join(os.path.dirname(os.path.abspath(__file__)), "titulo robot.png"),
            # Directorios comunes para imágenes
            os.path.join(os.getcwd(), "img", "titulo robot.png"),
            os.path.join(os.getcwd(), "imagenes", "titulo robot.png"),
            os.path.join(os.getcwd(), "assets", "titulo robot.png")
        ]
        
        # Para pruebas adicionales, intentar detectar la ruta base del proyecto
        ruta_base = r"C:\Users\hhugo\Dropbox\SCRIPTS DE PYTHON\ROBOT INTEGRADOR TESIS SCJN V5 CURSOR"
        if os.path.exists(ruta_base.replace("}", "")):
            # Usar ruta corregida
            ruta_base = ruta_base.replace("}", "")
            # Añadir búsqueda en subdirectorios de la ruta base
            for subdir in ['', 'img', 'imagenes', 'assets', 'recursos']:
                posible_ruta = os.path.join(ruta_base, subdir, "titulo robot.png")
                if posible_ruta not in rutas_titulo and os.path.exists(posible_ruta):
                    rutas_titulo.append(posible_ruta)
        
        # Definir tamaño máximo para la imagen del título
        max_width_titulo = 800  # Ajusta este valor según necesites
        max_height_titulo = 150  # Ajusta este valor según necesites

        # Intentar cargar la imagen del título
        titulo_imagen = None
        error_final = ""
        for ruta in rutas_titulo:
            imagen, error = cargar_imagen(ruta, max_width_titulo, max_height_titulo)
            if imagen is not None:
                titulo_imagen = imagen
                gui_logger.info(f"Imagen de título cargada desde: {ruta}")
                break
            else:
                error_final += f"\n- {error}"
        
        # Frame para contener y centrar la imagen (o texto de fallback)
        titulo_container = ttk.Frame(header_frame, style="Header.TFrame")
        titulo_container.grid(row=0, column=0, sticky="nsew", padx=10)
        
        # Si se logró cargar la imagen, mostrarla
        if titulo_imagen is not None:
            # Guardar referencia para evitar garbage collection
            self.titulo_imagen_ref = titulo_imagen
            
            # Label para mostrar la imagen con fondo adecuado
            lbl_titulo_imagen = ttk.Label(
                titulo_container, 
                image=titulo_imagen, 
                background=self.color_fondo_oscuro
            )
            # Centrar la imagen
            lbl_titulo_imagen.pack(fill=None, expand=True, pady=5)
            
            gui_logger.info("Imagen de título principal cargada correctamente")
            
            # Registrar en el log
            self.log("Título principal cargado con imagen.", "info")
        else:
            # Si falla, usar el texto original como fallback
            gui_logger.warning(f"No se pudo cargar la imagen del título.{error_final}")
            
            # Crear etiquetas para el texto fallback
            ttk.Label(
                titulo_container, 
                text="🤖 Robot Integrador de Jurisprudencia SCJN", 
                style="Titulo.TLabel", 
                anchor=tk.CENTER
            ).pack(fill=None, expand=True, pady=(5, 0))
            
            ttk.Label(
                titulo_container, 
                text="Sistema Automatizado para Búsqueda y Procesamiento de Tesis Judiciales", 
                style="Subtitulo.TLabel", 
                anchor=tk.CENTER
            ).pack(fill=None, expand=True, pady=(0, 5))
            
            # Registrar el problema
            self.log("No se pudo cargar la imagen del título. Se muestra texto alternativo.", "warning")

    def crear_seccion_busqueda_rediseñada(self, parent):
        """Crea la sección de parámetros de búsqueda con operadores lógicos (Fondo Claro)."""
        # Usar estilo Card.TFrame (fondo claro)
        form_frame = ttk.LabelFrame(parent, text="1. Parámetros de Consulta", style="Card.TLabelframe")
        form_frame.pack(fill=tk.X, pady=10, padx=10) # Padding exterior

        form_frame.columnconfigure(1, weight=1)

        # --- Término General (sin cambios) ---
        ttk.Label(form_frame, text="Término General*:", style="Campo.TLabel").grid(row=0, column=0, sticky=tk.W, padx=5, pady=8)
        entry_general = ttk.Entry(form_frame, textvariable=self.termino_general, font=self.font_normal, width=60)
        entry_general.grid(row=0, column=1, columnspan=3, sticky=tk.EW, padx=5, pady=8)
        
        # --- Título Términos adicionales con botón de ayuda ---
        terminos_frame = ttk.Frame(form_frame, style="Light.TFrame")
        terminos_frame.grid(row=1, column=0, sticky=tk.W, padx=5, pady=(8, 0))
        
        ttk.Label(terminos_frame, text="Términos Adicionales:", style="Campo.TLabel").pack(side=tk.LEFT, padx=(0, 5))
        
        # Botón de ayuda pequeño
        btn_ayuda_terminos = ttk.Button(terminos_frame, text="?", width=2, 
                                        command=self.mostrar_ayuda_operadores,
                                        style="Primario.TButton")
        btn_ayuda_terminos.pack(side=tk.LEFT)

        # --- Términos adicionales con operadores lógicos ---
        # Primer término adicional con operador
        termino1_frame = ttk.Frame(form_frame, style="Light.TFrame")
        termino1_frame.grid(row=2, column=0, columnspan=3, sticky=tk.W, padx=5, pady=(0, 5))
        
        # Combobox para el operador
        combo_operador1 = ttk.Combobox(termino1_frame, 
                                      textvariable=self.operador_logico1, 
                                      values=["AND", "OR", "NOT"],
                                      state="readonly",
                                      width=5)
        combo_operador1.pack(side=tk.LEFT, padx=(0, 5))
        combo_operador1.bind("<<ComboboxSelected>>", lambda e: self.actualizar_descripcion_operador(1))
        
        # Etiqueta descriptiva del operador (se actualiza dinámicamente)
        self.lbl_desc_op1 = ttk.Label(termino1_frame, 
                                     text="Resultados que contienen AMBOS términos", 
                                     style="Info.TLabel")
        self.lbl_desc_op1.pack(side=tk.LEFT, padx=(0, 5))
        
        # Entrada del término
        entry_adic1 = ttk.Entry(form_frame, textvariable=self.termino_adicional1, font=self.font_normal)
        entry_adic1.grid(row=3, column=0, columnspan=3, sticky=tk.EW, padx=5, pady=(0, 8))
        
        # Segundo término adicional con operador
        termino2_frame = ttk.Frame(form_frame, style="Light.TFrame")
        termino2_frame.grid(row=4, column=0, columnspan=3, sticky=tk.W, padx=5, pady=(0, 5))
        
        combo_operador2 = ttk.Combobox(termino2_frame, 
                                      textvariable=self.operador_logico2, 
                                      values=["AND", "OR", "NOT"],
                                      state="readonly",
                                      width=5)
        combo_operador2.pack(side=tk.LEFT, padx=(0, 5))
        combo_operador2.bind("<<ComboboxSelected>>", lambda e: self.actualizar_descripcion_operador(2))
        
        self.lbl_desc_op2 = ttk.Label(termino2_frame, 
                                     text="Resultados que contienen AMBOS términos", 
                                     style="Info.TLabel")
        self.lbl_desc_op2.pack(side=tk.LEFT, padx=(0, 5))
        
        entry_adic2 = ttk.Entry(form_frame, textvariable=self.termino_adicional2, font=self.font_normal)
        entry_adic2.grid(row=5, column=0, columnspan=3, sticky=tk.EW, padx=5, pady=(0, 8))
        
        # Tercer término adicional con operador
        termino3_frame = ttk.Frame(form_frame, style="Light.TFrame")
        termino3_frame.grid(row=6, column=0, columnspan=3, sticky=tk.W, padx=5, pady=(0, 5))
        
        combo_operador3 = ttk.Combobox(termino3_frame, 
                                      textvariable=self.operador_logico3, 
                                      values=["AND", "OR", "NOT"],
                                      state="readonly",
                                      width=5)
        combo_operador3.pack(side=tk.LEFT, padx=(0, 5))
        combo_operador3.bind("<<ComboboxSelected>>", lambda e: self.actualizar_descripcion_operador(3))
        
        self.lbl_desc_op3 = ttk.Label(termino3_frame, 
                                     text="Resultados que contienen AMBOS términos", 
                                     style="Info.TLabel")
        self.lbl_desc_op3.pack(side=tk.LEFT, padx=(0, 5))
        
        entry_adic3 = ttk.Entry(form_frame, textvariable=self.termino_adicional3, font=self.font_normal)
        entry_adic3.grid(row=7, column=0, columnspan=3, sticky=tk.EW, padx=5, pady=(0, 8))

        # --- Vista previa de la consulta ---
        ttk.Label(form_frame, text="Vista previa de consulta:", style="Campo.TLabel").grid(row=8, column=0, sticky=tk.W, padx=5, pady=(8, 0))
        vista_previa_frame = ttk.Frame(form_frame, style="Card.TFrame", padding=5)
        vista_previa_frame.grid(row=9, column=0, columnspan=3, sticky=tk.EW, padx=5, pady=(0, 8))
        
        lbl_vista_previa = ttk.Label(vista_previa_frame, 
                                    textvariable=self.vista_previa_consulta, 
                                    style="Info.TLabel",
                                    wraplength=800)
        lbl_vista_previa.pack(fill=tk.X, padx=5, pady=5)

        # --- Máx. Resultados y Directorio (sin cambios) ---
        ttk.Label(form_frame, text="Máx. Resultados:", style="Campo.TLabel").grid(row=10, column=0, sticky=tk.W, padx=5, pady=8)
        spin_resultados = ttk.Spinbox(form_frame, from_=1, to=500, increment=10, textvariable=self.max_resultados, width=8, font=self.font_normal)
        spin_resultados.grid(row=10, column=1, sticky=tk.W, padx=5, pady=8)

        ttk.Label(form_frame, text="Directorio Salida:", style="Campo.TLabel").grid(row=11, column=0, sticky=tk.W, padx=5, pady=8)
        dir_frame = ttk.Frame(form_frame, style="Light.TFrame") # Frame interno claro
        dir_frame.grid(row=11, column=1, columnspan=2, sticky=tk.EW, padx=5, pady=8)
        dir_frame.columnconfigure(0, weight=1)
        entry_dir = ttk.Entry(dir_frame, textvariable=self.directorio_salida, font=self.font_normal, width=50)
        entry_dir.grid(row=0, column=0, sticky=tk.EW, padx=(0, 5))
        btn_dir = ttk.Button(dir_frame, text="Examinar...", width=12, style="Primario.TButton", command=self.seleccionar_directorio)
        btn_dir.grid(row=0, column=1, sticky=tk.E)
        
        # Registrar callbacks para actualizar la vista previa cuando cambie cualquier campo
        self.termino_general.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.termino_adicional1.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.termino_adicional2.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.termino_adicional3.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.operador_logico1.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.operador_logico2.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        self.operador_logico3.trace_add("write", lambda *args: self.actualizar_vista_previa_consulta())
        
        # Inicializar vista previa y descripciones de operadores
        self.actualizar_vista_previa_consulta()
        self.actualizar_descripcion_operador(1)
        self.actualizar_descripcion_operador(2)
        self.actualizar_descripcion_operador(3)
    
    def actualizar_descripcion_operador(self, num_operador):
        """Actualiza la descripción del operador lógico según la selección."""
        descripciones = {
            "AND": "Resultados que contienen AMBOS términos",
            "OR": "Resultados que contienen CUALQUIERA de los términos",
            "NOT": "Resultados que NO contienen este término"
        }
        
        if num_operador == 1:
            operador = self.operador_logico1.get()
            self.lbl_desc_op1.config(text=descripciones.get(operador, ""))
        elif num_operador == 2:
            operador = self.operador_logico2.get()
            self.lbl_desc_op2.config(text=descripciones.get(operador, ""))
        elif num_operador == 3:
            operador = self.operador_logico3.get()
            self.lbl_desc_op3.config(text=descripciones.get(operador, ""))
        
        # Actualizar vista previa después de cambiar el operador
        self.actualizar_vista_previa_consulta()
    
    def actualizar_vista_previa_consulta(self):
        """Actualiza la vista previa de la consulta con los términos y operadores actuales."""
        termino_general = self.termino_general.get().strip()
        
        # Si no hay término general, mostrar mensaje de ayuda
        if not termino_general:
            self.vista_previa_consulta.set("Ingrese un término general para ver la vista previa de la consulta.")
            return
        
        # Construir la consulta
        consulta = [f'"{termino_general}"' if ' ' in termino_general else termino_general]
        
        # Añadir términos adicionales con sus operadores si no están vacíos
        termino1 = self.termino_adicional1.get().strip()
        if termino1:
            operador1 = self.operador_logico1.get()
            term_formateado = f'"{termino1}"' if ' ' in termino1 else termino1
            consulta.append(f"{operador1} {term_formateado}")
        
        termino2 = self.termino_adicional2.get().strip()
        if termino2:
            operador2 = self.operador_logico2.get()
            term_formateado = f'"{termino2}"' if ' ' in termino2 else termino2
            consulta.append(f"{operador2} {term_formateado}")
        
        termino3 = self.termino_adicional3.get().strip()
        if termino3:
            operador3 = self.operador_logico3.get()
            term_formateado = f'"{termino3}"' if ' ' in termino3 else termino3
            consulta.append(f"{operador3} {term_formateado}")
        
        # Actualizar la variable de vista previa
        self.vista_previa_consulta.set(" ".join(consulta))
    
    def mostrar_ayuda_operadores(self):
        """Muestra una ventana de ayuda sobre el uso de operadores lógicos."""
        messagebox.showinfo(
            "Ayuda de Operadores Lógicos",
            "Operadores disponibles para términos adicionales:\n\n"
            "AND: Ambos términos deben estar presentes en los resultados.\n"
            "Ejemplo: 'amparo AND judicial' → Documentos con 'amparo' Y 'judicial'.\n\n"
            "OR: Al menos uno de los términos debe estar presente.\n"
            "Ejemplo: 'civil OR mercantil' → Documentos con 'civil' O 'mercantil'.\n\n"
            "NOT: Excluye documentos que contienen el término.\n"
            "Ejemplo: 'penal NOT federal' → Documentos con 'penal' PERO SIN 'federal'.\n\n"
            "CONSEJOS:\n"
            "• Use comillas para frases exactas: \"debido proceso\"\n"
            "• Combine operadores de manera lógica (AND tiene prioridad sobre OR)\n"
            "• El término general siempre se incluye en la búsqueda",
            parent=self.root
        )

    def crear_seccion_instrucciones_rediseñada(self, parent):
        """Crea la sección de instrucciones con imagen (Fondo Oscuro)."""
        # Usar el estilo Dark.Card.TLabelframe
        info_frame = ttk.LabelFrame(parent, text="Guía Rápida de Consulta", style="Dark.Card.TLabelframe")
        info_frame.pack(fill=tk.X, pady=0, padx=10, ipady=5) # Padding interior y lateral

        # Crear un contenedor para la imagen con un tamaño definido adecuado
        imagen_frame = ttk.Frame(info_frame, style="Dark.TFrame")
        imagen_frame.pack(fill=tk.BOTH, expand=True, pady=5, padx=5)
        
        # Hacer accesible la función de redimensionamiento
        redimensionar_imagen = self.redimensionar_imagen
        
        # Intentar importar PIL para tener una alternativa
        try:
            from PIL import Image, ImageTk
            PIL_DISPONIBLE = True
        except ImportError:
            PIL_DISPONIBLE = False
            gui_logger.warning("PIL no está disponible. Se usará sólo PhotoImage para cargar la imagen.")
        
        # Función para cargar imagen con soporte para múltiples métodos
        def cargar_imagen(ruta, max_width=None, max_height=None):
            """
            Carga una imagen y opcionalmente la redimensiona.
            
            Args:
                ruta: Ruta del archivo de imagen
                max_width: Ancho máximo (solo aplica si se usa PIL)
                max_height: Alto máximo (solo aplica si se usa PIL)
            """
            if not os.path.exists(ruta):
                return None, f"Archivo no existe: {ruta}"
                
            # Intentar con PhotoImage nativo primero
            try:
                return tk.PhotoImage(file=ruta), None
            except Exception as e:
                # Intentar importar PIL para alternativa
                try:
                    from PIL import Image, ImageTk
                    try:
                        img_pil = Image.open(ruta)
                        
                        # Si se especificaron dimensiones máximas, redimensionar
                        if max_width is not None or max_height is not None:
                            img_pil = redimensionar_imagen(img_pil, max_width, max_height)
                        
                        return ImageTk.PhotoImage(img_pil), None
                    except Exception as e2:
                        return None, f"Error PIL: {str(e2)}"
                except ImportError:
                    return None, f"Error PhotoImage: {str(e)}"
        
        # Rutas a intentar en orden de preferencia
        rutas_a_intentar = [
            # Ruta original
            r"C:\Users\hhugo\Dropbox\SCRIPTS DE PYTHON\ROBOT INTEGRADOR TESIS SCJN V5 CURSOR}\guia rapida consulta.png",
            # Ruta corregida
            r"C:\Users\hhugo\Dropbox\SCRIPTS DE PYTHON\ROBOT INTEGRADOR TESIS SCJN V5 CURSOR\guia rapida consulta.png",
            # Ruta en directorio actual
            os.path.join(os.getcwd(), "guia rapida consulta.png"),
            # Ruta en directorio del script
            os.path.join(os.path.dirname(os.path.abspath(__file__)), "guia rapida consulta.png")
        ]
        
        # Definir tamaño máximo para la imagen de guía rápida
        max_width_guia = 595  # Reducido a 0.85x del tamaño anterior
        max_height_guia = 340  # Reducido a 0.85x del tamaño anterior
        
        # Intentar cada ruta
        imagen = None
        error_final = ""
        
        for ruta in rutas_a_intentar:
            imagen, error = cargar_imagen(ruta, max_width_guia, max_height_guia)
            if imagen is not None:
                gui_logger.info(f"Imagen cargada desde: {ruta}")
                break
            error_final += f"\nFalló ruta {ruta}: {error}"
        
        # Si se logró cargar la imagen
        if imagen is not None:
            # Guardar referencia a la imagen (importante para evitar garbage collection)
            self.guia_rapida_imagen = imagen
            
            # Crear un contenedor para centrar la imagen
            imagen_container = ttk.Frame(imagen_frame, style="Dark.TFrame")
            imagen_container.pack(fill=None, expand=True, pady=5, padx=5)
            
            # Crear label para mostrar la imagen
            lbl_imagen = ttk.Label(imagen_container, image=imagen, style="Dark.TLabel", background=self.color_fondo_oscuro)
            lbl_imagen.pack(fill=None, expand=True)
            
            gui_logger.info("Imagen de guía rápida cargada correctamente.")
            
            # Log para confirmar carga exitosa
            self.log("Guía rápida de consulta cargada correctamente.", "info")
        else:
            # Registrar el error
            gui_logger.error(f"No se pudo cargar la imagen de guía rápida.{error_final}")
            
            # Texto de fallback
            instrucciones_text = """
Sintaxis de Búsqueda:
- Frase Exacta:   Encerrar entre comillas ("ejemplo frase").
- Palabras AND:   Separar palabras por espacio (palabra1 palabra2).
- Operadores:     Use los selectores AND, OR, NOT para cada término adicional.

Ejemplos de Operadores:
- AND: Resultados que contienen AMBOS términos ("amparo" Y "judicial")
- OR:  Resultados que contienen CUALQUIERA de los términos ("civil" O "mercantil")
- NOT: Resultados que NO contienen este término (con "penal" pero SIN "federal")

Notas:
- El 'Término General' es obligatorio.
- Cada 'Término Adicional' puede tener su propio operador lógico.
- La búsqueda es sensible a acentos.
"""
            # Usar el estilo Dark.Bold.TLabel para texto blanco y negrita sobre fondo oscuro
            ttk_label = ttk.Label(imagen_frame, text=instrucciones_text, justify=tk.LEFT, style="Dark.Bold.TLabel")
            ttk_label.pack(anchor=tk.W, pady=5, padx=5)
            
            # Registrar el problema en el log
            self.log("No se pudo cargar la imagen de guía rápida. Se muestra texto alternativo.", "warning")

    def redimensionar_imagen(self, imagen_pil, max_width=None, max_height=None):
        """
        Redimensiona una imagen PIL manteniendo su proporción.
        
        Args:
            imagen_pil: Imagen PIL a redimensionar
            max_width: Ancho máximo deseado
            max_height: Alto máximo deseado
        
        Returns:
            Imagen PIL redimensionada
        """
        if max_width is None and max_height is None:
            return imagen_pil
        
        # Obtener dimensiones originales
        orig_width, orig_height = imagen_pil.size
        
        # Calcular factor de escala
        scale_factor = 1.0
        
        if max_width is not None and orig_width > max_width:
            scale_factor = max_width / orig_width
        
        if max_height is not None and orig_height > max_height:
            height_scale = max_height / orig_height
            scale_factor = min(scale_factor, height_scale)
        
        # Si no necesita redimensionar, retornar original
        if scale_factor >= 1.0:
            return imagen_pil
        
        # Calcular nuevas dimensiones
        new_width = int(orig_width * scale_factor)
        new_height = int(orig_height * scale_factor)
        
        # Redimensionar manteniendo calidad
        return imagen_pil.resize((new_width, new_height), Image.Resampling.LANCZOS)

    def crear_seccion_filtros_rediseñada(self, parent):
        """Crea la sección de filtros (Tipo y Épocas) (Fondo Claro)."""
        # Contenedor claro
        filtros_container = ttk.Frame(parent, style="Light.TFrame")
        filtros_container.pack(fill=tk.X, pady=10, padx=10)

        # Frame Izquierdo: Tipo de Tesis (Claro)
        tesis_frame = ttk.LabelFrame(filtros_container, text="2. Tipo de Tesis", style="Card.TLabelframe")
        tesis_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(0, 5))
        # Usar TRadiobutton (estilo claro por defecto)
        ttk.Radiobutton(tesis_frame, text="Todas (Jurisprudencia y Aislada)", variable=self.tipo_tesis, value=0).pack(anchor=tk.W, pady=5, padx=10)
        ttk.Radiobutton(tesis_frame, text="Sólo Jurisprudencia", variable=self.tipo_tesis, value=1).pack(anchor=tk.W, pady=5, padx=10)
        ttk.Radiobutton(tesis_frame, text="Sólo Aislada", variable=self.tipo_tesis, value=2).pack(anchor=tk.W, pady=5, padx=10)

        # Frame Derecho: Épocas Adicionales (Claro)
        epocas_frame = ttk.LabelFrame(filtros_container, text="3. Épocas Adicionales (11ª, 10ª, 9ª siempre)", style="Card.TLabelframe")
        epocas_frame.pack(side=tk.LEFT, fill=tk.BOTH, expand=True, padx=(5, 0))
        # Usar TCheckbutton (estilo claro por defecto)
        ttk.Checkbutton(epocas_frame, text="Incluir 8ª Época", variable=self.incluir_8a_epoca).pack(anchor=tk.W, pady=5, padx=10)
        ttk.Checkbutton(epocas_frame, text="Incluir 7ª Época", variable=self.incluir_7a_epoca).pack(anchor=tk.W, pady=5, padx=10)
        ttk.Checkbutton(epocas_frame, text="Incluir 6ª Época", variable=self.incluir_6a_epoca).pack(anchor=tk.W, pady=5, padx=10)
        ttk.Checkbutton(epocas_frame, text="Incluir 5ª Época", variable=self.incluir_5a_epoca).pack(anchor=tk.W, pady=5, padx=10)

    def crear_seccion_opciones_rediseñada(self, parent):
        """Crea la sección de opciones de salida (Fondo Oscuro)."""
        # Usar Dark.Card.TLabelframe
        options_frame = ttk.LabelFrame(parent, text="4. Opciones de Salida", style="Dark.Card.TLabelframe")
        options_frame.pack(fill=tk.X, pady=0, padx=10, ipady=5) # Padding interior y lateral

        # Contenedor oscuro
        options_container = ttk.Frame(options_frame, style="Dark.TFrame")
        options_container.pack(fill=tk.X, expand=True, padx=5, pady=5)
        options_container.columnconfigure(0, weight=1) # Solo una columna ahora

        # Usar Dark.TCheckbutton para texto claro sobre oscuro
        check1 = ttk.Checkbutton(options_container, 
                                text="Compilar resultados en archivo PDF", 
                                variable=self.compilar_pdfs, 
                                style="Dark.TCheckbutton")
        check1.grid(row=0, column=0, sticky=tk.W, pady=6, padx=10)
        
        check2 = ttk.Checkbutton(options_container, 
                                text="Abrir archivos generados al finalizar", 
                                variable=self.abrir_archivo, 
                                style="Dark.TCheckbutton")
        check2.grid(row=1, column=0, sticky=tk.W, pady=6, padx=10)

        # Notas sobre opciones fijas (texto claro/grisáceo)
        ttk.Label(options_container, 
                  text="Nota: Modo oculto y descarga optimizada siempre activos.", 
                  style="Dark.Info.TLabel").grid(row=2, column=0, sticky=tk.W, pady=(10,3), padx=5)

    def crear_seccion_resultados_rediseñada(self, parent):
        """Crea la sección para mostrar resultados preliminares (Fondo Claro)."""
        # Usar Card.TLabelframe (claro)
        self.resultados_frame = ttk.LabelFrame(parent, text="Resultados Preliminares", style="Card.TLabelframe")
        self.resultados_frame.pack(fill=tk.BOTH, expand=True, pady=10, padx=10)

        # Frame interno claro
        tree_frame = ttk.Frame(self.resultados_frame, style="Light.TFrame")
        tree_frame.pack(fill=tk.BOTH, expand=True, pady=(5,0))

        # Usar scrollbar estilo claro
        tree_scrolly = ttk.Scrollbar(tree_frame, orient="vertical", style="Vertical.TScrollbar")
        tree_scrollx = ttk.Scrollbar(tree_frame, orient="horizontal", style="Horizontal.TScrollbar")

        # Usar Treeview estilo claro
        self.resultados_tree = ttk.Treeview(tree_frame,
                                          columns=("numero", "registro", "num_tesis", "rubro"),
                                          show="headings",
                                          height=8,
                                          yscrollcommand=tree_scrolly.set,
                                          xscrollcommand=tree_scrollx.set,
                                          style="Treeview") # Estilo base claro

        tree_scrolly.config(command=self.resultados_tree.yview)
        tree_scrollx.config(command=self.resultados_tree.xview)

        self.resultados_tree.heading("numero", text="Número", anchor=tk.CENTER)
        self.resultados_tree.heading("registro", text="Reg. Digital", anchor=tk.W)
        self.resultados_tree.heading("num_tesis", text="No. Tesis / Clave", anchor=tk.W)
        self.resultados_tree.heading("rubro", text="Rubro / Título", anchor=tk.W)
        self.resultados_tree.column("numero", width=60, stretch=tk.NO, anchor=tk.CENTER) # Columna para numeración
        self.resultados_tree.column("registro", width=120, stretch=tk.NO, anchor=tk.W) # Ajustar ancho
        self.resultados_tree.column("num_tesis", width=240, stretch=tk.NO, anchor=tk.W) # Ajustar ancho
        self.resultados_tree.column("rubro", width=730, stretch=tk.YES, anchor=tk.W) # Ajustar ancho

        # Configurar tags para filas alternadas (opcional, ajustar colores si se usa)
        self.resultados_tree.tag_configure('oddrow', background='#FFFFFF', foreground=self.color_texto_oscuro)
        self.resultados_tree.tag_configure('evenrow', background='#F7F9FA', foreground=self.color_texto_oscuro)
        self.resultados_tree.tag_configure('datos_oficiales', background='#E3F2FD', foreground=self.color_texto_oscuro) # Fondo azul claro

        # Enlazar evento de doble clic
        self.resultados_tree.bind('<Double-1>', self.mostrar_detalle_tesis)

        tree_scrollx.pack(side=tk.BOTTOM, fill=tk.X)
        tree_scrolly.pack(side=tk.RIGHT, fill=tk.Y)
        self.resultados_tree.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        # Usar Info.TLabel (claro)
        info_frame = ttk.Frame(self.resultados_frame, style="Light.TFrame")
        info_frame.pack(fill=tk.X, pady=(5,0))
        
        ttk.Label(info_frame, text="[Doble clic en una fila para ver detalles completos]", 
                 style="Info.TLabel").pack(side=tk.LEFT, padx=(0, 10))
        
        # Botón para generar reporte preliminar
        self.btn_reporte = ttk.Button(info_frame, 
                                     text="Reporte Preliminar", 
                                     command=self.generar_reporte_preliminar,
                                     style="Primario.TButton")
        self.btn_reporte.pack(side=tk.LEFT)

    def crear_seccion_log_rediseñada(self, parent):
        """Crea la sección de registro de actividad (Fondo Oscuro)."""
        # Usar Dark.Card.TLabelframe
        log_frame = ttk.LabelFrame(parent, text="Registro de Actividad del Sistema", style="Dark.Card.TLabelframe")
        log_frame.pack(fill=tk.BOTH, expand=True, pady=0, padx=10, ipady=5) # Padding interior y lateral

        # Marco interno para el ScrolledText con fondo oscuro
        inner_frame = ttk.Frame(log_frame, style="Dark.TFrame")
        inner_frame.pack(fill=tk.BOTH, expand=True, pady=(5,0))
        
        # Configurar ScrolledText con colores específicos para fondo oscuro
        self.log_text = ScrolledText(
            inner_frame, 
            height=8, 
            width=80, 
            wrap=tk.WORD,
            font=self.font_normal,
            background=self.color_fondo_oscuro,          # Fondo oscuro EXPLÍCITO
            foreground=self.color_texto_claro,           # Texto blanco para mejor contraste
            insertbackground=self.color_texto_claro,     # Cursor blanco
            selectbackground=self.color_acento,          # Selección azul
            selectforeground=self.color_texto_claro,     # Texto blanco en selección
            highlightthickness=1,                        # Borde cuando tiene foco
            highlightbackground=self.color_borde_oscuro, # Color borde normal
            highlightcolor=self.color_acento,            # Color borde con foco
            bd=0,                                        # Sin borde adicional
            padx=5, 
            pady=5
        )
        
        # Usar pack en lugar de grid para llenar el frame completamente
        self.log_text.pack(fill=tk.BOTH, expand=True)
        self.log_text.config(state=tk.DISABLED)

        # Configurar etiquetas para coloreado (usar colores definidos en self.log_tags)
        for tag_name, config in self.log_tags.items():
            self.log_text.tag_configure(tag_name, **config)
            
        # Insertar un mensaje inicial para confirmar que el estilo se aplica correctamente
        self.log_text.config(state=tk.NORMAL)
        self.log_text.insert(tk.END, "[Sistema] Interfaz iniciada. Esperando consulta...\n", "info")
        self.log_text.config(state=tk.DISABLED)

    def actualizar_progreso(self):
        """Actualiza la barra y etiquetas de progreso periódicamente."""
        if not self.busqueda_en_progreso: 
            return

        if self.progreso_actual < 98:
            incremento = max(0.1, (100 - self.progreso_actual) / 50)
            self.progreso_actual = min(98, self.progreso_actual + incremento)

        # Actualizar la barra de progreso
        if hasattr(self, 'barra_progreso'): 
            self.barra_progreso["value"] = self.progreso_actual
        elif hasattr(self, 'progress'):
            self.progress["value"] = self.progreso_actual
                
        # Actualizar el porcentaje
        if hasattr(self, 'etiqueta_porcentaje'):
            self.etiqueta_porcentaje.config(text=f"{int(self.progreso_actual)}%")
        elif hasattr(self, 'progress_percent'):
            self.progress_percent.config(text=f"{int(self.progreso_actual)}%")
                
        # Actualizar el texto de estado
        indice_etapa = min(int(self.progreso_actual / (100 / len(self.etapas_progreso))), len(self.etapas_progreso) - 1)
        
        if hasattr(self, 'etiqueta_estado'):
            self.etiqueta_estado.config(text=f"Estado: {self.etapas_progreso[indice_etapa]}")
        elif hasattr(self, 'progress_status'):
            self.progress_status.config(text=f"Estado: {self.etapas_progreso[indice_etapa]}")
                
        # Actualizar la descripción de fase
        if hasattr(self, 'etiqueta_fase'):
            self.etiqueta_fase.config(text=f"Ejecutando: {self.etapas_progreso[indice_etapa]} (Avance: {int(self.progreso_actual)}%)")
        elif hasattr(self, 'progress_label'):
            self.progress_label.config(text=f"{self.etapas_progreso[indice_etapa]} ({int(self.progreso_actual)}%)")

        # Programar la próxima actualización
        self.root.after(150, self.actualizar_progreso)

    def detener_busqueda(self):
        """Detiene la búsqueda en progreso"""
        if self.busqueda_en_progreso:
            self.busqueda_en_progreso = False
            self.log("Proceso de búsqueda detenido por el usuario", "warning")
            self.status_var.set("Estado: Búsqueda detenida")
            if hasattr(self, 'progress_label'):
                self.progress_label.config(text="Estado: Detenido")
                
            # Actualizar botones de manera segura
            try:
                if hasattr(self, 'btn_extraer'): self.btn_extraer.config(state=tk.NORMAL)
                if hasattr(self, 'btn_detener'): self.btn_detener.config(state=tk.DISABLED)
            except Exception as e:
                gui_logger.warning(f"Advertencia: No se pudieron actualizar algunos botones: {str(e)}")

    def abrir_compilado(self):
        """Abre el último archivo compilado generado"""
        # Buscar el archivo más reciente en el directorio de salida
        directorio = self.directorio_salida.get().strip()
        if not os.path.exists(directorio):
            messagebox.showerror("Error", "El directorio de salida no existe", parent=self.root)
            return
            
        # Buscar el archivo compilado más reciente (primero PDF, luego TXT)
        archivos_pdf = [f for f in os.listdir(directorio) if f.endswith('.pdf')]
        archivos_txt = [f for f in os.listdir(directorio) if f.endswith('.txt')]
        
        if archivos_pdf:
            # Ordenar por fecha de modificación (más reciente primero)
            ultimo_pdf = sorted(archivos_pdf, 
                               key=lambda x: os.path.getmtime(os.path.join(directorio, x)), 
                               reverse=True)[0]
            self.abrir_archivo_os(os.path.join(directorio, ultimo_pdf))
            return
        
        if archivos_txt:
            # Ordenar por fecha de modificación (más reciente primero)
            ultimo_txt = sorted(archivos_txt, 
                               key=lambda x: os.path.getmtime(os.path.join(directorio, x)), 
                               reverse=True)[0]
            self.abrir_archivo_os(os.path.join(directorio, ultimo_txt))
            return
            
        messagebox.showinfo("Información", 
                           "No se encontraron archivos compilados en el directorio de salida", 
                           parent=self.root)

    def seleccionar_directorio(self):
        """Abre diálogo para seleccionar directorio de salida."""
        directorio_inicial = self.directorio_salida.get()
        if not os.path.isdir(directorio_inicial):
             directorio_inicial = os.getcwd()

        directorio = filedialog.askdirectory(
            title="Seleccionar directorio para guardar resultados",
            initialdir=directorio_inicial)
        if directorio:
            self.directorio_salida.set(directorio)
            gui_logger.info(f"Directorio de salida seleccionado: {directorio}")

    def mostrar_detalle_tesis(self, event):
        """Muestra ventana con detalle completo de la tesis seleccionada."""
        seleccion = self.resultados_tree.selection()
        if not seleccion: return
        item_id = seleccion[0]
        item_data = self.resultados_tree.item(item_id)
        valores = item_data['values']
        registro_buscado = valores[0]

        gui_logger.info(f"Mostrando detalle para registro: {registro_buscado}")
        tesis_completa = next((t for t in self.resultados_completos if str(t.get('registro_digital', '')) == str(registro_buscado)), None)

        if not tesis_completa:
            gui_logger.warning(f"No se encontraron datos completos para registro: {registro_buscado}")
            messagebox.showwarning("Datos no encontrados", f"No se pudo encontrar la información completa para el registro {registro_buscado}.", parent=self.root)
            return

        # --- Crear Ventana de Detalle (Aplicar estilo claro base) ---
        detalle_window = tk.Toplevel(self.root)
        detalle_window.title(f"Detalle Tesis: {registro_buscado}")
        detalle_window.geometry("950x750")
        detalle_window.minsize(800, 600)
        detalle_window.configure(background=self.color_fondo_claro) # Fondo claro

        # Frame principal con scroll (usar estilos claros)
        main_container_det = ttk.Frame(detalle_window, style="Light.TFrame")
        main_container_det.pack(fill=tk.BOTH, expand=True)
        canvas_det = tk.Canvas(main_container_det, background=self.color_fondo_claro, highlightthickness=0)
        scrollbar_det = ttk.Scrollbar(main_container_det, orient="vertical", command=canvas_det.yview, style="Vertical.TScrollbar")
        scrollable_frame_det = ttk.Frame(canvas_det, style="Light.TFrame", padding=20)

        scrollable_frame_det.bind("<Configure>", lambda e: canvas_det.configure(scrollregion=canvas_det.bbox("all")))
        canvas_window_det = canvas_det.create_window((0, 0), window=scrollable_frame_det, anchor="nw")
        canvas_det.configure(yscrollcommand=scrollbar_det.set)

        def configure_scroll_frame_detalle(event):
            canvas_width = event.width
            canvas_det.itemconfig(canvas_window_det, width=canvas_width)
        canvas_det.bind("<Configure>", configure_scroll_frame_detalle)

        scrollbar_det.pack(side="right", fill="y")
        canvas_det.pack(side="left", fill="both", expand=True)

        # --- Contenido de la Ventana de Detalle (Usar estilos claros) ---
        es_descarga_directa = tesis_completa.get("metodo_obtencion") == "descarga_directa"
        if es_descarga_directa:
            oficial_frame = ttk.Frame(scrollable_frame_det, style="Light.TFrame")
            oficial_frame.pack(fill=tk.X, pady=(0, 10))
            # Usar Success.TLabel con fondo claro explícito
            ttk.Label(oficial_frame, text="✓ Datos Oficiales SCJN", style="Success.TLabel").pack(anchor=tk.W)

        rubro = tesis_completa.get('rubro', 'Rubro no disponible')
        # Usar un estilo de título claro, ej. Campo.TLabel o crear uno nuevo
        ttk.Label(scrollable_frame_det, text=rubro, style="Campo.TLabel", wraplength=850, font=self.font_seccion).pack(fill=tk.X, pady=(0, 10)) # Usar font_seccion

        ttk.Separator(scrollable_frame_det, orient="horizontal").pack(fill=tk.X, pady=10)

        meta_frame = ttk.Frame(scrollable_frame_det, style="Light.TFrame")
        meta_frame.pack(fill=tk.X, pady=5)
        meta_frame.columnconfigure(1, weight=1)
        meta_frame.columnconfigure(3, weight=1)
        row_idx, col_idx = 0, 0
        campos_mostrar = [
            ('registro_digital', 'Reg. Digital'), ('numero_tesis', 'No. Tesis/Clave'),
            ('epoca', 'Época'), ('instancia', 'Instancia'),
            ('fuente', 'Fuente'), ('materia', 'Materia'),
            ('tipo_tesis', 'Tipo'), ('fecha_publicacion', 'Fec. Pub.'),
            ('pagina', 'Página'), ('estado_vigencia', 'Vigencia'),
            ('libro', 'Libro/Tomo'), ('volumen', 'Volumen') ]

        for key, label_text in campos_mostrar:
            valor = tesis_completa.get(key, '')
            if valor and valor != 'No identificado':
                ttk.Label(meta_frame, text=f"{label_text}:", style="Campo.TLabel").grid(row=row_idx, column=col_idx, sticky=tk.W, padx=5, pady=2)
                ttk.Label(meta_frame, text=valor, style="Info.TLabel", wraplength=350).grid(row=row_idx, column=col_idx + 1, sticky=tk.W, padx=5, pady=2)
                col_idx = (col_idx + 2) % 4
                if col_idx == 0: row_idx += 1

        ttk.Separator(scrollable_frame_det, orient="horizontal").pack(fill=tk.X, pady=10)

        ttk.Label(scrollable_frame_det, text="Texto Íntegro:", style="Campo.TLabel").pack(anchor=tk.W, pady=(5, 5))
        texto_frame = ttk.Frame(scrollable_frame_det, style="Card.TFrame") # Con borde claro
        texto_frame.pack(fill=tk.BOTH, expand=True, pady=5)

        texto_detalle = tk.Text(texto_frame, wrap=tk.WORD, font=self.font_normal,
                                padx=10, pady=10, background=self.color_fondo_widget, # Fondo widget claro
                                foreground=self.color_texto_oscuro, # Texto oscuro
                                borderwidth=0, relief="flat", height=15)
        scroll_texto = ttk.Scrollbar(texto_frame, orient="vertical", command=texto_detalle.yview, style="Vertical.TScrollbar")
        texto_detalle.configure(yscrollcommand=scroll_texto.set)
        texto_detalle.insert(tk.END, tesis_completa.get('texto_completo', 'Texto no disponible'))
        texto_detalle.config(state=tk.DISABLED)
        scroll_texto.pack(side=tk.RIGHT, fill=tk.Y)
        texto_detalle.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)

        ttk.Button(scrollable_frame_det, text="Cerrar Detalle", command=detalle_window.destroy, style="Primario.TButton").pack(pady=(15, 5))

        detalle_window.update_idletasks()
        w, h = detalle_window.winfo_width(), detalle_window.winfo_height()
        ws, hs = detalle_window.winfo_screenwidth(), detalle_window.winfo_screenheight()
        x, y = max(0, (ws/2) - (w/2)), max(0, (hs/2) - (h/2))
        detalle_window.geometry('%dx%d+%d+%d' % (w, h, x, y))
        detalle_window.transient(self.root)
        detalle_window.grab_set()


    def log(self, mensaje, tipo="info"):
        """Añade mensaje al área de log con estilo."""
        if not hasattr(self, 'log_text') or not self.log_text.winfo_exists(): return

        self.log_text.config(state=tk.NORMAL)
        timestamp = datetime.now().strftime("[%H:%M:%S] ")
        tag = tipo if tipo in self.log_tags else "info"

        self.log_text.insert(tk.END, timestamp)
        self.log_text.insert(tk.END, mensaje + "\n", tag)
        self.log_text.see(tk.END)
        self.log_text.config(state=tk.DISABLED)
        # self.root.update_idletasks() # Puede ralentizar, usar con cuidado


    def limpiar_campos(self):
        """Limpia todos los campos del formulario y resetea la interfaz."""
        gui_logger.info("Limpiando campos y resultados...")
        self.termino_general.set("")
        self.termino_adicional1.set("")
        self.termino_adicional2.set("")
        self.termino_adicional3.set("")
        # Reiniciar operadores lógicos a sus valores por defecto
        self.operador_logico1.set("AND")
        self.operador_logico2.set("AND")
        self.operador_logico3.set("AND")
        # Actualizar vista previa de consulta
        self.actualizar_vista_previa_consulta()
        # Actualizar descripciones de operadores
        self.actualizar_descripcion_operador(1)
        self.actualizar_descripcion_operador(2)
        self.actualizar_descripcion_operador(3)
        
        self.max_resultados.set(50)
        # No resetear directorio
        self.compilar_pdfs.set(True)
        self.abrir_archivo.set(False)
        self.incluir_8a_epoca.set(False); self.incluir_7a_epoca.set(False)
        self.incluir_6a_epoca.set(False); self.incluir_5a_epoca.set(False)
        self.tipo_tesis.set(0)

        if hasattr(self, 'resultados_tree'):
            for item in self.resultados_tree.get_children(): self.resultados_tree.delete(item)
            self.resultados_frame.config(text="Resultados Preliminares") # Reset title

        if hasattr(self, 'log_text') and self.log_text.winfo_exists():
            self.log_text.config(state=tk.NORMAL); self.log_text.delete(1.0, tk.END); 
            self.log_text.insert(tk.END, "[Sistema] Campos y resultados limpiados. Esperando nueva consulta...\n", "info")
            self.log_text.config(state=tk.DISABLED)

        # Reiniciar barra de progreso y etiquetas
        if hasattr(self, 'barra_progreso'): 
            self.barra_progreso["value"] = 0
            
        if hasattr(self, 'etiqueta_estado'):
            self.etiqueta_estado.config(text="Estado: Listo para iniciar búsqueda")
            
        if hasattr(self, 'etiqueta_porcentaje'):
            self.etiqueta_porcentaje.config(text="0%")
            
        if hasattr(self, 'etiqueta_fase'):
            self.etiqueta_fase.config(text="Listo para iniciar el proceso de extracción de jurisprudencia")
        
        # Compatibilidad con código anterior
        if hasattr(self, 'progress'): self.progress["value"] = 0
        if hasattr(self, 'progress_label'): self.progress_label.config(text="Estado: Listo")
        
        # Actualizar estado de los botones
        self.btn_extraer.config(state=tk.NORMAL)
        if hasattr(self, 'btn_detener'):
            self.btn_detener.config(state=tk.DISABLED)
        if hasattr(self, 'btn_abrir'):
            self.btn_abrir.config(state=tk.DISABLED)
        
        self.status_var.set("Estado: Listo para nueva búsqueda")
        self.resultados_completos = []
        self.total_resultados_real = 0


    def iniciar_busqueda(self):
        """Valida entradas e inicia el proceso de búsqueda en un hilo."""
        gui_logger.info("Iniciando validación para búsqueda...")
        if not self.termino_general.get().strip():
            messagebox.showerror("Error de Validación", "El campo 'Término General' es obligatorio.", parent=self.root)
            gui_logger.error("Validación fallida: Término general vacío.")
            return

        directorio = self.directorio_salida.get().strip()
        if not directorio:
             messagebox.showerror("Error de Validación", "Debe especificar un 'Directorio Salida'.", parent=self.root)
             gui_logger.error("Validación fallida: Directorio de salida vacío.")
             return
        try:
            if not os.path.exists(directorio):
                os.makedirs(directorio)
                self.log(f"Directorio creado: {directorio}", "info")
                gui_logger.info(f"Directorio de salida creado: {directorio}")
        except Exception as e:
            messagebox.showerror("Error de Directorio", f"No se pudo crear el directorio:\n{directorio}\n\nError: {str(e)}", parent=self.root)
            gui_logger.error(f"Error al crear directorio '{directorio}': {str(e)}")
            return

        if hasattr(self, 'resultados_tree'):
            for item in self.resultados_tree.get_children(): self.resultados_tree.delete(item)
        self.resultados_completos = []; self.total_resultados_real = 0
        if hasattr(self, 'resultados_frame'): self.resultados_frame.config(text="Resultados Preliminares")

        # Actualizar la interfaz de la barra de progreso vanguardista
        try:
            if hasattr(self, 'etiqueta_estado'):
                self.etiqueta_estado.config(text="Estado: Iniciando proceso...")
            if hasattr(self, 'etiqueta_porcentaje'):
                self.etiqueta_porcentaje.config(text="0%")
            if hasattr(self, 'etiqueta_fase'):
                self.etiqueta_fase.config(text="Inicializando búsqueda...")
                
            # Compatibilidad con código anterior
            if hasattr(self, 'progress_status'):
                self.progress_status.config(text="Estado: Iniciando proceso...")
            if hasattr(self, 'progress_percent'):
                self.progress_percent.config(text="0%")
        except Exception as e:
            gui_logger.warning(f"Advertencia: No se pudo actualizar la barra de progreso: {str(e)}")

        # Manejo seguro de los atributos de botones
        try:
            # Deshabilitar el botón extraer
            if hasattr(self, 'btn_extraer'): 
                self.btn_extraer.config(state=tk.DISABLED)
                
            # Mostrar y habilitar el botón detener si existe
            if hasattr(self, 'btn_detener'):
                self.btn_detener.config(state=tk.NORMAL)
                
            # Deshabilitar botones adicionales
            if hasattr(self, 'btn_limpiar'):
                self.btn_limpiar.config(state=tk.DISABLED)
            if hasattr(self, 'btn_ayuda'):
                self.btn_ayuda.config(state=tk.DISABLED)
            if hasattr(self, 'btn_info'):
                self.btn_info.config(state=tk.DISABLED)
                
            # Ocultar abrir compilado
            if hasattr(self, 'btn_abrir'): 
                self.btn_abrir.config(state=tk.DISABLED)
        except Exception as e:
            gui_logger.warning(f"Advertencia: No se pudieron actualizar algunos botones: {str(e)}")

        self.busqueda_en_progreso = True; self.progreso_actual = 0
        
        # Actualizar barras de progreso
        if hasattr(self, 'barra_progreso'): 
            self.barra_progreso["value"] = 0
        if hasattr(self, 'progress'): 
            self.progress["value"] = 0
            
        if hasattr(self, 'progress_label'): 
            self.progress_label.config(text=self.etapas_progreso[0])
            
        self.status_var.set("Estado: Ejecutando búsqueda...")
        self.log("--- Iniciando Búsqueda y Extracción ---", "info")
        if self.usar_descarga_directa:
            self.log("Modo: Descarga directa optimizada ACTIVADA", "info")

        self.actualizar_progreso()
        threading.Thread(target=self.ejecutar_busqueda, daemon=True).start()


    def ejecutar_busqueda(self):
        """Ejecuta la lógica de búsqueda en el hilo secundario."""
        gui_logger.info("Hilo de búsqueda iniciado.")
        try:
            if not INTEGRADOR_IMPORTADO:
                 self.root.after(0, self.mostrar_error, "Error crítico: Módulo integrador no cargado.")
                 return

            # Obtener el valor booleano de manera segura
            compilar_textos_valor = self.compilar_textos.get() if hasattr(self.compilar_textos, 'get') else self.compilar_textos

            # Formatear términos adicionales con sus operadores
            termino_adicional1 = self.termino_adicional1.get().strip()
            termino_adicional2 = self.termino_adicional2.get().strip()
            termino_adicional3 = self.termino_adicional3.get().strip()
            
            # Si hay términos, añadir los operadores correspondientes
            if termino_adicional1:
                operador1 = self.operador_logico1.get()
                termino_adicional1 = f"{operador1} {termino_adicional1}"
            
            if termino_adicional2:
                operador2 = self.operador_logico2.get()
                termino_adicional2 = f"{operador2} {termino_adicional2}"
            
            if termino_adicional3:
                operador3 = self.operador_logico3.get()
                termino_adicional3 = f"{operador3} {termino_adicional3}"

            params = {
                "termino_general": self.termino_general.get().strip(),
                "termino_adicional1": termino_adicional1,
                "termino_adicional2": termino_adicional2,
                "termino_adicional3": termino_adicional3,
                "max_resultados": self.max_resultados.get(),
                "directorio_salida": self.directorio_salida.get().strip(),
                "headless": self.headless, # Valor fijo
                "compilar_textos": compilar_textos_valor, # Ahora puede ser tanto bool como BooleanVar
                "abrir_archivo": self.abrir_archivo.get(),
                "guardar_pdfs": False,
                "compilar_pdfs": self.compilar_pdfs.get(),
                "incluir_8a_epoca": self.incluir_8a_epoca.get(),
                "incluir_7a_epoca": self.incluir_7a_epoca.get(),
                "incluir_6a_epoca": self.incluir_6a_epoca.get(),
                "incluir_5a_epoca": self.incluir_5a_epoca.get(),
                "tipo_tesis": self.tipo_tesis.get(),
                "usar_descarga_directa": self.usar_descarga_directa # Valor fijo
            }
            for key, value in params.items(): self.log(f"Parámetro [{key}]: {value}", "debug")

            resultado = ejecutar_flujo_completo(**params)
            self.root.after(0, self.procesar_resultado, resultado)

        except Exception as e:
            gui_logger.error(f"Error fatal en hilo de búsqueda: {str(e)}", exc_info=True)
            self.root.after(0, self.mostrar_error, f"Error inesperado durante la búsqueda:\n{str(e)}")
            
    # Método auxiliar para obtener valores booleanos de manera segura
    def obtener_valor_booleano(self, attr_name):
        """
        Obtiene el valor booleano de un atributo de manera segura,
        independientemente de si es un bool o un BooleanVar.
        """
        attr = getattr(self, attr_name, None)
        if attr is None:
            return False
        if hasattr(attr, 'get'):
            return attr.get()
        return bool(attr)

    def procesar_resultado(self, resultado):
        """Procesa el resultado de la búsqueda y actualiza la GUI."""
        gui_logger.info("Procesando resultado de la búsqueda...")
        self.busqueda_en_progreso = False

        # Actualizar la barra de progreso vanguardista al 100%
        if hasattr(self, 'barra_progreso'): 
            self.barra_progreso["value"] = 100
        if hasattr(self, 'etiqueta_estado'):
            self.etiqueta_estado.config(text="Estado: Proceso finalizado")
        if hasattr(self, 'etiqueta_porcentaje'):
            self.etiqueta_porcentaje.config(text="100%")
        if hasattr(self, 'etiqueta_fase'):
            self.etiqueta_fase.config(text="Proceso completado correctamente")
            
        # Compatibilidad con código anterior
        if hasattr(self, 'progress'): self.progress["value"] = 100
        if hasattr(self, 'progress_label'): self.progress_label.config(text="Estado: Proceso finalizado")
        
        # Actualizar estado de los botones de manera segura
        try:
            # Rehabilitar los botones principales
            if hasattr(self, 'btn_extraer'): 
                self.btn_extraer.config(state=tk.NORMAL)
            if hasattr(self, 'btn_limpiar'):
                self.btn_limpiar.config(state=tk.NORMAL)
            if hasattr(self, 'btn_ayuda'):
                self.btn_ayuda.config(state=tk.NORMAL)
            if hasattr(self, 'btn_info'):
                self.btn_info.config(state=tk.NORMAL)
                
            # Desactivar botón detener
            if hasattr(self, 'btn_detener'): 
                self.btn_detener.config(state=tk.DISABLED)
        except Exception as e:
            gui_logger.warning(f"Advertencia: No se pudieron actualizar algunos botones: {str(e)}")

        if resultado and resultado.get("fase_extraccion", {}).get("completado", False):
            exitos = resultado["fase_extraccion"].get("exitos", 0)
            total_urls = resultado["fase_extraccion"].get("total", 0)
            archivo_compilado_texto = resultado["fase_extraccion"].get("archivo_compilado_texto")
            archivo_compilado_pdf = resultado["fase_extraccion"].get("archivo_compilado_pdf")
            archivo_json = resultado.get("archivo_json")
            metodo_usado = resultado.get("fase_extraccion", {}).get("metodo_usado", "tradicional")
            es_descarga_directa = "descarga_directa" in metodo_usado.lower()

            # Habilitar botón de abrir compilado si se generó algún archivo
            if archivo_compilado_texto or archivo_compilado_pdf:
                try:
                    if hasattr(self, 'btn_abrir'): self.btn_abrir.config(state=tk.NORMAL)
                except Exception as e:
                    gui_logger.warning(f"Advertencia: No se pudo actualizar el botón abrir: {str(e)}")

            tag_success = "success" # Usar tag para consistencia
            self.log(f"--- BÚSQUEDA COMPLETADA ({'Optimizado' if es_descarga_directa else 'Tradicional'}) ---", tag_success)
            self.log(f"Textos extraídos: {exitos} de {total_urls} URLs procesadas", "info")

            status_msg = f"Estado: Completado ({exitos}/{total_urls} textos)"
            if archivo_compilado_texto:
                 self.log(f"Archivo Texto Compilado: {os.path.basename(archivo_compilado_texto)}", "info")
                 status_msg += " [TXT]"
            if archivo_compilado_pdf:
                 self.log(f"Archivo PDF Compilado: {os.path.basename(archivo_compilado_pdf)}", "info")
                 status_msg += " [PDF]"
            self.status_var.set(status_msg)

            if archivo_json and os.path.exists(archivo_json):
                 self.cargar_resultados_en_treeview(archivo_json)
            else:
                 self.log("Archivo JSON de resultados no encontrado o no generado.", "warning")

            mensaje_final = f"Proceso completado.\n\nTextos extraídos: {exitos} de {total_urls}"
            if es_descarga_directa: mensaje_final += "\n(Se utilizó descarga directa optimizada)"
            if archivo_compilado_texto: mensaje_final += f"\n\nArchivo de Texto: {os.path.basename(archivo_compilado_texto)}"
            if archivo_compilado_pdf: mensaje_final += f"\nArchivo PDF: {os.path.basename(archivo_compilado_pdf)}"
            messagebox.showinfo("Proceso Finalizado", mensaje_final, parent=self.root)

            # ELIMINADO: Bloque que abría los archivos manualmente
            # La apertura de archivos ahora es responsabilidad exclusiva del módulo integrador_tesis_scjn.txt
            # if self.abrir_archivo.get():
            #     if archivo_compilado_texto and os.path.exists(archivo_compilado_texto): self.abrir_archivo_os(archivo_compilado_texto)
            #     if archivo_compilado_pdf and os.path.exists(archivo_compilado_pdf): self.abrir_archivo_os(archivo_compilado_pdf)
        else:
            error_msg = resultado.get("error", "Error desconocido") if resultado else "Proceso no devolvió resultado."
            self.mostrar_error(error_msg)


    def cargar_resultados_en_treeview(self, archivo_json):
        """Carga resultados desde archivo JSON al Treeview."""
        gui_logger.info(f"Cargando resultados en Treeview desde: {archivo_json}")
        if not hasattr(self, 'resultados_tree'): return

        try:
            with open(archivo_json, 'r', encoding='utf-8') as f: datos = json.load(f)

            for item in self.resultados_tree.get_children(): self.resultados_tree.delete(item)

            resultados = datos.get('resultados', [])
            self.resultados_completos = resultados
            self.total_resultados_real = len(resultados)
            count_datos_oficiales = 0

            for i, res in enumerate(resultados):
                registro = res.get('registro_digital', 'N/D')
                num_tesis = res.get('numero_tesis', 'N/D')
                rubro = res.get('rubro', 'N/D')
                rubro_corto = (rubro[:100] + '...') if len(rubro) > 103 else rubro
                es_oficial = res.get('metodo_obtencion') == 'descarga_directa'
                # Determinar tag para fila alterna o datos oficiales
                tag = 'datos_oficiales' if es_oficial else ('evenrow' if i % 2 == 0 else 'oddrow')
                if es_oficial: count_datos_oficiales += 1
                # Añadir el número de fila como primer valor
                numero = i + 1
                self.resultados_tree.insert('', 'end', iid=f"item_{i}", values=(numero, registro, num_tesis, rubro_corto), tags=(tag,))

            if hasattr(self, 'resultados_frame'):
                nuevo_titulo = f"Resultados Preliminares ({self.total_resultados_real} Tesis"
                if count_datos_oficiales > 0: nuevo_titulo += f", {count_datos_oficiales} Oficiales"
                nuevo_titulo += ")"
                self.resultados_frame.config(text=nuevo_titulo)

            self.log(f"Tabla actualizada con {self.total_resultados_real} resultados.", "info")

        except FileNotFoundError:
             self.log(f"Error: Archivo JSON no encontrado en {archivo_json}", "error")
             messagebox.showerror("Error de Archivo", f"No se encontró el archivo:\n{archivo_json}", parent=self.root)
        except json.JSONDecodeError:
             self.log(f"Error: Formato JSON inválido en {archivo_json}", "error")
             messagebox.showerror("Error de Formato", f"El archivo JSON parece estar corrupto:\n{archivo_json}", parent=self.root)
        except Exception as e:
            gui_logger.error(f"Error inesperado al cargar Treeview: {str(e)}", exc_info=True)
            self.log(f"Error al cargar tabla de resultados: {str(e)}", "error")

    def mostrar_error(self, mensaje):
        """Muestra mensaje de error y actualiza estado."""
        gui_logger.error(f"Error presentado al usuario: {mensaje}")
        self.busqueda_en_progreso = False
        
        # Actualizar barras de progreso
        if hasattr(self, 'barra_progreso'): 
            self.barra_progreso["value"] = 0
        if hasattr(self, 'progress'): 
            self.progress["value"] = 0
            
        # Actualizar textos de estado
        if hasattr(self, 'etiqueta_estado'):
            self.etiqueta_estado.config(text="Estado: Error")
        if hasattr(self, 'etiqueta_fase'):
            self.etiqueta_fase.config(text=f"ERROR: {mensaje}")
        if hasattr(self, 'progress_label'): 
            self.progress_label.config(text="Estado: Error")
        
        # Actualizar botones de manera segura
        try:
            # Rehabilitar los botones principales
            if hasattr(self, 'btn_extraer'): 
                self.btn_extraer.config(state=tk.NORMAL)
            if hasattr(self, 'btn_limpiar'):
                self.btn_limpiar.config(state=tk.NORMAL)
            if hasattr(self, 'btn_ayuda'):
                self.btn_ayuda.config(state=tk.NORMAL)
            if hasattr(self, 'btn_info'):
                self.btn_info.config(state=tk.NORMAL)
                
            # Desactivar botón detener
            if hasattr(self, 'btn_detener'): 
                self.btn_detener.config(state=tk.DISABLED)
        except Exception as e:
            gui_logger.warning(f"Advertencia: No se pudieron actualizar algunos botones: {str(e)}")
        
        self.log(f"ERROR: {mensaje}", "error")
        self.status_var.set("Estado: Error durante la ejecución")
        messagebox.showerror("Error en Ejecución", mensaje, parent=self.root)

    def abrir_archivo_os(self, ruta_archivo):
        """
        Abre un archivo usando el visor predeterminado del sistema operativo.
        Utiliza la función centralizada si está disponible.
        """
        if utils_importados:
            # Usar la función centralizada
            return abrir_archivo_os(ruta_archivo, self.log)
        else:
            # Implementación original
            try:
                gui_logger.info(f"Intentando abrir archivo: {ruta_archivo}")
                if sys.platform.startswith('win'): os.startfile(ruta_archivo)
                elif sys.platform.startswith('darwin'): subprocess.run(['open', ruta_archivo], check=True)
                else: subprocess.run(['xdg-open', ruta_archivo], check=True)
                self.log(f"Archivo abierto: {os.path.basename(ruta_archivo)}", "info")
                return True
            except FileNotFoundError:
                 self.log(f"Error: Archivo no encontrado para abrir: {ruta_archivo}", "error")
                 messagebox.showerror("Error", f"No se encontró el archivo:\n{ruta_archivo}", parent=self.root)
                 return False
            except Exception as e:
                gui_logger.error(f"Error al abrir archivo '{ruta_archivo}': {str(e)}", exc_info=True)
                self.log(f"Error al abrir archivo '{os.path.basename(ruta_archivo)}': {str(e)}", "error")
                messagebox.showerror("Error", f"No se pudo abrir el archivo:\n{os.path.basename(ruta_archivo)}\n\nError: {str(e)}", parent=self.root)
                return False

    def mostrar_errores_importacion(self):
        """Muestra un mensaje consolidado si hubo errores al importar módulos."""
        if error_importacion:
             mensaje = "Ocurrieron errores al cargar componentes necesarios:\n\n" + "\n".join(f"- {err}" for err in error_importacion) + "\n\nLa aplicación podría no funcionar correctamente."
             messagebox.showerror("Error de Carga de Módulos", mensaje, parent=self.root)

    def mostrar_ayuda(self):
        """Muestra una ventana de ayuda sobre el uso de la aplicación."""
        ayuda_window = tk.Toplevel(self.root)
        ayuda_window.title("Ayuda del Integrador de Jurisprudencia")
        ayuda_window.geometry("800x600")
        ayuda_window.minsize(700, 500)
        ayuda_window.configure(background=self.color_fondo_claro)
        
        # Contenedor principal con scroll
        main_container = ttk.Frame(ayuda_window, style="Light.TFrame")
        main_container.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        # Título
        ttk.Label(main_container, text="Guía de Usuario", style="Campo.TLabel", font=self.font_titulo).pack(anchor=tk.W, pady=(0, 20))
        
        # Contenido de ayuda
        texto_ayuda = """
1. CONSULTA BÁSICA:
   • Ingrese el término general obligatorio.
   • Opcionalmente, agregue hasta tres términos adicionales para refinar la búsqueda.
   • Todos los términos se combinan con operador AND (Y lógico).

2. SINTAXIS DE BÚSQUEDA:
   • Para frases exactas: Use comillas dobles ("ejemplo de frase").
   • Para palabras individuales: Simplemente sepárelas con espacios (palabra1 palabra2).
   • Se puede combinar ambas sintaxis ("frase exacta" palabra_clave).

3. FILTROS:
   • Tipo de Tesis: Seleccione entre Todas, Jurisprudencia o Aislada.
   • Épocas: Por defecto siempre se incluyen 11ª, 10ª y 9ª épocas.
   • Marque las casillas para incluir épocas anteriores.

4. OPCIONES DE SALIDA:
   • Compilar PDF: Genera un PDF consolidado con todas las tesis encontradas.
   • Abrir archivos: Abre automáticamente los archivos generados al finalizar.

5. RESULTADOS:
   • La tabla muestra preliminarmente las tesis encontradas.
   • Haga doble clic en cualquier fila para ver el detalle completo.
   • El sistema identifica con fondo azul claro las tesis obtenidas directamente de SCJN.

6. REGISTRO DE ACTIVIDAD:
   • Muestra en tiempo real el progreso y estado de la extracción.
   • Diferentes colores indican información, advertencias o errores.

NOTAS IMPORTANTES:
• La búsqueda es sensible a acentos.
• Por defecto, se utiliza descarga directa optimizada que mejora precisión y rendimiento.
• Los resultados se guardan en formato texto (.txt) y opcionalmente en PDF.
• Se genera un archivo JSON con los metadatos para referencia futura.

Para más información técnica o soporte, use el botón "Info Extractor".
"""
        
        texto_widget = ScrolledText(main_container, wrap=tk.WORD, height=25, width=80, font=self.font_normal)
        texto_widget.pack(fill=tk.BOTH, expand=True)
        texto_widget.insert(tk.END, texto_ayuda)
        texto_widget.config(state=tk.DISABLED)
        
        # Botón para cerrar
        ttk.Button(main_container, text="Cerrar", command=ayuda_window.destroy, style="Primario.TButton", width=15).pack(pady=15)
        
        # Centrar ventana
        ayuda_window.update_idletasks()
        w, h = ayuda_window.winfo_width(), ayuda_window.winfo_height()
        ws, hs = ayuda_window.winfo_screenwidth(), ayuda_window.winfo_screenheight()
        x, y = max(0, (ws/2) - (w/2)), max(0, (hs/2) - (h/2))
        ayuda_window.geometry('%dx%d+%d+%d' % (w, h, x, y))
        ayuda_window.transient(self.root)
        ayuda_window.grab_set()
        
        gui_logger.info("Ventana de ayuda mostrada")

    def mostrar_info_extractor(self):
        """Muestra información técnica sobre el extractor y sus capacidades."""
        info_window = tk.Toplevel(self.root)
        info_window.title("Información del Extractor")
        info_window.geometry("800x600")
        info_window.minsize(700, 500)
        info_window.configure(background=self.color_fondo_oscuro)
        
        # Contenedor principal con scroll y fondo oscuro
        main_container = ttk.Frame(info_window, style="Dark.TFrame")
        main_container.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        # Título con estilo oscuro
        ttk.Label(main_container, text="Información Técnica del Extractor", style="Dark.Campo.TLabel", font=self.font_titulo).pack(anchor=tk.W, pady=(0, 20))
        
        # Contenido técnico
        texto_info = """
ESPECIFICACIONES TÉCNICAS
==========================

ARQUITECTURA DEL SISTEMA:
• Motor modular basado en Python 3.7+
• Procesamiento asíncrono optimizado
• Gestión de memoria adaptativa para grandes volúmenes

CAPACIDADES:
• Extracción simultánea de múltiples fuentes
• Descarga directa desde API SCJN
• Consulta optimizada con tokenización avanzada
• Compilación multiformato (TXT, PDF, JSON)
• Procesamiento UTF-8 para correcta visualización de acentos

RENDIMIENTO:
• Velocidad promedio: 2-5 tesis/segundo (con descarga directa)
• Límite recomendado: 250 tesis por consulta
• Concurrencia máxima: 8 conexiones

DEPENDENCIAS:
• Integración con integrador_tesis_scjn.py
• ScraperTesisCompuesto (versión 7+)
• Tkinter para interfaz gráfica
• SQLite para caché temporal

MÉTODOS DE EXTRACCIÓN:
1. DIRECTA: Conexión API optimizada (predeterminado)
   - Mayor precisión
   - Metadatos completos
   - Velocidad superior

2. TRADICIONAL: Scraping adaptativo (fallback)
   - Compatible con todas las épocas
   - Menor dependencia de estructura SCJN
   - Velocidad reducida

NOTAS PARA DESARROLLADORES:
• El código fuente está diseñado con patrones de diseño Factory y Adapter
• El logger registra información detallada para debugging
• Posibilidad de extender mediante plugins personalizados
• Preparado para sistemas Windows, MacOS y Linux

FECHA DE COMPILACIÓN: Marzo 2024
VERSIÓN: 1.2.5
"""
        
        # Usar colores de fondo oscuro para el ScrolledText
        texto_widget = ScrolledText(
            main_container, 
            wrap=tk.WORD,
            height=25, 
            width=80, 
            font=self.font_normal,
            background=self.color_fondo_oscuro,
            foreground=self.color_texto_claro,
            insertbackground=self.color_texto_claro,
            selectbackground=self.color_acento,
            selectforeground=self.color_texto_claro
        )
        texto_widget.pack(fill=tk.BOTH, expand=True)
        texto_widget.insert(tk.END, texto_info)
        texto_widget.config(state=tk.DISABLED)
        
        # Botón para cerrar
        ttk.Button(main_container, text="Cerrar", command=info_window.destroy, style="Accion.TButton", width=15).pack(pady=15)
        
        # Centrar ventana
        info_window.update_idletasks()
        w, h = info_window.winfo_width(), info_window.winfo_height()
        ws, hs = info_window.winfo_screenwidth(), info_window.winfo_screenheight()
        x, y = max(0, (ws/2) - (w/2)), max(0, (hs/2) - (h/2))
        info_window.geometry('%dx%d+%d+%d' % (w, h, x, y))
        info_window.transient(self.root)
        info_window.grab_set()
        
        gui_logger.info("Ventana de información técnica mostrada")

    def generar_reporte_preliminar(self):
        """Genera un reporte en formato TXT o JSON de los resultados preliminares en la tabla."""
        if not self.resultados_completos:
            messagebox.showinfo("Información", "No hay resultados para generar un reporte.", parent=self.root)
            return
            
        directorio = self.directorio_salida.get().strip()
        if not os.path.exists(directorio):
            try:
                os.makedirs(directorio)
            except Exception as e:
                messagebox.showerror("Error", f"No se pudo crear el directorio de salida:\n{str(e)}", parent=self.root)
                return
        
        timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
        nombre_archivo = os.path.join(directorio, f"reporte_preliminar_{timestamp}.txt")
        
        try:
            with open(nombre_archivo, 'w', encoding='utf-8') as f:
                f.write("REPORTE PRELIMINAR DE RESULTADOS\n")
                f.write("===============================\n\n")
                f.write(f"Fecha y hora: {datetime.now().strftime('%d/%m/%Y %H:%M:%S')}\n")
                f.write(f"Total de resultados: {self.total_resultados_real}\n\n")
                
                f.write("LISTADO DE TESIS\n")
                f.write("--------------\n\n")
                
                for i, tesis in enumerate(self.resultados_completos):
                    f.write(f"[{i+1}] Registro: {tesis.get('registro_digital', 'N/D')}\n")
                    f.write(f"    No. Tesis: {tesis.get('numero_tesis', 'N/D')}\n")
                    f.write(f"    Rubro: {tesis.get('rubro', 'N/D')}\n")
                    f.write(f"    Tipo: {tesis.get('tipo_tesis', 'N/D')}\n")
                    f.write(f"    Época: {tesis.get('epoca', 'N/D')}\n")
                    f.write(f"    Instancia: {tesis.get('instancia', 'N/D')}\n")
                    f.write("\n")
            
            # También crear versión JSON para usos más avanzados
            nombre_json = os.path.join(directorio, f"reporte_preliminar_{timestamp}.json")
            with open(nombre_json, 'w', encoding='utf-8') as f:
                reporte_data = {
                    "fecha_generacion": datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                    "total_resultados": self.total_resultados_real,
                    "resultados": self.resultados_completos
                }
                json.dump(reporte_data, f, ensure_ascii=False, indent=2)
            
            self.log(f"Reporte preliminar generado: {os.path.basename(nombre_archivo)}", "success")
            self.log(f"Versión JSON guardada: {os.path.basename(nombre_json)}", "info")
            
            # Preguntar si desea abrir el reporte
            if messagebox.askyesno("Reporte Generado", 
                                  f"Reporte generado exitosamente en:\n{nombre_archivo}\n\n¿Desea abrirlo ahora?", 
                                  parent=self.root):
                self.abrir_archivo_os(nombre_archivo)
                
        except Exception as e:
            error_msg = f"Error al generar reporte: {str(e)}"
            gui_logger.error(error_msg, exc_info=True)
            self.log(error_msg, "error")
            messagebox.showerror("Error", f"No se pudo generar el reporte:\n{str(e)}", parent=self.root)

    def mostrar_aviso_legal(self):
        """Muestra una ventana con el aviso legal de la aplicación."""
        aviso_window = tk.Toplevel(self.root)
        aviso_window.title("Aviso Legal de Cumplimiento")
        aviso_window.geometry("800x600")
        aviso_window.minsize(700, 500)
        aviso_window.configure(background=self.color_fondo_claro)
        
        # Contenedor principal con scroll
        main_container = ttk.Frame(aviso_window, style="Light.TFrame")
        main_container.pack(fill=tk.BOTH, expand=True, padx=20, pady=20)
        
        # Título
        ttk.Label(main_container, text="Aviso Legal de Cumplimiento", style="Campo.TLabel", font=self.font_titulo).pack(anchor=tk.W, pady=(0, 20))
        
        # Contenido del aviso legal
        texto_aviso = """
Esta aplicación respeta y se alinea con lo establecido en el Reglamento de la Suprema Corte de Justicia de la Nación y del Consejo de la Judicatura Federal para la aplicación de la Ley Federal de Transparencia y Acceso a la Información Pública Gubernamental.

De acuerdo con el artículo 5 de dicho Reglamento, "Es pública la información que tienen bajo su resguardo la Suprema Corte, el Consejo y los Órganos Jurisdiccionales, con las salvedades establecidas en la Ley".

Esta herramienta facilita la consulta de información que, conforme al artículo 7 del citado Reglamento, "Las sentencias ejecutorias y las demás resoluciones públicas podrán consultarse una vez que se emitan".

El desarrollo de esta aplicación contribuye al principio constitucional del derecho a la información y a la evolución jurisprudencial que ha determinado que "el derecho a la información obliga al Estado no solamente a informar sino a asegurar que todo individuo sea enterado de algún suceso de carácter público y de interés general".

La consulta de tesis y jurisprudencias se realiza en cumplimiento de las disposiciones sobre transparencia judicial, democratizando así el acceso a la justicia y contribuyendo a hacer efectivo el derecho fundamental de acceso a la información pública.
"""
        
        texto_widget = ScrolledText(main_container, wrap=tk.WORD, height=25, width=80, font=self.font_normal)
        texto_widget.pack(fill=tk.BOTH, expand=True)
        texto_widget.insert(tk.END, texto_aviso)
        texto_widget.config(state=tk.DISABLED)
        
        # Botón para cerrar
        ttk.Button(main_container, text="Cerrar", command=aviso_window.destroy, style="Primario.TButton", width=15).pack(pady=15)
        
        # Centrar ventana
        aviso_window.update_idletasks()
        w, h = aviso_window.winfo_width(), aviso_window.winfo_height()
        ws, hs = aviso_window.winfo_screenwidth(), aviso_window.winfo_screenheight()
        x, y = max(0, (ws/2) - (w/2)), max(0, (hs/2) - (h/2))
        aviso_window.geometry('%dx%d+%d+%d' % (w, h, x, y))
        aviso_window.transient(self.root)
        aviso_window.grab_set()
        
        gui_logger.info("Ventana de aviso legal mostrada")

# Función para iniciar la aplicación rediseñada
def iniciar_app_rediseñada():
    root = tk.Tk()
    app = IntegradorTesisGUIRediseñado(root)
    
    # Centrar ventana usando la función centralizada si está disponible
    if utils_importados:
        centrar_ventana(root, logger=gui_logger)
    
    root.mainloop()

if __name__ == "__main__":
    if not INTEGRADOR_IMPORTADO:
         print("ERROR CRÍTICO: No se pudo cargar 'integrador_tesis_scjn'. La aplicación no puede continuar.")
         try:
             root_temp = tk.Tk(); root_temp.withdraw()
             messagebox.showerror("Error Fatal", "Componente principal ('integrador_tesis_scjn') no encontrado.\nLa aplicación no puede iniciarse.\n\nAsegúrese de que todos los archivos .py estén juntos.", parent=None)
             root_temp.destroy()
         except Exception as tk_error: print(f"Error al mostrar mensaje Tkinter: {tk_error}")
         sys.exit(1)
    else:
        iniciar_app_rediseñada()