"""
Extractor de Precedentes de Jurisprudencias SCJN - VERSIÓN MEJORADA CON RECONOCIMIENTO VISUAL

Este módulo implementa la funcionalidad para:
1. Identificar jurisprudencias a partir de resultados de búsqueda
2. Extraer números de precedentes de las jurisprudencias usando reconocimiento visual
3. Acceder a la URL de cada precedente
4. Extraer el texto completo de la ejecutoria
5. Generar archivos TXT organizados con el contenido

Incorpora reconocimiento de imágenes para localizar el botón "P" de forma más robusta.
"""

import os
import sys
import time
import json
import logging
import traceback
import re
from datetime import datetime
import tempfile
import argparse
import threading
import random
import tkinter as tk
from tkinter import ttk, filedialog, messagebox, scrolledtext
from selenium import webdriver
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import (
    TimeoutException, NoSuchElementException, 
    StaleElementReferenceException, ElementClickInterceptedException,
    ElementNotInteractableException, WebDriverException
)
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.common.action_chains import ActionChains
from urllib.parse import urlparse, urlunparse

# Nuevas importaciones para reconocimiento visual
try:
    import cv2
    import numpy as np
    OPENCV_DISPONIBLE = True
except ImportError:
    OPENCV_DISPONIBLE = False
    print("ADVERTENCIA: OpenCV no está instalado. El reconocimiento visual no estará disponible.")
    print("Para instalar: pip install opencv-python numpy")

# --------------------------------------------------------
# Configuración de logging avanzado
# --------------------------------------------------------

# Configurar logging
LOG_FORMAT = '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
LOG_FILE = "extractor_precedentes_avanzado.log"

# Crear directorio de logs si no existe
os.makedirs("logs", exist_ok=True)

# Configurar logger
logging.basicConfig(
    level=logging.INFO,
    format=LOG_FORMAT,
    handlers=[
        logging.FileHandler(os.path.join("logs", LOG_FILE)),
        logging.StreamHandler()
    ]
)

# Crear logger para este módulo
logger = logging.getLogger("ExtractorPrecedentes")

# --------------------------------------------------------
# Verificar disponibilidad de los módulos requeridos
# --------------------------------------------------------

# Determinar si estamos ejecutando como script o como ejecutable
if getattr(sys, 'frozen', False):
    # Si es ejecutable, usar sys._MEIPASS para encontrar recursos
    base_path = sys._MEIPASS
else:
    # Si es script, usar el directorio actual
    base_path = os.path.dirname(os.path.abspath(__file__))

# Agregar la ruta base al sys.path para que pueda encontrar los módulos
sys.path.insert(0, base_path)

# Intentar importar módulos opcionales de soporte
try:
    # Intentar importar el scraper de tesis
    from scraper_compuesto7 import ScraperTesisCompuesto
    SCRAPER_IMPORTADO = True
    logger.info("Módulo scraper_compuesto7 importado correctamente")
except ImportError as e:
    logger.warning(f"Módulo scraper_compuesto7 no encontrado. Usando implementación interna: {str(e)}")
    SCRAPER_IMPORTADO = False

try:
    # Intentar importar el integrador de tesis
    from integrador_tesis_scjn import ejecutar_flujo_completo
    INTEGRADOR_IMPORTADO = True
    logger.info("Módulo integrador_tesis_scjn importado correctamente")
except ImportError as e:
    logger.warning(f"Módulo integrador_tesis_scjn no encontrado: {str(e)}")
    INTEGRADOR_IMPORTADO = False

# --------------------------------------------------------
# Clase de soporte para sitios dinámicos (Angular/React)
# --------------------------------------------------------

class EsperaDinamica:
    """
    Utilidades para esperar condiciones en sitios dinámicos basados en
    frameworks como Angular, React, etc.
    """
    
    @staticmethod
    def esperar_angular_quieto(driver, timeout=30):
        """
        Espera a que Angular termine de renderizar.
        
        Args:
            driver: Instancia de Selenium WebDriver
            timeout: Tiempo máximo de espera en segundos
            
        Returns:
            bool: True si Angular terminó, False si expiró el tiempo
        """
        script = """
        try {
            if (window.angular) {
                // Angular 1
                return window.angular.element(document).injector().get('$http').pendingRequests.length === 0;
            } else if (window.getAllAngularTestabilities) {
                // Angular 2+
                const testabilities = window.getAllAngularTestabilities();
                let stable = true;
                
                for (let i = 0; i < testabilities.length; i++) {
                    stable = stable && testabilities[i].isStable();
                }
                
                return stable;
            }
            // No se detecta Angular, asumir estable
            return true;
        } catch (e) {
            // Error accediendo a Angular, asumir estable
            return true;
        }
        """
        
        try:
            start_time = time.time()
            while time.time() - start_time < timeout:
                result = driver.execute_script(script)
                if result:
                    time.sleep(0.5)  # Pequeña espera adicional por estabilidad
                    return True
                time.sleep(0.5)
            
            logger.warning("Tiempo de espera de Angular excedido")
            return False
        except Exception as e:
            logger.warning(f"Error al verificar estabilidad de Angular: {str(e)}")
            return False
    
    @staticmethod
    def esperar_carga_completa(driver, timeout=30):
        """
        Espera a que el documento esté completamente cargado.
        
        Args:
            driver: Instancia de Selenium WebDriver
            timeout: Tiempo máximo de espera en segundos
            
        Returns:
            bool: True si el documento está listo, False si expiró el tiempo
        """
        script = """
        return document.readyState === 'complete' && 
               (typeof jQuery === 'undefined' || jQuery.active === 0);
        """
        
        try:
            start_time = time.time()
            while time.time() - start_time < timeout:
                result = driver.execute_script(script)
                if result:
                    return True
                time.sleep(0.5)
            
            logger.warning("Tiempo de espera de carga del documento excedido")
            return False
        except Exception as e:
            logger.warning(f"Error al verificar carga del documento: {str(e)}")
            return False
    
    @staticmethod
    def esperar_elemento_interactuable(driver, selector, tipo=By.CSS_SELECTOR, timeout=30):
        """
        Espera a que un elemento sea visible e interactuable.
        
        Args:
            driver: Instancia de Selenium WebDriver
            selector: Selector del elemento
            tipo: Tipo de selector (By.CSS_SELECTOR, By.XPATH, etc.)
            timeout: Tiempo máximo de espera en segundos
            
        Returns:
            WebElement: El elemento si está listo, None si expiró el tiempo
        """
        try:
            # Primero esperar que exista y sea visible
            elemento = WebDriverWait(driver, timeout).until(
                EC.visibility_of_element_located((tipo, selector))
            )
            
            # Luego verificar que sea interactuable
            WebDriverWait(driver, timeout).until(
                EC.element_to_be_clickable((tipo, selector))
            )
            
            # Verificación adicional con JavaScript
            script = """
            var element = arguments[0];
            var rect = element.getBoundingClientRect();
            return {
                isVisible: rect.width > 0 && rect.height > 0,
                isInteractable: !element.disabled && 
                              element.getAttribute('aria-disabled') !== 'true',
                position: {
                    top: rect.top,
                    left: rect.left,
                    bottom: window.innerHeight - rect.bottom,
                    right: window.innerWidth - rect.right
                }
            };
            """
            
            info = driver.execute_script(script, elemento)
            if info['isVisible'] and info['isInteractable']:
                # Elemento listo para interactuar
                # Scroll al elemento para asegurar visibilidad completa
                driver.execute_script("arguments[0].scrollIntoView({block: 'center'});", elemento)
                time.sleep(0.5)  # Pequeña espera después del scroll
                return elemento
            
            logger.warning(f"Elemento {selector} no interactuable: {info}")
            return None
        
        except (TimeoutException, StaleElementReferenceException) as e:
            logger.warning(f"Tiempo de espera excedido para el elemento {selector}: {str(e)}")
            return None
        except Exception as e:
            logger.error(f"Error al esperar elemento {selector}: {str(e)}")
            return None
    
    @staticmethod
    def hacer_clic_seguro(driver, elemento, intentos=3, usar_js=False):
        """
        Realiza un clic en un elemento con manejo de errores y alternativas.
        
        Args:
            driver: Instancia de Selenium WebDriver
            elemento: Elemento WebElement a hacer clic
            intentos: Número de intentos antes de fallar
            usar_js: Si se usa JavaScript para el clic inicial
            
        Returns:
            bool: True si el clic fue exitoso, False en caso contrario
        """
        for intento in range(intentos):
            try:
                # Asegurar que el elemento es interactuable
                if not elemento.is_displayed() or not elemento.is_enabled():
                    driver.execute_script("arguments[0].scrollIntoView({block: 'center'});", elemento)
                    time.sleep(0.5)
                
                if usar_js:
                    # Intentar con JavaScript primero
                    driver.execute_script("arguments[0].click();", elemento)
                    logger.info(f"Clic con JavaScript exitoso (intento {intento+1})")
                else:
                    # Intentar clic normal
                    elemento.click()
                    logger.info(f"Clic normal exitoso (intento {intento+1})")
                
                time.sleep(0.5)  # Pequeña espera después del clic
                return True
            
            except ElementClickInterceptedException:
                logger.warning(f"Clic interceptado (intento {intento+1}), intentando con JavaScript")
                try:
                    driver.execute_script("arguments[0].click();", elemento)
                    logger.info(f"Clic con JavaScript exitoso después de intercepción (intento {intento+1})")
                    time.sleep(0.5)
                    return True
                except Exception as e2:
                    logger.warning(f"Error en clic con JavaScript: {str(e2)}")
            
            except (StaleElementReferenceException, ElementNotInteractableException):
                logger.warning(f"Elemento obsoleto o no interactuable (intento {intento+1})")
                time.sleep(1)  # Esperar un poco más para que el DOM se estabilice
                
                try:
                    # Último recurso: clic con Actions
                    actions = ActionChains(driver)
                    actions.move_to_element(elemento).click().perform()
                    logger.info(f"Clic con Actions exitoso (intento {intento+1})")
                    time.sleep(0.5)
                    return True
                except Exception as e2:
                    logger.warning(f"Error en clic con Actions: {str(e2)}")
            
            except Exception as e:
                logger.warning(f"Error al hacer clic (intento {intento+1}): {str(e)}")
                
                if intento == intentos - 1:
                    # Último intento desesperado: TAB + ENTER
                    try:
                        elemento.send_keys(Keys.TAB)
                        elemento.send_keys(Keys.ENTER)
                        logger.info("Clic simulado con TAB+ENTER")
                        time.sleep(0.5)
                        return True
                    except:
                        pass
            
            # Esperar entre intentos con tiempo variable para evitar patrones detectables
            time.sleep(0.5 + random.random())
        
        return False
    
    @staticmethod
    def esperar_modal(driver, selector=".mat-dialog-container", tipo=By.CSS_SELECTOR, timeout=10):
        """
        Espera a que aparezca un modal dialog.
        
        Args:
            driver: Instancia de Selenium WebDriver
            selector: Selector del modal
            tipo: Tipo de selector
            timeout: Tiempo máximo de espera en segundos
            
        Returns:
            WebElement: El elemento modal, o None si no aparece
        """
        try:
            return WebDriverWait(driver, timeout).until(
                EC.presence_of_element_located((tipo, selector))
            )
        except TimeoutException:
            logger.warning(f"Modal no encontrado con selector {selector}")
            # Buscar con selectores alternativos
            selectores = [
                (By.CSS_SELECTOR, ".mat-dialog-container"),
                (By.CSS_SELECTOR, "[role='dialog']"),
                (By.CSS_SELECTOR, ".modal"),
                (By.CSS_SELECTOR, ".modal-dialog"),
                (By.XPATH, "//div[contains(@class, 'modal') or @role='dialog']")
            ]
            
            for sel_tipo, sel_valor in selectores:
                try:
                    return driver.find_element(sel_tipo, sel_valor)
                except:
                    pass
            
            return None

# --------------------------------------------------------
# ScraperTesisMinimo - Implementación optimizada
# --------------------------------------------------------

class ScraperTesisMinimo:
    """
    Implementación mínima del scraper para funcionar sin dependencias externas.
    Se usa cuando no está disponible el módulo scraper_compuesto7.
    """
    
    def __init__(self, headless=True, timeout_default=45, modo_debug=False):
        """
        Inicializa un scraper mínimo.
        
        Args:
            headless (bool): Si se usa navegador sin interfaz visual
            timeout_default (int): Tiempo máximo de espera para elementos
            modo_debug (bool): Activar modo de depuración
        """
        self.headless = headless
        self.timeout_default = timeout_default
        self.modo_debug = modo_debug
        self.driver = None
        
        logger.info(f"Inicializando scraper mínimo. Headless: {headless}")
        self._inicializar_driver()
    
    def _inicializar_driver(self):
        """Inicializa el driver de Chrome con opciones optimizadas"""
        try:
            options = Options()
            
            if self.headless:
                options.add_argument('--headless=new')
            
            # Opciones optimizadas para scraping
            options.add_argument('--disable-gpu')
            options.add_argument('--disable-dev-shm-usage')
            options.add_argument('--no-sandbox')
            options.add_argument('--disable-extensions')
            options.add_argument('--disable-notifications')
            options.add_argument('--disable-infobars')
            options.add_argument('--disable-blink-features=AutomationControlled')
            options.add_argument('--window-size=1920,1080')
            options.add_argument('--start-maximized')
            options.add_argument('--ignore-certificate-errors')
            options.add_argument('--disable-web-security')
            
            # OPCIONES PARA EVITAR DETECCIÓN
            options.add_argument(f"user-agent=Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36")
            options.add_argument("--disable-features=IsolateOrigins,site-per-process")
            options.add_argument("--disable-site-isolation-trials")
            
            options.add_experimental_option('excludeSwitches', ['enable-automation'])
            options.add_experimental_option('useAutomationExtension', False)
            
            # Inicializar driver
            # Intentar encontrar el chromedriver en el path o en lugar predeterminado
            try:
                self.driver = webdriver.Chrome(options=options)
            except:
                # Si fallamos, intentar buscar en ubicaciones comunes
                chromedriver_paths = [
                    "./chromedriver",
                    "./chromedriver.exe",
                    "/usr/local/bin/chromedriver",
                    "/usr/bin/chromedriver"
                ]
                
                for path in chromedriver_paths:
                    if os.path.exists(path):
                        service = Service(executable_path=path)
                        self.driver = webdriver.Chrome(service=service, options=options)
                        break
                
                # Si aún no tenemos driver, fallar
                if not self.driver:
                    raise Exception("No se pudo encontrar chromedriver")
            
            # Configurar tiempos de espera
            self.driver.set_page_load_timeout(60)  # Aumentado a 60 segundos
            self.driver.implicitly_wait(15)  # Aumentado a 15 segundos
            
            # Inyectar scripts para ocultar firma de automation
            self.driver.execute_script("""
                Object.defineProperty(navigator, 'webdriver', {
                    get: () => undefined
                });
                
                // Ocultar señales adicionales de automatización
                if (window.navigator.plugins) {
                    Object.defineProperty(navigator, 'plugins', {
                        get: () => [1, 2, 3, 4, 5]
                    });
                }
                
                if (window.navigator.languages) {
                    Object.defineProperty(navigator, 'languages', {
                        get: () => ['es-ES', 'es', 'en-US', 'en']
                    });
                }
            """)
            
            logger.info("Driver de Chrome inicializado correctamente")
        
        except Exception as e:
            logger.error(f"Error al inicializar driver: {str(e)}")
            raise
    
    def navegar_a_url(self, url, max_intentos=5):
        """
        Navega a una URL con reintentos y esperas inteligentes.
        
        Args:
            url (str): URL a visitar
            max_intentos (int): Número máximo de reintentos
            
        Returns:
            bool: True si la navegación fue exitosa
        """
        for intento in range(max_intentos):
            try:
                logger.info(f"Navegando a URL (intento {intento+1}): {url}")
                
                # Espera progresiva entre intentos
                if intento > 0:
                    tiempo_espera = 3 * (intento + 1)
                    logger.info(f"Esperando {tiempo_espera} segundos antes del reintento...")
                    time.sleep(tiempo_espera)
                
                # Limpiar cookies y almacenamiento en reintentos
                if intento > 0:
                    try:
                        self.driver.delete_all_cookies()
                        self.driver.execute_script("window.localStorage.clear();")
                        self.driver.execute_script("window.sessionStorage.clear();")
                        logger.info("Cookies y almacenamiento limpiados")
                    except Exception as e:
                        logger.debug(f"Error al limpiar datos: {str(e)}")
                
                # Simular comportamiento más humano
                time.sleep(random.uniform(2, 4))
                
                # Cargar la URL con timeout ampliado
                self.driver.set_page_load_timeout(60)
                self.driver.get(url)
                
                # Verificación básica antes de esperas complejas
                WebDriverWait(self.driver, 10).until(
                    lambda d: d.execute_script("return document.readyState") == "complete"
                )
                
                # Esperar a que la página cargue completamente
                timeout_carga = 30 + (intento * 10)
                logger.info(f"Esperando carga completa, timeout: {timeout_carga}s")
                
                # Esperar elementos básicos
                WebDriverWait(self.driver, timeout_carga).until(
                    EC.presence_of_element_located((By.TAG_NAME, "body"))
                )
                
                # Esperar a que desaparezcan posibles indicadores de carga
                try:
                    WebDriverWait(self.driver, 5).until_not(
                        EC.presence_of_element_located((By.CSS_SELECTOR, ".loading, .spinner, .loader"))
                    )
                except:
                    pass
                
                # Espera adicional para estabilidad
                time.sleep(3)
                
                # Verificar que la página no muestre error
                page_source = self.driver.page_source.lower()
                if "404" in page_source and "not found" in page_source:
                    logger.warning("La página muestra error 404")
                    if intento < max_intentos - 1:
                        continue
                
                # Verificar si hay contenido real
                if len(self.driver.page_source) < 1000:
                    logger.warning("La página cargada parece incompleta (muy poco contenido)")
                    if intento < max_intentos - 1:
                        continue
                
                logger.info(f"Navegación exitosa a {url}")
                return True
            
            except Exception as e:
                logger.warning(f"Error al navegar a {url} (intento {intento+1}): {str(e)}")
                
                # Intentar refrescar en caso de error
                if intento < max_intentos - 1:
                    try:
                        self.driver.refresh()
                        time.sleep(5)
                    except:
                        pass
        
        logger.error(f"Falló la navegación a {url} después de {max_intentos} intentos")
        return False
    
    def cerrar(self):
        """Cierra el navegador y libera recursos"""
        if self.driver:
            try:
                self.driver.quit()
                logger.info("Navegador cerrado correctamente")
            except Exception as e:
                logger.error(f"Error al cerrar navegador: {str(e)}")
            finally:
                self.driver = None
    
    def generar_url_registro(self, registro_digital):
        """Genera URL para un registro digital de tesis"""
        return f"https://sjf2.scjn.gob.mx/detalle/tesis/{registro_digital}"

# --------------------------------------------------------
# Clase principal del Extractor de Precedentes - Versión con Reconocimiento Visual
# --------------------------------------------------------

class ExtractorPrecedentes:
    """
    Clase principal para extraer precedentes de jurisprudencias.
    Incorpora reconocimiento visual para localizar elementos clave.
    """
    
    def __init__(self, headless=False, timeout_default=45, modo_debug=True,
                directorio_salida="precedentes_extraidos", max_intentos=3):
        """
        Inicializa el extractor de precedentes.
        
        Args:
            headless (bool): Si se usa navegador sin interfaz visual
            timeout_default (int): Tiempo máximo de espera para elementos
            modo_debug (bool): Activar modo de depuración
            directorio_salida (str): Directorio donde guardar precedentes
            max_intentos (int): Número máximo de intentos para operaciones
        """
        self.headless = headless
        self.timeout_default = timeout_default
        self.modo_debug = modo_debug
        self.directorio_salida = directorio_salida
        self.max_intentos = max_intentos
        
        # Verificar reconocimiento visual
        self.reconocimiento_visual = OPENCV_DISPONIBLE
        if not self.reconocimiento_visual:
            logger.warning("OpenCV no disponible. El reconocimiento visual será desactivado.")
            if self.headless:
                logger.warning("Modo headless con reconocimiento visual desactivado puede reducir la efectividad.")
        
        # Rutas de imágenes de referencia
        self.directorio_trabajo = os.path.dirname(os.path.abspath(__file__))
        self.ruta_imagen_boton_p = os.path.join(self.directorio_trabajo, "boton_p_referencia.png")
        
        # Verificar imagen de referencia
        if self.reconocimiento_visual and not os.path.exists(self.ruta_imagen_boton_p):
            logger.warning(f"Imagen de referencia no encontrada: {self.ruta_imagen_boton_p}")
            logger.warning("El sistema pedirá crear esta imagen en la primera ejecución.")
        
        # Crear carpetas necesarias
        if not os.path.exists(self.directorio_salida):
            os.makedirs(self.directorio_salida)
            logger.info(f"Creado directorio de salida: {self.directorio_salida}")
        
        # Carpeta para capturas de debug
        self.debug_dir = os.path.join(self.directorio_salida, "debug")
        if self.modo_debug and not os.path.exists(self.debug_dir):
            os.makedirs(self.debug_dir)
            logger.info(f"Creado directorio de debug: {self.debug_dir}")
        
        # Carpeta para imágenes de referencia
        self.ref_images_dir = os.path.join(self.directorio_trabajo, "ref_images")
        if self.reconocimiento_visual and not os.path.exists(self.ref_images_dir):
            os.makedirs(self.ref_images_dir)
            logger.info(f"Creado directorio de imágenes de referencia: {self.ref_images_dir}")
        
        # Estadísticas del proceso
        self.estadisticas = {
            "tiempo_inicio": time.time(),
            "jurisprudencias_identificadas": 0,
            "precedentes_encontrados": 0,
            "precedentes_extraidos": 0,
            "errores": 0,
            "archivos_generados": []
        }
        
        # Instancia de scraper (se inicializará cuando sea necesario)
        self.scraper = None
        
        logger.info(f"Inicializado extractor de precedentes con reconocimiento visual. Modo headless: {headless}, Debug: {modo_debug}")
    
    def inicializar_scraper(self):
        """
        Inicializa el scraper solo cuando sea necesario.
        Usa ScraperTesisCompuesto si está disponible, sino usa la implementación interna.
        
        Returns:
            bool: True si se inicializó correctamente, False en caso contrario
        """
        try:
            if self.scraper is not None:
                # Verificar que el driver sigue siendo válido
                try:
                    # Si podemos acceder a current_url, el driver está activo
                    _ = self.scraper.driver.current_url
                    logger.info("Reutilizando scraper existente")
                    return True
                except:
                    # Si hay excepción, el driver está cerrado o en mal estado
                    logger.warning("Scraper existente no válido, reinicializando")
                    try:
                        self.cerrar_scraper()
                    except:
                        pass
            
            # Crear nueva instancia del scraper
            if SCRAPER_IMPORTADO:
                logger.info("Usando ScraperTesisCompuesto externo")
                self.scraper = ScraperTesisCompuesto(
                    headless=self.headless,
                    modo_debug=self.modo_debug,
                    timeout_default=self.timeout_default
                )
            else:
                logger.info("Usando implementación interna ScraperTesisMinimo")
                self.scraper = ScraperTesisMinimo(
                    headless=self.headless,
                    modo_debug=self.modo_debug,
                    timeout_default=self.timeout_default
                )
            
            logger.info("Scraper inicializado correctamente")
            return True
        
        except Exception as e:
            logger.error(f"Error al inicializar el scraper: {str(e)}")
            traceback.print_exc()
            return False
    
    def cerrar_scraper(self):
        """Cierra el scraper y libera recursos"""
        if self.scraper is not None:
            try:
                self.scraper.cerrar()
                logger.info("Scraper cerrado correctamente")
            except Exception as e:
                logger.error(f"Error al cerrar el scraper: {str(e)}")
            finally:
                self.scraper = None
    
    def guardar_captura(self, nombre, suffix=""):
        """
        Guarda una captura de pantalla si está en modo debug.
        
        Args:
            nombre (str): Nombre base para la captura
            suffix (str): Sufijo para añadir al nombre
            
        Returns:
            str: Ruta a la captura o None si no se guardó
        """
        if not self.modo_debug or not self.scraper or not self.scraper.driver:
            return None
        
        try:
            # Crear nombre de archivo con timestamp para evitar colisiones
            timestamp = datetime.now().strftime("%H%M%S")
            if suffix:
                nombre_archivo = f"{nombre}_{suffix}_{timestamp}.png"
            else:
                nombre_archivo = f"{nombre}_{timestamp}.png"
            
            ruta_captura = os.path.join(self.debug_dir, nombre_archivo)
            
            # Guardar captura
            self.scraper.driver.save_screenshot(ruta_captura)
            logger.info(f"Captura guardada en: {ruta_captura}")
            
            # Guardar también HTML para diagnóstico avanzado
            html_path = os.path.join(self.debug_dir, f"{nombre}_{suffix}_{timestamp}.html")
            with open(html_path, 'w', encoding='utf-8') as f:
                f.write(self.scraper.driver.page_source)
            
            return ruta_captura
        
        except Exception as e:
            logger.warning(f"Error al guardar captura {nombre}: {str(e)}")
            return None
    
    def preparar_imagen_referencia(self, tipo_elemento="boton_p"):
        """
        Prepara y verifica la imagen de referencia para el reconocimiento visual.
        
        Args:
            tipo_elemento (str): Tipo de elemento para el que se necesita la imagen
            
        Returns:
            bool: True si la imagen está disponible, False en caso contrario
        """
        if not self.reconocimiento_visual:
            return False
        
        # Definir ruta de imagen según el tipo
        if tipo_elemento == "boton_p":
            ruta_imagen = self.ruta_imagen_boton_p
        else:
            ruta_imagen = os.path.join(self.ref_images_dir, f"{tipo_elemento}_referencia.png")
        
        # Verificar si existe la imagen
        if os.path.exists(ruta_imagen):
            logger.info(f"Imagen de referencia para {tipo_elemento} encontrada: {ruta_imagen}")
            return True
        
        # Si no existe, intentar capturarla
        logger.warning(f"Imagen de referencia para {tipo_elemento} no encontrada")
        
        # Guardar una captura completa para que el usuario pueda recortar
        if self.scraper and self.scraper.driver:
            try:
                timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
                ruta_captura = os.path.join(self.debug_dir, f"captura_para_{tipo_elemento}_{timestamp}.png")
                self.scraper.driver.save_screenshot(ruta_captura)
                
                logger.info(f"Se ha guardado una captura en: {ruta_captura}")
                logger.info(f"Por favor, recorte manualmente el {tipo_elemento} y guárdelo como: {ruta_imagen}")
                
                # En modo GUI, mostrar mensaje
                if not self.headless:
                    try:
                        import tkinter as tk
                        from tkinter import messagebox
                        root = tk.Tk()
                        root.withdraw()
                        messagebox.showinfo(
                            "Imagen de Referencia Necesaria", 
                            f"Se requiere una imagen de referencia para el {tipo_elemento}.\n\n"
                            f"1. Se ha guardado una captura en:\n{ruta_captura}\n\n"
                            f"2. Por favor, recorte manualmente el {tipo_elemento}\n"
                            f"3. Guárdelo como:\n{ruta_imagen}\n\n"
                            f"El proceso continuará usando métodos alternativos."
                        )
                        root.destroy()
                    except:
                        pass
            except Exception as e:
                logger.error(f"Error al capturar imagen para referencia: {str(e)}")
        
        return False
    
    def localizar_elemento_visual(self, tipo_elemento="boton_p", umbral=0.7, max_intentos=3):
        """
        Localiza visualmente un elemento en la página actual.
        
        Args:
            tipo_elemento (str): Tipo de elemento a localizar
            umbral (float): Umbral de confianza (0-1)
            max_intentos (int): Número máximo de intentos
            
        Returns:
            dict: Coordenadas del elemento o None si no se encuentra
        """
        if not self.reconocimiento_visual:
            logger.warning("Reconocimiento visual no disponible. OpenCV no está instalado.")
            return None
        
        if not self.scraper or not self.scraper.driver:
            logger.error("No hay un navegador activo para reconocimiento visual")
            return None
        
        # Verificar imagen de referencia
        if tipo_elemento == "boton_p":
            ruta_imagen_ref = self.ruta_imagen_boton_p
        else:
            ruta_imagen_ref = os.path.join(self.ref_images_dir, f"{tipo_elemento}_referencia.png")
        
        if not os.path.exists(ruta_imagen_ref):
            logger.warning(f"Imagen de referencia no encontrada: {ruta_imagen_ref}")
            self.preparar_imagen_referencia(tipo_elemento)
            return None
        
        try:
            logger.info(f"Iniciando reconocimiento visual para: {tipo_elemento}")
            
            # Cargar imagen de referencia
            imagen_referencia = cv2.imread(ruta_imagen_ref)
            if imagen_referencia is None:
                logger.error(f"No se pudo cargar la imagen de referencia: {ruta_imagen_ref}")
                return None
            
            # Convertir a escala de grises
            gray_referencia = cv2.cvtColor(imagen_referencia, cv2.COLOR_BGR2GRAY)
            
            for intento in range(max_intentos):
                try:
                    # Capturar pantalla actual
                    with tempfile.NamedTemporaryFile(suffix='.png', delete=False) as temp:
                        temp_path = temp.name
                    
                    self.scraper.driver.save_screenshot(temp_path)
                    
                    # Cargar captura
                    captura = cv2.imread(temp_path)
                    if captura is None:
                        logger.error(f"No se pudo cargar la captura: {temp_path}")
                        continue
                    
                    # Convertir a escala de grises
                    gray_captura = cv2.cvtColor(captura, cv2.COLOR_BGR2GRAY)
                    
                    # Buscar coincidencia
                    result = cv2.matchTemplate(gray_captura, gray_referencia, cv2.TM_CCOEFF_NORMED)
                    min_val, max_val, min_loc, max_loc = cv2.minMaxLoc(result)
                    
                    logger.info(f"Confianza de reconocimiento: {max_val:.4f} (umbral: {umbral})")
                    
                    # Si la confianza supera el umbral
                    if max_val >= umbral:
                        # Obtener dimensiones
                        h, w = gray_referencia.shape
                        
                        # Calcular coordenadas
                        top_left = max_loc
                        bottom_right = (top_left[0] + w, top_left[1] + h)
                        center = (top_left[0] + w//2, top_left[1] + h//2)
                        
                        # Guardar imagen con la detección si estamos en modo debug
                        if self.modo_debug:
                            # Dibujar rectángulo en la imagen
                            imagen_resultado = captura.copy()
                            cv2.rectangle(imagen_resultado, top_left, bottom_right, (0, 255, 0), 2)
                            
                            # Guardar imagen
                            timestamp = datetime.now().strftime("%H%M%S")
                            ruta_resultado = os.path.join(self.debug_dir, f"deteccion_{tipo_elemento}_{timestamp}.png")
                            cv2.imwrite(ruta_resultado, imagen_resultado)
                            logger.info(f"Detección guardada en: {ruta_resultado}")
                        
                        # Limpiar archivo temporal
                        try:
                            os.unlink(temp_path)
                        except:
                            pass
                        
                        logger.info(f"Elemento {tipo_elemento} localizado en: {center}")
                        return {
                            "confianza": max_val,
                            "top_left": top_left,
                            "bottom_right": bottom_right,
                            "center": center,
                            "width": w,
                            "height": h
                        }
                    else:
                        logger.warning(f"No se encontró {tipo_elemento} con suficiente confianza (intento {intento+1})")
                        
                        # Si estamos en modo debug, guardar la captura actual
                        if self.modo_debug:
                            timestamp = datetime.now().strftime("%H%M%S")
                            ruta_debug = os.path.join(self.debug_dir, f"no_deteccion_{tipo_elemento}_{timestamp}.png")
                            cv2.imwrite(ruta_debug, captura)
                        
                        # Esperar y hacer scroll para próximo intento
                        if intento < max_intentos - 1:
                            # Hacer pequeño scroll
                            self.scraper.driver.execute_script("window.scrollBy(0, 100);")
                            time.sleep(1)
                
                except Exception as e:
                    logger.error(f"Error en reconocimiento visual (intento {intento+1}): {str(e)}")
                    if intento < max_intentos - 1:
                        time.sleep(1)
                
                finally:
                    # Limpiar archivo temporal
                    try:
                        if os.path.exists(temp_path):
                            os.unlink(temp_path)
                    except:
                        pass
            
            logger.warning(f"No se pudo localizar visualmente el elemento {tipo_elemento} después de {max_intentos} intentos")
            return None
            
        except Exception as e:
            logger.error(f"Error en reconocimiento visual: {str(e)}")
            traceback.print_exc()
            return None
    
    def hacer_clic_en_coordenadas(self, coordenadas):
        """
        Hace clic en las coordenadas especificadas.
        
        Args:
            coordenadas (dict): Diccionario con información de coordenadas
            
        Returns:
            bool: True si el clic fue exitoso, False en caso contrario
        """
        if not coordenadas or not self.scraper or not self.scraper.driver:
            return False
        
        try:
            # Extraer punto central
            center_x, center_y = coordenadas["center"]
            
            logger.info(f"Haciendo clic en coordenadas: ({center_x}, {center_y})")
            
            # Hacer scroll a la posición
            y_scroll = max(0, center_y - 200)  # 200px arriba para mejor visibilidad
            self.scraper.driver.execute_script(f"window.scrollTo(0, {y_scroll});")
            time.sleep(1)
            
            # Ajustar coordenadas después del scroll
            scroll_y = self.scraper.driver.execute_script("return window.pageYOffset;")
            adjusted_y = center_y - scroll_y
            
            # Intentar hacer clic con JavaScript (más preciso)
            resultado = self.scraper.driver.execute_script(f"""
                const element = document.elementFromPoint({center_x}, {adjusted_y});
                if (element) {{
                    const tagName = element.tagName.toLowerCase();
                    const innerHTML = element.innerHTML;
                    const outerHTML = element.outerHTML;
                    element.click();
                    return {{
                        success: true,
                        tagName: tagName,
                        innerHTML: innerHTML.substring(0, 100),
                        outerHTML: outerHTML.substring(0, 100)
                    }};
                }}
                return {{ success: false }};
            """)
            
            if resultado and resultado.get("success", False):
                logger.info(f"Clic exitoso mediante JavaScript")
                logger.debug(f"Elemento clicado: {resultado.get('tagName')} - {resultado.get('innerHTML')}")
                time.sleep(2)  # Esperar a que cualquier acción ocurra tras el clic
                return True
            
            # Si falló el JavaScript, intentar con ActionChains
            logger.info("Intentando clic con ActionChains...")
            actions = ActionChains(self.scraper.driver)
            actions.move_by_offset(center_x, adjusted_y).click().perform()
            actions.reset_actions()  # Resetear para próximas acciones
            
            logger.info("Clic con ActionChains completado")
            time.sleep(2)
            return True
        
        except Exception as e:
            logger.error(f"Error al hacer clic en coordenadas: {str(e)}")
            return False
    
    def procesar_archivo_resultados(self, archivo_json):
        """
        Procesa un archivo JSON con resultados de búsqueda 
        para identificar jurisprudencias.
        
        Args:
            archivo_json (str): Ruta al archivo JSON con resultados
            
        Returns:
            list: Lista de jurisprudencias identificadas
        """
        logger.info(f"Procesando archivo de resultados: {archivo_json}")
        
        if not os.path.exists(archivo_json):
            logger.error(f"El archivo {archivo_json} no existe")
            return []
        
        try:
            # Cargar el archivo JSON
            with open(archivo_json, 'r', encoding='utf-8') as f:
                datos = json.load(f)
            
            # Extraer resultados
            resultados = datos.get('resultados', [])
            if not resultados:
                logger.warning(f"No se encontraron resultados en {archivo_json}")
                return []
            
            logger.info(f"Encontrados {len(resultados)} resultados en el archivo JSON")
            
            # Identificar jurisprudencias (que contienen 'J' en el número de tesis)
            jurisprudencias = []
            
            for resultado in resultados:
                # Obtener número de tesis
                numero_tesis = resultado.get('numero_tesis', '')
                
                # Verificar si es una jurisprudencia (contiene 'J' o 'j')
                if 'J' in numero_tesis or 'j' in numero_tesis:
                    # Agregar a la lista de jurisprudencias
                    jurisprudencias.append(resultado)
                    logger.debug(f"Identificada jurisprudencia: {numero_tesis}")
            
            # Actualizar estadísticas
            self.estadisticas["jurisprudencias_identificadas"] = len(jurisprudencias)
            
            logger.info(f"Se identificaron {len(jurisprudencias)} jurisprudencias de {len(resultados)} resultados")
            return jurisprudencias
        
        except Exception as e:
            logger.error(f"Error al procesar archivo JSON {archivo_json}: {str(e)}")
            traceback.print_exc()
            return []
    
    def obtener_numero_precedente(self, registro_digital, numero_tesis):
        """
        Accede a la página de detalle de una tesis y obtiene
        el número de precedente de la jurisprudencia.
        
        Args:
            registro_digital (str): Registro digital de la tesis
            numero_tesis (str): Número de tesis
            
        Returns:
            str: Número de precedente, o None si no se encuentra
        """
        logger.info(f"Obteniendo número de precedente para tesis: {numero_tesis} (Registro: {registro_digital})")
        
        if not self.inicializar_scraper():
            logger.error("No se pudo inicializar el scraper para obtener número de precedente")
            return None
        
        try:
            # 1. CONSTRUIR Y ACCEDER A LA URL DE LA JURISPRUDENCIA
            url_tesis = f"https://sjf2.scjn.gob.mx/detalle/tesis/{registro_digital}"
            logger.info(f"Accediendo a URL: {url_tesis}")
            
            # Navegación optimizada
            exito = False
            try:
                # Método directo con timeout ampliado
                self.scraper.driver.set_page_load_timeout(90)
                self.scraper.driver.get(url_tesis)
                time.sleep(5)
                
                # Verificación rápida
                if registro_digital in self.scraper.driver.current_url:
                    logger.info("Navegación directa exitosa")
                    exito = True
            except Exception as e:
                logger.warning(f"Error en navegación directa: {str(e)}")
            
            # Si falla el método directo, usar método avanzado
            if not exito:
                if hasattr(self.scraper, 'navegar_a_url'):
                    exito = self.scraper.navegar_a_url(url_tesis, max_intentos=5)
                else:
                    # Compatibilidad con ScraperTesisCompuesto
                    exito = False
                    max_intentos_navegacion = 5
                    for intento in range(max_intentos_navegacion):
                        try:
                            self.scraper.driver.get(url_tesis)
                            WebDriverWait(self.scraper.driver, 45).until(
                                EC.presence_of_element_located((By.TAG_NAME, "body"))
                            )
                            exito = True
                            break
                        except Exception as e:
                            logger.warning(f"Error al cargar página (intento {intento+1}): {str(e)}")
                            time.sleep(3 * (intento + 1))
            
            if not exito:
                logger.error(f"No se pudo cargar la URL: {url_tesis}")
                self.guardar_captura(f"error_carga_{registro_digital}")
                return None
            
            # Esperas para carga completa
            time.sleep(3)
            EsperaDinamica.esperar_angular_quieto(self.scraper.driver, timeout=30)
            
            # Captura de diagnóstico
            self.guardar_captura(f"pagina_tesis_{registro_digital}")
            
            # 2. LOCALIZAR Y HACER CLIC EN EL BOTÓN "P" USANDO RECONOCIMIENTO VISUAL
            logger.info("Intentando localizar botón P mediante reconocimiento visual...")
            boton_encontrado = False
            
            # Verificar si tenemos OpenCV y la imagen de referencia
            if self.reconocimiento_visual:
                # Preparar imagen de referencia
                if self.preparar_imagen_referencia("boton_p"):
                    # Localizar el botón visualmente
                    coordenadas = self.localizar_elemento_visual("boton_p", umbral=0.6, max_intentos=3)
                    
                    if coordenadas:
                        logger.info(f"Botón P localizado visualmente con confianza: {coordenadas['confianza']:.4f}")
                        
                        # Hacer clic en el botón
                        if self.hacer_clic_en_coordenadas(coordenadas):
                            logger.info("Clic en botón P exitoso mediante reconocimiento visual")
                            boton_encontrado = True
                        else:
                            logger.warning("Fallo al hacer clic en el botón P localizado visualmente")
                    else:
                        logger.warning("No se pudo localizar el botón P visualmente")
            else:
                logger.info("Reconocimiento visual no disponible, usando métodos alternativos")
            
            # Si falló el reconocimiento visual, intentar métodos basados en DOM
            if not boton_encontrado:
                logger.info("Intentando localizar botón P mediante métodos basados en DOM...")
                
                # Lista de selectores para encontrar el botón P
                selectores_boton = [
                    # Selectores específicos para el botón con tooltip
                    {"tipo": By.CSS_SELECTOR, "valor": 'button[ngbtooltip="Precedente(s) de la tesis"]'},
                    {"tipo": By.XPATH, "valor": '//button[@ngbtooltip="Precedente(s) de la tesis"]'},
                    # Selectores para buscar el botón con mat-icon
                    {"tipo": By.XPATH, "valor": '//button[.//mat-icon[text()="P"]]'},
                    {"tipo": By.XPATH, "valor": '//button[.//mat-icon[contains(text(),"P")]]'},
                    # Selectores por clase CSS
                    {"tipo": By.CSS_SELECTOR, "valor": 'button.btn.btn-scjn.ng-star-inserted'},
                    # Selectores generales como respaldo
                    {"tipo": By.XPATH, "valor": '//button[text()="P"]'},
                    {"tipo": By.CSS_SELECTOR, "valor": '.btn-group button'}
                ]
                
                boton_precedentes = None
                
                # Probar cada selector hasta encontrar el botón
                for selector in selectores_boton:
                    try:
                        boton = EsperaDinamica.esperar_elemento_interactuable(
                            self.scraper.driver, 
                            selector["valor"], 
                            tipo=selector["tipo"],
                            timeout=10
                        )
                        
                        if boton:
                            logger.info(f"Botón de precedentes encontrado con selector: {selector['valor']}")
                            boton_precedentes = boton
                            break
                    except Exception as e:
                        logger.debug(f"No se encontró el botón con selector {selector['valor']}: {str(e)}")
                
                # Si no encontramos el botón con los selectores, intentar con JavaScript
                if not boton_precedentes:
                    script_busqueda = """
                    function encontrarBotonP() {
                        // 1. Buscar por atributo ngbtooltip específico
                        const botonesConTooltip = document.querySelectorAll('button[ngbtooltip="Precedente(s) de la tesis"]');
                        if (botonesConTooltip.length > 0) {
                            console.log("Botón encontrado por ngbtooltip exacto");
                            return botonesConTooltip[0];
                        }
                        
                        // 2. Buscar cualquier botón con tooltip parcial
                        const botonesConTooltipParcial = Array.from(document.querySelectorAll('button'))
                            .filter(btn => {
                                const tooltip = btn.getAttribute('ngbtooltip') || '';
                                return tooltip.includes('Precedente');
                            });
                        
                        if (botonesConTooltipParcial.length > 0) {
                            console.log("Botón encontrado por tooltip parcial");
                            return botonesConTooltipParcial[0];
                        }
                        
                        // 3. Buscar botones con mat-icon que contenga P
                        const botones = document.querySelectorAll('button');
                        for (const btn of botones) {
                            const matIcon = btn.querySelector('mat-icon');
                            if (matIcon && matIcon.textContent.trim() === 'P') {
                                console.log("Botón encontrado por mat-icon con P");
                                return btn;
                            }
                        }
                        
                        // 4. Buscar en el grupo de Asociaciones
                        const grupoAsociaciones = document.querySelector('div[role="group"][aria-label="Asociaciones"]');
                        if (grupoAsociaciones) {
                            const botonesBarra = grupoAsociaciones.querySelectorAll('button');
                            if (botonesBarra.length > 0) {
                                console.log("Devolviendo primer botón del grupo Asociaciones");
                                return botonesBarra[0];
                            }
                        }
                        
                        // 5. Última opción: cualquier botón con P
                        const todosLosBotones = Array.from(document.querySelectorAll('button'));
                        const botonP = todosLosBotones.find(b => {
                            const texto = b.textContent.trim();
                            return texto === 'P' || (texto.length < 3 && texto.includes('P'));
                        });
                        
                        if (botonP) {
                            console.log("Botón encontrado por texto P");
                            return botonP;
                        }
                        
                        console.log("No se encontró ningún botón P");
                        return null;
                    }
                    return encontrarBotonP();
                    """
                    
                    try:
                        boton_precedentes = self.scraper.driver.execute_script(script_busqueda)
                        if boton_precedentes:
                            logger.info("Botón de precedentes encontrado con JavaScript")
                    except Exception as e:
                        logger.warning(f"Error al ejecutar script de búsqueda: {str(e)}")
                
                # Si encontramos el botón, intentar hacer clic
                if boton_precedentes:
                    # Mostrar información de diagnóstico
                    try:
                        html_boton = self.scraper.driver.execute_script(
                            "return arguments[0].outerHTML;", boton_precedentes)
                        logger.info(f"HTML del botón encontrado: {html_boton}")
                    except:
                        pass
                    
                    # Hacer clic en el botón
                    exito_clic = EsperaDinamica.hacer_clic_seguro(
                        self.scraper.driver, 
                        boton_precedentes,
                        intentos=3,
                        usar_js=True
                    )
                    
                    if exito_clic:
                        logger.info("Clic en botón P exitoso mediante métodos DOM")
                        boton_encontrado = True
                    else:
                        logger.warning("Fallo al hacer clic en el botón P mediante métodos DOM")
            
            # Si no pudimos encontrar o hacer clic en el botón, fallar
            if not boton_encontrado:
                logger.error("No se pudo encontrar o hacer clic en el botón de precedentes")
                self.guardar_captura(f"no_boton_precedentes_{registro_digital}")
                return None
            
            # 3. ESPERAR QUE APAREZCA EL MODAL Y EXTRAER NÚMERO DE PRECEDENTE
            logger.info("Esperando a que aparezca el modal de precedentes...")
            time.sleep(2)
            
            # Capturar estado después del clic
            self.guardar_captura(f"despues_clic_precedentes_{registro_digital}")
            
            # Detectar el modal
            modal = EsperaDinamica.esperar_modal(self.scraper.driver, timeout=15)
            
            if not modal:
                logger.error("No se pudo detectar el modal de precedentes")
                self.guardar_captura(f"no_modal_precedentes_{registro_digital}")
                return None
            
            # Extracción optimizada del número de precedente
            script_extraccion = """
            function extraerNumerosPrecedentes() {
                // Obtener el modal activo
                const modal = document.querySelector('.mat-dialog-container') || 
                              document.querySelector('[role="dialog"]');
                              
                if (!modal) return null;
                
                // Estrategia 1: Buscar enlaces con el detalle del precedente
                const enlaces = modal.querySelectorAll('a[ngbtooltip="Ver detalle"]');
                if (enlaces && enlaces.length > 0) {
                    const numeros = [];
                    enlaces.forEach(enlace => {
                        const textoEnlace = enlace.textContent || '';
                        const match = textoEnlace.match(/\\d+\\.\\s*(\\d+)/);
                        if (match && match[1]) {
                            numeros.push(match[1]);
                        }
                    });
                    if (numeros.length > 0) return numeros;
                }
                
                // Estrategia 2: Buscar en elementos h6
                const h6Elements = modal.querySelectorAll('h6');
                if (h6Elements && h6Elements.length > 0) {
                    const numeros = [];
                    h6Elements.forEach(h6 => {
                        const texto = h6.textContent || '';
                        const match = texto.match(/\\d+\\.\\s*(\\d+)/);
                        if (match && match[1]) {
                            numeros.push(match[1]);
                        }
                    });
                    if (numeros.length > 0) return numeros;
                }
                
                // Estrategia 3: Buscar en cualquier elemento del modal
                const numeros = [];
                const elementos = modal.querySelectorAll('*');
                for (const elem of elementos) {
                    if (elem.children.length === 0) {
                        const texto = elem.textContent || '';
                        const match = texto.match(/\\d+\\.\\s*(\\d+)/);
                        if (match && match[1]) {
                            numeros.push(match[1]);
                        }
                    }
                }
                return numeros.length > 0 ? numeros : null;
            }
            return extraerNumerosPrecedentes();
            """
            
            numeros_precedentes = None
            
            try:
                numeros_precedentes = self.scraper.driver.execute_script(script_extraccion)
                if numeros_precedentes and len(numeros_precedentes) > 0:
                    logger.info(f"Números de precedentes extraídos: {numeros_precedentes}")
                else:
                    logger.warning("No se encontraron números de precedente con script")
            except Exception as e:
                logger.warning(f"Error al ejecutar script de extracción: {str(e)}")
            
            # Si falló el script, intentar extracción manual
            if not numeros_precedentes:
                logger.info("Intentando extracción manual del número de precedente")
                numeros_precedentes = []
                
                try:
                    # Buscar enlaces con 'Ver detalle'
                    enlaces = modal.find_elements(By.CSS_SELECTOR, 'a[ngbtooltip="Ver detalle"]')
                    
                    if enlaces:
                        for enlace in enlaces:
                            texto = enlace.text.strip()
                            match = re.search(r'\d+\.\s*(\d+)', texto)
                            if match:
                                numeros_precedentes.append(match.group(1))
                                logger.info(f"Número de precedente encontrado: {match.group(1)}")
                    
                    # Si no hay enlaces, buscar en cualquier texto del modal
                    if not numeros_precedentes:
                        texto_modal = modal.text
                        matches = re.findall(r'\d+\.\s*(\d+)', texto_modal)
                        if matches:
                            numeros_precedentes = matches
                            logger.info(f"Números de precedente extraídos del texto: {matches}")
                
                except Exception as e:
                    logger.error(f"Error en extracción manual: {str(e)}")
            
            # Cerrar el modal
            try:
                # Buscar botón de cierre
                botones_cierre = [
                    (By.CSS_SELECTOR, "button.close"),
                    (By.CSS_SELECTOR, "button[aria-label='Close']"),
                    (By.CSS_SELECTOR, "button.btn-close"),
                    (By.XPATH, "//button[@aria-label='Close' or contains(@class, 'close')]")
                ]
                
                boton_cerrar = None
                for tipo, selector in botones_cierre:
                    try:
                        boton_cerrar = modal.find_element(tipo, selector)
                        break
                    except:
                        continue
                
                if boton_cerrar:
                    EsperaDinamica.hacer_clic_seguro(self.scraper.driver, boton_cerrar)
                else:
                    # Intentar cerrar con ESC
                    ActionChains(self.scraper.driver).send_keys(Keys.ESCAPE).perform()
            except Exception as e:
                logger.warning(f"Error al cerrar modal: {str(e)}")
            
            # Verificar si tenemos números de precedente
            if not numeros_precedentes or len(numeros_precedentes) == 0:
                logger.error("No se pudo extraer ningún número de precedente")
                self.guardar_captura(f"no_numeros_precedente_{registro_digital}")
                return None
            
            # Tomar el primer número de precedente
            numero_precedente = numeros_precedentes[0]
            logger.info(f"Número de precedente seleccionado: {numero_precedente}")
            
            # Actualizar estadísticas
            self.estadisticas["precedentes_encontrados"] += 1
            
            return numero_precedente
        
        except Exception as e:
            logger.error(f"Error al obtener número de precedente para {numero_tesis}: {str(e)}")
            traceback.print_exc()
            
            # Guardar captura en caso de error
            self.guardar_captura(f"error_obtener_precedente_{registro_digital}")
            
            return None

    def extraer_texto_precedente(self, numero_precedente, registro_digital, numero_tesis):
        """
        Accede a la URL del precedente y extrae el texto completo.
        
        Args:
            numero_precedente (str): Número del precedente
            registro_digital (str): Registro digital (para referencias)
            numero_tesis (str): Número de tesis (para referencias)
            
        Returns:
            str: Texto completo del precedente, o None si falla
        """
        logger.info(f"Extrayendo texto del precedente {numero_precedente} (Tesis: {numero_tesis})")
        
        if not self.inicializar_scraper():
            logger.error("No se pudo inicializar el scraper para extraer texto del precedente")
            return None
        
        try:
            # 1. CONSTRUIR LA URL DEL PRECEDENTE
            url_precedente = f"https://sjf2.scjn.gob.mx/detalle/ejecutoria/{numero_precedente}"
            logger.info(f"Accediendo a URL del precedente: {url_precedente}")
            
            # Navegación optimizada
            exito = False
            try:
                # Método directo con timeout ampliado
                self.scraper.driver.set_page_load_timeout(90)
                self.scraper.driver.get(url_precedente)
                time.sleep(5)
                
                # Verificación rápida
                if numero_precedente in self.scraper.driver.current_url:
                    logger.info("Navegación directa exitosa")
                    exito = True
            except Exception as e:
                logger.warning(f"Error en navegación directa: {str(e)}")
            
            # Si falla el método directo, usar método avanzado
            if not exito:
                if hasattr(self.scraper, 'navegar_a_url'):
                    exito = self.scraper.navegar_a_url(url_precedente, max_intentos=5)
                else:
                    # Compatibilidad con ScraperTesisCompuesto
                    exito = False
                    max_intentos_navegacion = 5
                    for intento in range(max_intentos_navegacion):
                        try:
                            self.scraper.driver.get(url_precedente)
                            WebDriverWait(self.scraper.driver, 45).until(
                                EC.presence_of_element_located((By.TAG_NAME, "body"))
                            )
                            exito = True
                            break
                        except Exception as e:
                            logger.warning(f"Error al cargar página (intento {intento+1}): {str(e)}")
                            time.sleep(3 * (intento + 1))
            
            if not exito:
                logger.error(f"No se pudo cargar la URL del precedente: {url_precedente}")
                self.guardar_captura(f"error_carga_precedente_{numero_precedente}")
                return None
            
            # Esperas para carga completa
            time.sleep(3)
            EsperaDinamica.esperar_angular_quieto(self.scraper.driver, timeout=30)
            
            # Diagnóstico visual
            self.guardar_captura(f"pagina_precedente_{numero_precedente}")
            
            # 2. BUSCAR EL BOTÓN "COPIAR" USANDO RECONOCIMIENTO VISUAL
            logger.info("Intentando localizar botón de Copiar...")
            boton_encontrado = False
            
            # Verificar si tenemos OpenCV y preparar imagen de referencia si es necesario
            if self.reconocimiento_visual:
                # Ruta de imagen de referencia para botón copiar
                ruta_imagen_boton_copiar = os.path.join(self.ref_images_dir, "boton_copiar_referencia.png")
                
                # Si no existe, intentar capturarla
                if not os.path.exists(ruta_imagen_boton_copiar):
                    logger.info("Imagen de referencia para botón copiar no encontrada, intentando buscar por DOM")
                else:
                    # Localizar el botón visualmente
                    logger.info("Buscando botón copiar mediante reconocimiento visual...")
                    coordenadas = self.localizar_elemento_visual("boton_copiar", umbral=0.6, max_intentos=3)
                    
                    if coordenadas:
                        logger.info(f"Botón copiar localizado visualmente con confianza: {coordenadas['confianza']:.4f}")
                        
                        # Hacer clic en el botón
                        if self.hacer_clic_en_coordenadas(coordenadas):
                            logger.info("Clic en botón copiar exitoso mediante reconocimiento visual")
                            boton_encontrado = True
                            # Esperar a que se copie al portapapeles
                            time.sleep(2)
                        else:
                            logger.warning("Fallo al hacer clic en el botón copiar localizado visualmente")
                    else:
                        logger.warning("No se pudo localizar el botón copiar visualmente")
            
            # Si falló el reconocimiento visual, intentar métodos basados en DOM
            if not boton_encontrado:
                logger.info("Intentando localizar botón copiar mediante métodos basados en DOM...")
                
                # Lista de selectores para encontrar el botón de copiar
                selectores_boton_copiar = [
                    {"tipo": By.CSS_SELECTOR, "valor": 'button[ngbtooltip="Copiar"][aria-label="Copiar texto"]'},
                    {"tipo": By.XPATH, "valor": '//button[@ngbtooltip="Copiar" and @aria-label="Copiar texto"]'},
                    {"tipo": By.CSS_SELECTOR, "valor": 'button[ngbtooltip="Copiar"]'},
                    {"tipo": By.XPATH, "valor": '//button[contains(@ngbtooltip, "Copiar")]'},
                    {"tipo": By.XPATH, "valor": '//button[.//fa-icon[@icon="copy"]]'},
                    {"tipo": By.XPATH, "valor": '//button[contains(., "Copiar")]'}
                ]
                
                boton_copiar = None
                
                # Probar cada selector hasta encontrar el botón
                for selector in selectores_boton_copiar:
                    try:
                        boton = EsperaDinamica.esperar_elemento_interactuable(
                            self.scraper.driver, 
                            selector["valor"], 
                            tipo=selector["tipo"],
                            timeout=10
                        )
                        
                        if boton:
                            logger.info(f"Botón de copiar encontrado con selector: {selector['valor']}")
                            boton_copiar = boton
                            break
                    except Exception as e:
                        logger.debug(f"No se encontró el botón con selector {selector['valor']}: {str(e)}")
                
                # Si no encontramos el botón con los selectores, intentar con JavaScript
                if not boton_copiar:
                    script_busqueda = """
                    function encontrarBotonCopiar() {
                        // 1. Buscar por atributo ngbtooltip exacto
                        let boton = document.querySelector('button[ngbtooltip="Copiar"]');
                        if (boton) return boton;
                        
                        // 2. Buscar en grupo 'Third group'
                        const grupo = document.querySelector('div[role="group"][aria-label="Third group"]');
                        if (grupo) {
                            const botones = grupo.querySelectorAll('button');
                            for (const btn of botones) {
                                const tooltip = btn.getAttribute('ngbtooltip') || '';
                                if (tooltip.includes('Copiar')) return btn;
                            }
                        }
                        
                        // 3. Buscar por ícono de copia
                        const botones = document.querySelectorAll('button');
                        for (const btn of botones) {
                            if (btn.querySelector('fa-icon[icon="copy"], svg.fa-copy')) {
                                return btn;
                            }
                        }
                        
                        // 4. Buscar por texto o atributos parciales
                        const todosLosBotones = document.querySelectorAll('button');
                        for (const btn of todosLosBotones) {
                            const tooltip = btn.getAttribute('ngbtooltip') || '';
                            const ariaLabel = btn.getAttribute('aria-label') || '';
                            const texto = btn.textContent || '';
                            if (tooltip.includes('Copiar') || ariaLabel.includes('Copiar') || 
                                texto.toLowerCase().includes('copiar')) {
                                return btn;
                            }
                        }
                        
                        return null;
                    }
                    return encontrarBotonCopiar();
                    """
                    
                    try:
                        boton_copiar = self.scraper.driver.execute_script(script_busqueda)
                        if boton_copiar:
                            logger.info("Botón de copiar encontrado con JavaScript")
                    except Exception as e:
                        logger.warning(f"Error al ejecutar script de búsqueda: {str(e)}")
                
                # Si encontramos el botón, hacer clic
                if boton_copiar:
                    # Guardar captura antes del clic
                    self.guardar_captura(f"antes_clic_copiar_{numero_precedente}")
                    
                    # Hacer clic
                    exito_clic = EsperaDinamica.hacer_clic_seguro(
                        self.scraper.driver, 
                        boton_copiar,
                        intentos=3,
                        usar_js=True
                    )
                    
                    if exito_clic:
                        logger.info("Clic en botón de copiar exitoso mediante métodos DOM")
                        boton_encontrado = True
                        # Esperar a que se copie al portapapeles
                        time.sleep(2)
                    else:
                        logger.warning("Fallo al hacer clic en el botón de copiar mediante métodos DOM")
            
            # 3. EXTRAER TEXTO (MULTIESTRATEGIA)
            texto_precedente = None
            
            # Estrategia 1: Portapapeles (si se hizo clic en copiar)
            if boton_encontrado:
                logger.info("Obteniendo texto del portapapeles...")
                texto_precedente = self._obtener_texto_del_portapapeles()
                
                if texto_precedente and len(texto_precedente) > 500:
                    logger.info(f"Texto obtenido del portapapeles: {len(texto_precedente)} caracteres")
                else:
                    logger.warning("Texto del portapapeles vacío o muy corto")
                    texto_precedente = None
            
            # Estrategia 2: Extracción directa del DOM
            if not texto_precedente:
                logger.info("Intentando extraer texto directamente del DOM...")
                
                script_extraccion = """
                function extraerTextoPrecedente() {
                    // Estrategia 1: Selectores específicos
                    const selectores = [
                        'div.mat-card-content',
                        'mat-card-content',
                        'div.ejecutoria-content',
                        'div.content-wrapper',
                        'div.ng-star-inserted > div.row'
                    ];
                    
                    for (const selector of selectores) {
                        const elemento = document.querySelector(selector);
                        if (elemento) {
                            return elemento.innerText;
                        }
                    }
                    
                    // Estrategia 2: Heurística - buscar el elemento más grande
                    const candidatos = Array.from(document.querySelectorAll('div, main, article, section'))
                        .filter(el => {
                            const texto = el.innerText || '';
                            return texto.length > 1000 && 
                                   !texto.includes('menu') && 
                                   !texto.includes('navigation');
                        })
                        .sort((a, b) => {
                            const textoA = a.innerText || '';
                            const textoB = b.innerText || '';
                            return textoB.length - textoA.length;
                        });
                    
                    if (candidatos.length > 0) {
                        return candidatos[0].innerText;
                    }
                    
                    // Estrategia 3: Todo el texto del body filtrado
                    const bodyText = document.body.innerText;
                    const lineas = bodyText.split('\\n')
                        .filter(linea => 
                            linea.trim().length > 5 && 
                            !linea.includes('Buscar') &&
                            !linea.includes('Iniciar sesión') &&
                            !linea.includes('Cerrar sesión')
                        );
                    
                    return lineas.join('\\n');
                }
                return extraerTextoPrecedente();
                """
                
                try:
                    texto_dom = self.scraper.driver.execute_script(script_extraccion)
                    if texto_dom and len(texto_dom) > 500:
                        logger.info(f"Texto extraído del DOM: {len(texto_dom)} caracteres")
                        texto_precedente = texto_dom
                    else:
                        logger.warning("Texto extraído del DOM vacío o muy corto")
                except Exception as e:
                    logger.warning(f"Error al extraer texto del DOM: {str(e)}")
            
            # Estrategia 3: Selección de todo el texto
            if not texto_precedente:
                logger.info("Intentando seleccionar todo el texto con Ctrl+A...")
                
                try:
                    # Enfocar el body
                    body = self.scraper.driver.find_element(By.TAG_NAME, "body")
                    body.click()
                    time.sleep(0.5)
                    
                    # Seleccionar todo (Ctrl+A)
                    actions = ActionChains(self.scraper.driver)
                    actions.key_down(Keys.CONTROL).send_keys('a').key_up(Keys.CONTROL).perform()
                    time.sleep(1)
                    
                    # Copiar (Ctrl+C)
                    actions.key_down(Keys.CONTROL).send_keys('c').key_up(Keys.CONTROL).perform()
                    time.sleep(1)
                    
                    # Obtener del portapapeles
                    texto_seleccionado = self._obtener_texto_del_portapapeles()
                    
                    if texto_seleccionado and len(texto_seleccionado) > 500:
                        logger.info(f"Texto obtenido mediante selección: {len(texto_seleccionado)} caracteres")
                        # Limpiar el texto
                        lineas = texto_seleccionado.split('\n')
                        texto_filtrado = '\n'.join([l for l in lineas if len(l) > 5 and not ('Menu' in l or 'Buscar' in l)])
                        texto_precedente = texto_filtrado
                except Exception as e:
                    logger.warning(f"Error al seleccionar texto: {str(e)}")
            
            # Verificar si tenemos texto
            if not texto_precedente or len(texto_precedente) < 200:
                logger.error("No se pudo extraer texto del precedente")
                self.guardar_captura(f"no_texto_precedente_{numero_precedente}")
                return None
            
            # Mejora: limpiar texto extraído para mejor formato
            texto_precedente = self._limpiar_texto_precedente(texto_precedente)
            
            return texto_precedente
        
        except Exception as e:
            logger.error(f"Error al extraer texto del precedente {numero_precedente}: {str(e)}")
            traceback.print_exc()
            
            # Guardar captura en caso de error
            self.guardar_captura(f"error_extraer_texto_{numero_precedente}")
            
            return None
    
    def _limpiar_texto_precedente(self, texto):
        """Limpia y formatea el texto extraído del precedente"""
        if not texto:
            return texto
        
        # Eliminar líneas vacías consecutivas
        texto = re.sub(r'\n\s*\n', '\n\n', texto)
        
        # Eliminar marcadores de carga
        texto = re.sub(r'Cargando\.\.\.', '', texto)
        
        # Eliminar elementos de interfaz comunes
        patrones_eliminar = [
            r'Iniciar sesión',
            r'Cerrar sesión',
            r'Menú',
            r'Búsqueda',
            r'Copyright © \d+',
            r'Suprema Corte de Justicia de la Nación',
            r'Todos los derechos reservados'
        ]
        
        for patron in patrones_eliminar:
            texto = re.sub(patron, '', texto)
        
        # Eliminar espacios múltiples
        texto = re.sub(r' +', ' ', texto)
        
        # Eliminar espacios al inicio y final de líneas
        texto = '\n'.join([linea.strip() for linea in texto.split('\n')])
        
        return texto
    
    def _obtener_texto_del_portapapeles(self):
        """
        Método avanzado para obtener texto del portapapeles con múltiples estrategias.
        
        Returns:
            str: Texto del portapapeles o None si no se puede obtener
        """
        # Estrategia 1: Usar document.execCommand (más compatible)
        script_execcommand = """
        try {
            // Crear un elemento textarea para obtener el portapapeles
            var textarea = document.createElement('textarea');
            document.body.appendChild(textarea);
            textarea.style.position = 'fixed';
            textarea.style.opacity = 0;
            
            // Usar document.execCommand para pegar el contenido del portapapeles
            textarea.focus();
            var success = document.execCommand('paste');
            var contenido = textarea.value;
            
            // Limpiar
            document.body.removeChild(textarea);
            
            if (success && contenido && contenido.length > 0) {
                return contenido;
            } else {
                return null;
            }
        } catch (e) {
            console.error("Error con execCommand:", e);
            return null;
        }
        """
        
        # Estrategia 2: Usar navigator.clipboard API (navegadores modernos)
        script_clipboard_api = """
        return new Promise((resolve, reject) => {
            if (navigator.clipboard && navigator.clipboard.readText) {
                navigator.clipboard.readText()
                    .then(text => resolve(text))
                    .catch(err => {
                        console.error("Error al leer portapapeles:", err);
                        resolve(null);
                    });
            } else {
                resolve(null);
            }
        });
        """
        
        # Intentar cada estrategia en orden
        try:
            # Intento 1: execCommand
            texto = self.scraper.driver.execute_script(script_execcommand)
            if texto and len(texto) > 10:
                logger.info("Texto obtenido del portapapeles con execCommand")
                return texto
            
            # Intento 2: Clipboard API
            texto = self.scraper.driver.execute_async_script(script_clipboard_api)
            if texto and len(texto) > 10:
                logger.info("Texto obtenido del portapapeles con Clipboard API")
                return texto
            
            logger.warning("No se pudo obtener texto del portapapeles con ninguna estrategia")
            return None
        
        except Exception as e:
            logger.warning(f"Error al obtener texto del portapapeles: {str(e)}")
            return None
    
    def guardar_texto_precedente(self, texto_precedente, numero_precedente, registro_digital, numero_tesis):
        """
        Guarda el texto de un precedente en un archivo formateado.
        
        Args:
            texto_precedente (str): Texto completo del precedente
            numero_precedente (str): Número del precedente
            registro_digital (str): Registro digital de la tesis
            numero_tesis (str): Número de tesis
            
        Returns:
            str: Ruta al archivo generado, o None si falla
        """
        if not texto_precedente:
            logger.warning(f"No hay texto para guardar (Precedente: {numero_precedente})")
            return None
        
        try:
            # Generar nombre de archivo
            # Formato: precedente_[NUMERO]_tesis_[REGISTRO]_[TIMESTAMP].txt
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            nombre_archivo = f"precedente_{numero_precedente}_tesis_{registro_digital}_{timestamp}.txt"
            ruta_archivo = os.path.join(self.directorio_salida, nombre_archivo)
            
            # Preparar encabezado
            encabezado = "=" * 80 + "\n"
            encabezado += f"PRECEDENTE JURISPRUDENCIAL\n"
            encabezado += f"Número de Precedente: {numero_precedente}\n"
            encabezado += f"Tesis relacionada: {numero_tesis}\n"
            encabezado += f"Registro Digital: {registro_digital}\n"
            encabezado += f"Fecha de extracción: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n"
            encabezado += f"URL: https://sjf2.scjn.gob.mx/detalle/ejecutoria/{numero_precedente}\n"
            encabezado += "=" * 80 + "\n\n"
            
            # Guardar archivo
            with open(ruta_archivo, 'w', encoding='utf-8') as f:
                f.write(encabezado)
                f.write(texto_precedente)
                f.write("\n\n" + "=" * 80 + "\n")
                f.write("FIN DEL PRECEDENTE\n")
                f.write("=" * 80 + "\n")
            
            # Actualizar estadísticas
            self.estadisticas["archivos_generados"].append(ruta_archivo)
            
            logger.info(f"Precedente guardado en: {ruta_archivo}")
            return ruta_archivo
        
        except Exception as e:
            logger.error(f"Error al guardar texto del precedente {numero_precedente}: {str(e)}")
            traceback.print_exc()
            return None
    
    def generar_informe_extraccion(self):
        """
        Genera un informe detallado de la extracción de precedentes.
        
        Returns:
            str: Ruta al archivo de informe
        """
        try:
            # Calcular tiempo total
            tiempo_total = time.time() - self.estadisticas["tiempo_inicio"]
            
            # Generar nombre de archivo
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            nombre_archivo = f"informe_extraccion_precedentes_{timestamp}.txt"
            ruta_archivo = os.path.join(self.directorio_salida, nombre_archivo)
            
            # Preparar informe
            informe = "=" * 80 + "\n"
            informe += "INFORME DE EXTRACCIÓN DE PRECEDENTES JURISPRUDENCIALES\n"
            informe += f"Fecha de generación: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n"
            informe += "=" * 80 + "\n\n"
            
            # Estadísticas generales
            informe += "ESTADÍSTICAS GENERALES:\n"
            informe += f"- Jurisprudencias identificadas: {self.estadisticas['jurisprudencias_identificadas']}\n"
            informe += f"- Precedentes encontrados: {self.estadisticas['precedentes_encontrados']}\n"
            informe += f"- Precedentes extraídos correctamente: {self.estadisticas['precedentes_extraidos']}\n"
            informe += f"- Errores: {self.estadisticas['errores']}\n"
            informe += f"- Eficacia: {self.calcular_eficacia():.2f}%\n"
            informe += f"- Tiempo total de ejecución: {tiempo_total:.2f} segundos\n"
            
            # Información de configuración
            informe += "\nCONFIGURACIÓN:\n"
            informe += f"- Modo headless: {'Sí' if self.headless else 'No'}\n"
            informe += f"- Modo debug: {'Sí' if self.modo_debug else 'No'}\n"
            informe += f"- Reconocimiento visual: {'Disponible' if self.reconocimiento_visual else 'No disponible'}\n"
            informe += f"- Tiempo máximo de espera: {self.timeout_default} segundos\n"
            informe += f"- Directorio de salida: {self.directorio_salida}\n"
            
            # Lista de archivos generados
            informe += "\nARCHIVOS GENERADOS:\n"
            for i, archivo in enumerate(self.estadisticas["archivos_generados"]):
                informe += f"{i+1}. {os.path.basename(archivo)}\n"
            
            # Guardar informe
            with open(ruta_archivo, 'w', encoding='utf-8') as f:
                f.write(informe)
            
            logger.info(f"Informe de extracción generado en: {ruta_archivo}")
            return ruta_archivo
        
        except Exception as e:
            logger.error(f"Error al generar informe de extracción: {str(e)}")
            return None
    
    def calcular_eficacia(self):
        """Calcula el porcentaje de éxito en la extracción"""
        if self.estadisticas["jurisprudencias_identificadas"] == 0:
            return 0
        
        return (self.estadisticas["precedentes_extraidos"] / 
                self.estadisticas["jurisprudencias_identificadas"]) * 100
    
    def extraer_precedentes(self, archivo_json):
        """
        Proceso principal: extrae precedentes a partir de un archivo de resultados.
        
        Args:
            archivo_json (str): Ruta al archivo JSON con resultados
            
        Returns:
            dict: Estadísticas del proceso de extracción
        """
        logger.info(f"Iniciando extracción de precedentes desde: {archivo_json}")
        
        try:
            # Inicializar estadísticas
            self.estadisticas = {
                "tiempo_inicio": time.time(),
                "jurisprudencias_identificadas": 0,
                "precedentes_encontrados": 0, 
                "precedentes_extraidos": 0,
                "errores": 0,
                "archivos_generados": []
            }
            
            # Paso 1: Identificar jurisprudencias
            jurisprudencias = self.procesar_archivo_resultados(archivo_json)
            
            if not jurisprudencias:
                logger.warning("No se identificaron jurisprudencias en el archivo de resultados")
                return self.estadisticas
            
            # Paso 2: Inicializar el scraper
            if not self.inicializar_scraper():
                logger.error("No se pudo inicializar el scraper. Abortando extracción.")
                self.estadisticas["errores"] += 1
                return self.estadisticas
            
            # Paso 3: Procesar cada jurisprudencia
            for i, jurisprudencia in enumerate(jurisprudencias):
                registro_digital = jurisprudencia.get('registro_digital', '')
                numero_tesis = jurisprudencia.get('numero_tesis', '')
                
                if not registro_digital or registro_digital == "No identificado":
                    logger.warning(f"Jurisprudencia {i+1}/{len(jurisprudencias)}: Registro digital no válido")
                    continue
                
                logger.info(f"Procesando jurisprudencia {i+1}/{len(jurisprudencias)}: {numero_tesis}")
                
                try:
                    # Obtener número de precedente
                    numero_precedente = self.obtener_numero_precedente(registro_digital, numero_tesis)
                    
                    if not numero_precedente:
                        logger.warning(f"No se encontró número de precedente para {numero_tesis}")
                        continue
                    
                    # Extraer texto del precedente
                    texto_precedente = self.extraer_texto_precedente(
                        numero_precedente, registro_digital, numero_tesis
                    )
                    
                    if not texto_precedente:
                        logger.warning(f"No se pudo extraer texto del precedente {numero_precedente}")
                        self.estadisticas["errores"] += 1
                        continue
                    
                    # Guardar texto del precedente
                    ruta_archivo = self.guardar_texto_precedente(
                        texto_precedente, numero_precedente, registro_digital, numero_tesis
                    )
                    
                    if ruta_archivo:
                        # Actualizar estadísticas
                        self.estadisticas["precedentes_extraidos"] += 1
                        logger.info(f"Precedente {numero_precedente} extraído y guardado exitosamente")
                    else:
                        self.estadisticas["errores"] += 1
                
                except Exception as e:
                    logger.error(f"Error procesando jurisprudencia {numero_tesis}: {str(e)}")
                    self.estadisticas["errores"] += 1
                    traceback.print_exc()
                
                # Pequeña pausa entre jurisprudencias
                time.sleep(2 + random.random())
            
            # Paso 4: Generar informe de extracción
            self.generar_informe_extraccion()
            
            # Paso 5: Cerrar el scraper
            self.cerrar_scraper()
            
            # Calcular tiempo total
            tiempo_total = time.time() - self.estadisticas["tiempo_inicio"]
            self.estadisticas["tiempo_total"] = tiempo_total
            
            logger.info(f"Extracción completada en {tiempo_total:.2f} segundos")
            logger.info(f"Jurisprudencias identificadas: {self.estadisticas['jurisprudencias_identificadas']}")
            logger.info(f"Precedentes encontrados: {self.estadisticas['precedentes_encontrados']}")
            logger.info(f"Precedentes extraídos correctamente: {self.estadisticas['precedentes_extraidos']}")
            logger.info(f"Errores: {self.estadisticas['errores']}")
            
            return self.estadisticas
        
        except Exception as e:
            logger.error(f"Error en proceso de extracción de precedentes: {str(e)}")
            traceback.print_exc()
            
            # Asegurar que el scraper se cierra
            self.cerrar_scraper()
            
            return self.estadisticas
    
    def compilar_precedentes(self):
        """
        Compila todos los precedentes extraídos en un solo archivo.
        
        Returns:
            str: Ruta al archivo compilado
        """
        if not self.estadisticas.get("archivos_generados"):
            logger.warning("No hay archivos para compilar")
            return None
        
        try:
            # Generar nombre de archivo
            timestamp = datetime.now().strftime("%Y%m%d_%H%M%S")
            nombre_archivo = f"precedentes_compilados_{timestamp}.txt"
            ruta_archivo = os.path.join(self.directorio_salida, nombre_archivo)
            
            # Preparar encabezado
            encabezado = "=" * 80 + "\n"
            encabezado += "COMPILACIÓN DE PRECEDENTES JURISPRUDENCIALES\n"
            encabezado += f"Fecha de generación: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}\n"
            encabezado += f"Total de precedentes: {len(self.estadisticas['archivos_generados'])}\n"
            encabezado += "=" * 80 + "\n\n"
            
            # Escribir archivo compilado
            with open(ruta_archivo, 'w', encoding='utf-8') as f_compilado:
                # Escribir encabezado
                f_compilado.write(encabezado)
                
                # Procesar cada archivo
                for i, archivo in enumerate(self.estadisticas["archivos_generados"]):
                    try:
                        # Leer contenido del archivo individual
                        with open(archivo, 'r', encoding='utf-8') as f_individual:
                            contenido = f_individual.read()
                        
                        # Escribir separador
                        f_compilado.write("\n" + "=" * 80 + "\n")
                        f_compilado.write(f"PRECEDENTE #{i+1}\n")
                        f_compilado.write("=" * 80 + "\n\n")
                        
                        # Escribir contenido
                        f_compilado.write(contenido)
                        f_compilado.write("\n\n" + "-" * 80 + "\n\n")
                    
                    except Exception as e:
                        logger.error(f"Error al compilar archivo {archivo}: {str(e)}")
                        continue
                
                # Escribir final
                f_compilado.write("\n\n" + "=" * 80 + "\n")
                f_compilado.write("FIN DE LA COMPILACIÓN\n")
                f_compilado.write("=" * 80)
            
            logger.info(f"Precedentes compilados en: {ruta_archivo}")
            return ruta_archivo
        
        except Exception as e:
            logger.error(f"Error al compilar precedentes: {str(e)}")
            traceback.print_exc()
            return None


# --------------------------------------------------------
# Interfaz de línea de comandos
# --------------------------------------------------------

def ejecutar_extraccion_precedentes(archivo_json, headless=False, timeout=45, 
                                   directorio_salida="precedentes_extraidos", 
                                   modo_debug=True, compilar=True, abrir_archivo=True):
    """
    Función principal para ejecutar la extracción de precedentes desde CLI.
    
    Args:
        archivo_json (str): Ruta al archivo JSON con resultados
        headless (bool): Si se usa navegador sin interfaz visual
        timeout (int): Tiempo máximo de espera para elementos
        directorio_salida (str): Directorio donde guardar precedentes
        modo_debug (bool): Activar modo de depuración
        compilar (bool): Si se compilan todos los precedentes en un archivo
        abrir_archivo (bool): Si se abre el archivo compilado al finalizar
        
    Returns:
        dict: Estadísticas del proceso
    """
    print("\n" + "=" * 70)
    print("EXTRACTOR DE PRECEDENTES JURISPRUDENCIALES - VERSIÓN CON RECONOCIMIENTO VISUAL")
    print("=" * 70)
    
    print(f"Archivo de resultados: {archivo_json}")
    print(f"Directorio de salida: {directorio_salida}")
    print(f"Modo headless: {'Sí' if headless else 'No'}")
    print(f"Modo debug: {'Sí' if modo_debug else 'No'}")
    print(f"Reconocimiento visual: {'Disponible' if OPENCV_DISPONIBLE else 'No disponible'}")
    print(f"Compilar precedentes: {'Sí' if compilar else 'No'}")
    print(f"Abrir archivo al finalizar: {'Sí' if abrir_archivo else 'No'}")
    print("-" * 70)
    
    try:
        # Verificar que existe el archivo JSON
        if not os.path.exists(archivo_json):
            print(f"ERROR: El archivo {archivo_json} no existe.")
            return {"error": f"Archivo no encontrado: {archivo_json}"}
        
        # Inicializar extractor
        extractor = ExtractorPrecedentes(
            headless=headless,
            timeout_default=timeout,
            modo_debug=modo_debug,
            directorio_salida=directorio_salida
        )
        
        # Ejecutar extracción
        print("\nIniciando proceso de extracción...")
        estadisticas = extractor.extraer_precedentes(archivo_json)
        
        # Mostrar resultados
        print("\n" + "=" * 70)
        print("PROCESO COMPLETADO")
        print("=" * 70)
        
        print(f"Jurisprudencias identificadas: {estadisticas['jurisprudencias_identificadas']}")
        print(f"Precedentes encontrados: {estadisticas['precedentes_encontrados']}")
        print(f"Precedentes extraídos correctamente: {estadisticas['precedentes_extraidos']}")
        print(f"Errores: {estadisticas['errores']}")
        
        if estadisticas.get("tiempo_total"):
            print(f"Tiempo total: {estadisticas['tiempo_total']:.2f} segundos")
        else:
            tiempo_total = time.time() - estadisticas["tiempo_inicio"]
            print(f"Tiempo total: {tiempo_total:.2f} segundos")
        
        if estadisticas["archivos_generados"]:
            print(f"Archivos generados: {len(estadisticas['archivos_generados'])}")
            
            # Compilar precedentes si se solicita
            if compilar and len(estadisticas["archivos_generados"]) > 0:
                print("\nCompilando precedentes en un solo archivo...")
                archivo_compilado = extractor.compilar_precedentes()
                
                if archivo_compilado:
                    print(f"Precedentes compilados en: {archivo_compilado}")
                    
                    # Abrir archivo si se solicita
                    if abrir_archivo:
                        try:
                            import subprocess
                            import platform
                            
                            print("\nAbriendo archivo compilado...")
                            
                            sistema = platform.system()
                            if sistema == "Windows":
                                os.startfile(archivo_compilado)
                            elif sistema == "Darwin":  # macOS
                                subprocess.run(["open", archivo_compilado])
                            else:  # Linux y otros
                                subprocess.run(["xdg-open", archivo_compilado])
                        except Exception as e:
                            print(f"No se pudo abrir el archivo: {str(e)}")
                else:
                    print("Error al compilar precedentes. Revise el log para más detalles.")
        else:
            print("No se generaron archivos de precedentes.")
        
        print("\nProceso completado. Revise los archivos generados en el directorio de salida.")
        
        return estadisticas
    
    except Exception as e:
        print(f"\nERROR FATAL: {str(e)}")
        traceback.print_exc()
        return {"error": str(e)}


# --------------------------------------------------------
# Interfaz gráfica
# --------------------------------------------------------

class InterfazExtractorPrecedentes:
    """Interfaz gráfica mejorada para el extractor de precedentes"""
    
    def __init__(self, root):
        self.root = root
        self.root.title("Extractor de Precedentes Jurisprudenciales - Con Reconocimiento Visual")
        self.root.geometry("900x700")
        self.root.minsize(800, 600)
        
        # Variables de control
        self.archivo_json = tk.StringVar()
        self.directorio_salida = tk.StringVar(value="precedentes_extraidos")
        self.headless = tk.BooleanVar(value=False)  # Cambiado a False por defecto para reconocimiento visual
        self.modo_debug = tk.BooleanVar(value=True)  # Cambiado a True por defecto para ayudar en el diagnóstico
        self.compilar = tk.BooleanVar(value=True)
        self.abrir_archivo = tk.BooleanVar(value=True)
        self.usar_reconocimiento = tk.BooleanVar(value=True)
        
        # Crear la interfaz
        self.crear_interfaz()
    
    def crear_interfaz(self):
        """Crear los elementos de la interfaz gráfica"""
        # Estilo
        estilo = ttk.Style()
        estilo.configure("TFrame", background="#f5f5f7")
        estilo.configure("TLabel", background="#f5f5f7", font=("Roboto", 12))
        estilo.configure("TButton", font=("Roboto", 12))
        estilo.configure("TCheckbutton", background="#f5f5f7", font=("Roboto", 12))
        estilo.configure("Titulo.TLabel", font=("Roboto", 16, "bold"), foreground="#1a237e")
        
        # Frame principal
        main_frame = ttk.Frame(self.root, padding="20")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Título
        ttk.Label(main_frame, text="Extractor de Precedentes Jurisprudenciales", 
                 style="Titulo.TLabel").pack(pady=(0, 10))
        
        # Subtítulo con reconocimiento visual
        ttk.Label(main_frame, text="Con Reconocimiento Visual", 
                 font=("Roboto", 12, "italic")).pack(pady=(0, 20))
        
        # Frame para formulario
        form_frame = ttk.Frame(main_frame)
        form_frame.pack(fill=tk.X, pady=10)
        
        # Selección de archivo JSON
        ttk.Label(form_frame, text="Archivo de resultados JSON:").grid(row=0, column=0, sticky=tk.W, pady=5)
        
        file_frame = ttk.Frame(form_frame)
        file_frame.grid(row=0, column=1, sticky=tk.W+tk.E, pady=5)
        
        ttk.Entry(file_frame, textvariable=self.archivo_json, width=50).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(file_frame, text="Buscar...", command=self.seleccionar_archivo).pack(side=tk.LEFT, padx=(5, 0))
        
        # Directorio de salida
        ttk.Label(form_frame, text="Directorio de salida:").grid(row=1, column=0, sticky=tk.W, pady=5)
        
        dir_frame = ttk.Frame(form_frame)
        dir_frame.grid(row=1, column=1, sticky=tk.W+tk.E, pady=5)
        
        ttk.Entry(dir_frame, textvariable=self.directorio_salida, width=50).pack(side=tk.LEFT, fill=tk.X, expand=True)
        ttk.Button(dir_frame, text="Buscar...", command=self.seleccionar_directorio).pack(side=tk.LEFT, padx=(5, 0))
        
        # Status del reconocimiento visual
        status_frame = ttk.Frame(main_frame)
        status_frame.pack(fill=tk.X, pady=(10, 5))
        
        status_label = ttk.Label(status_frame, text="Estado del reconocimiento visual:")
        status_label.pack(side=tk.LEFT)
        
        if OPENCV_DISPONIBLE:
            status_value = ttk.Label(status_frame, text="DISPONIBLE", foreground="green", font=("Roboto", 12, "bold"))
        else:
            status_value = ttk.Label(status_frame, text="NO DISPONIBLE", foreground="red", font=("Roboto", 12, "bold"))
            # Desactivar opción de reconocimiento si no está disponible
            self.usar_reconocimiento.set(False)
        
        status_value.pack(side=tk.LEFT, padx=(10, 0))
        
        # Opciones
        options_frame = ttk.Frame(main_frame)
        options_frame.pack(fill=tk.X, pady=10)
        
        # Opciones en dos columnas
        left_options = ttk.Frame(options_frame)
        left_options.pack(side=tk.LEFT, fill=tk.Y, expand=True)
        
        right_options = ttk.Frame(options_frame)
        right_options.pack(side=tk.LEFT, fill=tk.Y, expand=True)
        
        # Columna izquierda
        ttk.Checkbutton(left_options, text="Modo sin interfaz gráfica (headless)", 
                       variable=self.headless).pack(anchor=tk.W, pady=5)
        
        ttk.Checkbutton(left_options, text="Modo debug (genera capturas)",
                       variable=self.modo_debug).pack(anchor=tk.W, pady=5)
        
        # Columna derecha
        ttk.Checkbutton(right_options, text="Usar reconocimiento visual",
                       variable=self.usar_reconocimiento,
                       state=tk.NORMAL if OPENCV_DISPONIBLE else tk.DISABLED).pack(anchor=tk.W, pady=5)
        
        ttk.Checkbutton(right_options, text="Compilar precedentes en un archivo",
                       variable=self.compilar).pack(anchor=tk.W, pady=5)
        
        ttk.Checkbutton(right_options, text="Abrir archivo compilado al finalizar",
                       variable=self.abrir_archivo).pack(anchor=tk.W, pady=5)
        
        # Área de log
        log_frame = ttk.Frame(main_frame)
        log_frame.pack(fill=tk.BOTH, expand=True, pady=10)
        
        ttk.Label(log_frame, text="Registro de actividad:").pack(anchor=tk.W)
        
        self.log_text = scrolledtext.ScrolledText(log_frame, height=15, width=80, font=("Consolas", 10))
        self.log_text.pack(fill=tk.BOTH, expand=True, pady=5)
        self.log_text.config(state=tk.DISABLED)
        
        # Barra de progreso
        progress_frame = ttk.Frame(main_frame)
        progress_frame.pack(fill=tk.X, pady=10)
        
        self.progress_label = ttk.Label(progress_frame, text="Listo para iniciar")
        self.progress_label.pack(anchor=tk.W, pady=(0, 5))
        
        self.progress = ttk.Progressbar(progress_frame, mode='indeterminate', length=100)
        self.progress.pack(fill=tk.X)
        
        # Botones de acción
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)
        
        ttk.Button(button_frame, text="Iniciar Extracción", command=self.iniciar_extraccion).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="Limpiar Log", command=self.limpiar_log).pack(side=tk.LEFT, padx=5)
        ttk.Button(button_frame, text="Salir", command=self.root.destroy).pack(side=tk.RIGHT, padx=5)
        
        # Información sobre el reconocimiento visual
        info_frame = ttk.Frame(main_frame)
        info_frame.pack(fill=tk.X, pady=(10, 0))
        
        info_text = (
            "Nota: El reconocimiento visual requiere OpenCV (pip install opencv-python numpy)."
            " En la primera ejecución, se le pedirá crear una imagen de referencia del botón P."
        )
        ttk.Label(info_frame, text=info_text, wraplength=800, font=("Roboto", 10, "italic")).pack()
    
    def seleccionar_archivo(self):
        """Abre diálogo para seleccionar archivo JSON"""
        archivo = filedialog.askopenfilename(
            title="Seleccionar archivo de resultados",
            filetypes=[("Archivos JSON", "*.json"), ("Todos los archivos", "*.*")]
        )
        if archivo:
            self.archivo_json.set(archivo)
    
    def seleccionar_directorio(self):
        """Abre diálogo para seleccionar directorio de salida"""
        directorio = filedialog.askdirectory(title="Seleccionar directorio de salida")
        if directorio:
            self.directorio_salida.set(directorio)
    
    def log(self, mensaje, nivel="INFO"):
        """Añade mensaje al área de log"""
        self.log_text.config(state=tk.NORMAL)
        timestamp = datetime.now().strftime("[%H:%M:%S] ")
        self.log_text.insert(tk.END, timestamp + f"{nivel}: {mensaje}\n")
        self.log_text.see(tk.END)
        self.log_text.config(state=tk.DISABLED)
        self.root.update_idletasks()
    
    def limpiar_log(self):
        """Limpia el área de log"""
        self.log_text.config(state=tk.NORMAL)
        self.log_text.delete(1.0, tk.END)
        self.log_text.config(state=tk.DISABLED)
    
    def iniciar_extraccion(self):
        """Inicia el proceso de extracción"""
        # Verificar que se ha seleccionado un archivo
        if not self.archivo_json.get():
            messagebox.showerror("Error", "Debe seleccionar un archivo de resultados JSON")
            return
        
        # Verificar que el archivo existe
        if not os.path.exists(self.archivo_json.get()):
            messagebox.showerror("Error", "El archivo seleccionado no existe")
            return
        
        # Verificar si el reconocimiento visual está seleccionado pero no disponible
        if self.usar_reconocimiento.get() and not OPENCV_DISPONIBLE:
            respuesta = messagebox.askquestion(
                "Reconocimiento Visual No Disponible",
                "El reconocimiento visual requiere OpenCV, que no está instalado.\n\n"
                "¿Desea continuar usando solo métodos basados en DOM?",
                icon="warning"
            )
            if respuesta != "yes":
                return
            self.usar_reconocimiento.set(False)
        
        # Verificar modo headless si se usa reconocimiento visual
        if self.headless.get() and self.usar_reconocimiento.get():
            respuesta = messagebox.askquestion(
                "Advertencia",
                "El reconocimiento visual funciona mejor con el navegador visible (modo no-headless).\n\n"
                "¿Desea desactivar el modo headless para mejor reconocimiento?",
                icon="question"
            )
            if respuesta == "yes":
                self.headless.set(False)
        
        # Deshabilitar botones durante la ejecución
        for widget in self.root.winfo_children():
            if isinstance(widget, ttk.Button):
                widget.config(state=tk.DISABLED)
        
        # Iniciar barra de progreso
        self.progress.start()
        self.progress_label.config(text="Extrayendo precedentes...")
        
        # Limpiar log
        self.limpiar_log()
        
        # Log inicial
        self.log(f"Iniciando extracción desde: {self.archivo_json.get()}")
        self.log(f"Directorio de salida: {self.directorio_salida.get()}")
        self.log(f"Modo headless: {self.headless.get()}")
        self.log(f"Modo debug: {self.modo_debug.get()}")
        self.log(f"Usar reconocimiento visual: {self.usar_reconocimiento.get()}")
        self.log(f"Compilar precedentes: {self.compilar.get()}")
        self.log(f"Abrir archivo al finalizar: {self.abrir_archivo.get()}")
        
        # Ejecutar en hilo separado para no bloquear la interfaz
        threading.Thread(
            target=self.ejecutar_extraccion_thread,
            daemon=True
        ).start()
    
    def ejecutar_extraccion_thread(self):
        """Ejecuta la extracción en un hilo separado"""
        try:
            # Crear directorio de salida si no existe
            if not os.path.exists(self.directorio_salida.get()):
                os.makedirs(self.directorio_salida.get())
                self.log(f"Creado directorio: {self.directorio_salida.get()}")
            
            # Inicializar y configurar el extractor
            extractor = ExtractorPrecedentes(
                headless=self.headless.get(),
                timeout_default=45,
                modo_debug=self.modo_debug.get(),
                directorio_salida=self.directorio_salida.get()
            )
            
            # Desactivar reconocimiento visual si no se desea usar
            if not self.usar_reconocimiento.get():
                extractor.reconocimiento_visual = False
                self.log("Reconocimiento visual desactivado manualmente", "INFO")
            
            # Redireccionar log de la clase al widget
            handler = logging.Handler()
            handler.setLevel(logging.INFO)
            handler.emit = lambda record: self.root.after(
                0, self.log, record.getMessage(), record.levelname
            )
            logger.addHandler(handler)
            
            # Ejecutar extracción
            self.log("Iniciando proceso de extracción...", "INFO")
            estadisticas = extractor.extraer_precedentes(self.archivo_json.get())
            
            # Mostrar resultados
            self.log("\nProceso completado", "INFO")
            self.log(f"Jurisprudencias identificadas: {estadisticas['jurisprudencias_identificadas']}")
            self.log(f"Precedentes encontrados: {estadisticas['precedentes_encontrados']}")
            self.log(f"Precedentes extraídos correctamente: {estadisticas['precedentes_extraidos']}")
            self.log(f"Errores: {estadisticas['errores']}")
            
            if estadisticas.get("tiempo_total"):
                self.log(f"Tiempo total: {estadisticas['tiempo_total']:.2f} segundos")
            else:
                tiempo_total = time.time() - estadisticas["tiempo_inicio"]
                self.log(f"Tiempo total: {tiempo_total:.2f} segundos")
            
            # Compilar precedentes si se solicita
            archivo_compilado = None
            if self.compilar.get() and estadisticas.get("archivos_generados"):
                self.log("\nCompilando precedentes en un solo archivo...", "INFO")
                archivo_compilado = extractor.compilar_precedentes()
                
                if archivo_compilado:
                    self.log(f"Precedentes compilados en: {archivo_compilado}", "INFO")
                else:
                    self.log("Error al compilar precedentes", "ERROR")
            
            # Mostrar mensaje de éxito
            self.root.after(0, lambda: messagebox.showinfo(
                "Proceso Completado",
                f"Se han extraído {estadisticas['precedentes_extraidos']} precedentes de "
                f"{estadisticas['jurisprudencias_identificadas']} jurisprudencias identificadas."
            ))
            
            # Abrir archivo si se solicita
            if self.abrir_archivo.get() and archivo_compilado:
                try:
                    import subprocess
                    import platform
                    
                    self.log("Abriendo archivo compilado...", "INFO")
                    
                    sistema = platform.system()
                    if sistema == "Windows":
                        os.startfile(archivo_compilado)
                    elif sistema == "Darwin":  # macOS
                        subprocess.run(["open", archivo_compilado])
                    else:  # Linux y otros
                        subprocess.run(["xdg-open", archivo_compilado])
                except Exception as e:
                    self.log(f"No se pudo abrir el archivo: {str(e)}", "ERROR")
            
        except Exception as e:
            self.log(f"ERROR FATAL: {str(e)}", "ERROR")
            traceback.print_exc()
            self.root.after(0, lambda: messagebox.showerror(
                "Error",
                f"Error en el proceso de extracción: {str(e)}\n\n"
                "Revise el log para más detalles."
            ))
        
        finally:
            # Detener barra de progreso
            self.root.after(0, self.progress.stop)
            self.root.after(0, lambda: self.progress_label.config(text="Proceso finalizado"))
            
            # Habilitar botones
            def habilitar_botones():
                for widget in self.root.winfo_children():
                    if isinstance(widget, ttk.Button):
                        widget.config(state=tk.NORMAL)
            
            self.root.after(0, habilitar_botones)
            
            # Eliminar el handler personalizado
            for handler in logger.handlers:
                if isinstance(handler, logging.Handler) and not isinstance(handler, (logging.StreamHandler, logging.FileHandler)):
                    logger.removeHandler(handler)
                    break


# --------------------------------------------------------
# Punto de entrada principal
# --------------------------------------------------------

if __name__ == "__main__":
    # Verificar si se ejecuta desde línea de comandos o como GUI
    if len(sys.argv) > 1:
        # Modo línea de comandos
        parser = argparse.ArgumentParser(description="Extractor de Precedentes Jurisprudenciales - Con Reconocimiento Visual")
        
        parser.add_argument("archivo_json", help="Ruta al archivo JSON con resultados de búsqueda")
        parser.add_argument("--dir", "-d", dest="directorio_salida", default="precedentes_extraidos",
                           help="Directorio donde guardar precedentes extraídos")
        parser.add_argument("--headless", dest="headless", action="store_true",
                           help="Activar modo headless (navegador invisible)")
        parser.add_argument("--no-headless", dest="headless", action="store_false",
                           help="Desactivar modo headless (mostrar navegador)")
        parser.add_argument("--debug", dest="modo_debug", action="store_true",
                           help="Activar modo debug (genera capturas de pantalla)")
        parser.add_argument("--no-debug", dest="modo_debug", action="store_false",
                           help="Desactivar modo debug")
        parser.add_argument("--no-compilar", dest="compilar", action="store_false",
                           help="No compilar precedentes en un archivo")
        parser.add_argument("--no-abrir", dest="abrir_archivo", action="store_false",
                           help="No abrir archivo compilado al finalizar")
        parser.add_argument("--timeout", "-t", dest="timeout", type=int, default=45,
                           help="Tiempo máximo de espera en segundos")
        parser.add_argument("--no-visual", dest="usar_visual", action="store_false",
                           help="Desactivar reconocimiento visual incluso si está disponible")
        
        # Valores predeterminados
        parser.set_defaults(headless=False, modo_debug=True, compilar=True, abrir_archivo=True, usar_visual=True)
        
        args = parser.parse_args()
        
        # Si se desactiva el reconocimiento visual, advertir sobre mejor rendimiento sin headless
        if args.usar_visual and args.headless and OPENCV_DISPONIBLE:
            print("ADVERTENCIA: El reconocimiento visual funciona mejor con navegador visible (--no-headless).")
            respuesta = input("¿Desea continuar con modo headless? (s/N): ").strip().lower()
            if respuesta != "s":
                args.headless = False
                print("Modo headless desactivado para mejor reconocimiento visual.")
        
        # Ejecutar extracción
        extractor = ExtractorPrecedentes(
            headless=args.headless,
            timeout_default=args.timeout,
            modo_debug=args.modo_debug,
            directorio_salida=args.directorio_salida
        )
        
        # Desactivar reconocimiento visual si se solicita
        if not args.usar_visual:
            extractor.reconocimiento_visual = False
            print("Reconocimiento visual desactivado manualmente.")
        
        # Ejecutar el proceso
        estadisticas = extractor.extraer_precedentes(args.archivo_json)
        
        # Compilar si se solicita
        if args.compilar and estadisticas.get("archivos_generados"):
            print("\nCompilando precedentes en un solo archivo...")
            archivo_compilado = extractor.compilar_precedentes()
            
            if archivo_compilado and args.abrir_archivo:
                try:
                    import subprocess
                    import platform
                    
                    sistema = platform.system()
                    if sistema == "Windows":
                        os.startfile(archivo_compilado)
                    elif sistema == "Darwin":  # macOS
                        subprocess.run(["open", archivo_compilado])
                    else:  # Linux y otros
                        subprocess.run(["xdg-open", archivo_compilado])
                except Exception as e:
                    print(f"No se pudo abrir el archivo: {str(e)}")
    
    else:
        # Mostrar mensaje de bienvenida
        print("\n" + "=" * 70)
        print("EXTRACTOR DE PRECEDENTES JURISPRUDENCIALES - CON RECONOCIMIENTO VISUAL")
        print("=" * 70)
        print("\nEsta herramienta extrae textos completos de precedentes de jurisprudencias")
        print("de la Suprema Corte de Justicia de la Nación usando reconocimiento visual.\n")
        
        if not OPENCV_DISPONIBLE:
            print("\nADVERTENCIA: OpenCV no está instalado. El reconocimiento visual no estará disponible.")
            print("Para habilitarlo, instale: pip install opencv-python numpy\n")
        
        # Preguntar si usar GUI o CLI interactiva
        try:
            respuesta = input("¿Desea iniciar la interfaz gráfica? (S/n): ").strip().lower()
            
            if respuesta == "" or respuesta.startswith("s"):
                # Modo GUI
                print("Iniciando interfaz gráfica...\n")
                root = tk.Tk()
                app = InterfazExtractorPrecedentes(root)
                root.mainloop()
            else:
                # Modo CLI interactivo
                print("\nModo consola interactiva seleccionado.\n")
                
                # Solicitar información de archivo JSON
                archivo_json = input("Ruta del archivo JSON con resultados: ")
                if not archivo_json:
                    print("Error: Debe proporcionar un archivo JSON válido.")
                    sys.exit(1)
                
                # Verificar que existe
                if not os.path.exists(archivo_json):
                    print(f"Error: El archivo '{archivo_json}' no existe.")
                    sys.exit(1)
                
                # Directorio de salida
                directorio_salida = input("Directorio de salida [precedentes_extraidos]: ")
                if not directorio_salida:
                    directorio_salida = "precedentes_extraidos"
                
                # Opciones adicionales
                headless = input("¿Usar navegador invisible? (s/N): ").strip().lower()
                headless = headless.startswith("s")
                
                modo_debug = input("¿Activar modo debug con capturas? (S/n): ").strip().lower()
                modo_debug = not modo_debug.startswith("n")
                
                usar_visual = "s"
                if OPENCV_DISPONIBLE:
                    usar_visual = input("¿Usar reconocimiento visual? (S/n): ").strip().lower()
                    usar_visual = not usar_visual.startswith("n")
                else:
                    print("Reconocimiento visual no disponible (OpenCV no instalado).")
                    usar_visual = False
                
                if usar_visual and headless:
                    print("ADVERTENCIA: El reconocimiento visual funciona mejor con navegador visible.")
                    respuesta = input("¿Desea desactivar modo headless para mejor reconocimiento? (S/n): ").strip().lower()
                    if respuesta == "" or not respuesta.startswith("n"):
                        headless = False
                        print("Modo headless desactivado para mejor reconocimiento visual.")
                
                compilar = input("¿Compilar resultados en un archivo? (S/n): ").strip().lower()
                compilar = not compilar.startswith("n")
                
                abrir_archivo = input("¿Abrir archivo al finalizar? (S/n): ").strip().lower()
                abrir_archivo = not abrir_archivo.startswith("n")
                
                # Ejecutar extracción
                print("\nIniciando extracción de precedentes...\n")
                
                # Crear instancia del extractor
                extractor = ExtractorPrecedentes(
                    headless=headless,
                    timeout_default=45,
                    modo_debug=modo_debug,
                    directorio_salida=directorio_salida
                )
                
                # Desactivar reconocimiento visual si se solicita
                if not usar_visual:
                    extractor.reconocimiento_visual = False
                    print("Reconocimiento visual desactivado manualmente.")
                
                # Ejecutar el proceso
                estadisticas = extractor.extraer_precedentes(archivo_json)
                
                # Compilar si se solicita
                if compilar and estadisticas.get("archivos_generados"):
                    print("\nCompilando precedentes en un solo archivo...")
                    archivo_compilado = extractor.compilar_precedentes()
                    
                    if archivo_compilado and abrir_archivo:
                        try:
                            import subprocess
                            import platform
                            
                            sistema = platform.system()
                            if sistema == "Windows":
                                os.startfile(archivo_compilado)
                            elif sistema == "Darwin":  # macOS
                                subprocess.run(["open", archivo_compilado])
                            else:  # Linux y otros
                                subprocess.run(["xdg-open", archivo_compilado])
                        except Exception as e:
                            print(f"No se pudo abrir el archivo: {str(e)}")
                
        except KeyboardInterrupt:
            print("\nOperación cancelada por el usuario.")
            sys.exit(0)