#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
SCRIPT LANZADOR DEL INTEGRADOR DE TESIS SCJN
Este script garantiza la correcta carga de módulos independientemente del entorno
(script directo o ejecutable PyInstaller).
"""

import os
import sys
import importlib.util
import traceback
import tkinter as tk
from tkinter import messagebox

def determine_base_dir():
    """Determina el directorio base de la aplicación con mejor detección para PyInstaller."""
    base_dir = None
    
    # Verificar primero si estamos en un entorno PyInstaller
    if getattr(sys, 'frozen', False):
        # Estamos en un ejecutable PyInstaller
        base_dir = sys._MEIPASS  # Esta es la ruta especial de PyInstaller
        print(f"Entorno PyInstaller detectado. Directorio base (_MEIPASS): {base_dir}")
    else:
        # Estamos ejecutando como script normal
        base_dir = os.path.dirname(os.path.abspath(__file__))
        print(f"Entorno de script normal. Directorio base: {base_dir}")
    
    # Imprimir información de depuración
    print(f"Listado de archivos en el directorio base:")
    try:
        for file in os.listdir(base_dir):
            print(f"  - {file}")
    except Exception as e:
        print(f"Error al listar archivos: {str(e)}")
    
    # También verificar archivos en el directorio actual
    print(f"Directorio de trabajo actual: {os.getcwd()}")
    print(f"Listado de archivos en el directorio actual:")
    try:
        for file in os.listdir(os.getcwd()):
            print(f"  - {file}")
    except Exception as e:
        print(f"Error al listar archivos: {str(e)}")
        
    return base_dir

def add_to_sys_path(path):
    """Añade una ruta al sys.path si no está presente."""
    if path not in sys.path:
        sys.path.insert(0, path)
        print(f"Añadido al sys.path: {path}")

def check_critical_modules(base_dir, modules):
    """Verifica la existencia de módulos críticos en múltiples ubicaciones posibles."""
    missing_modules = []
    
    for module_file in modules:
        # Comprobar en múltiples ubicaciones
        possible_paths = [
            os.path.join(base_dir, module_file),                        # Directorio base
            os.path.join(os.getcwd(), module_file),                     # Directorio actual
        ]
        
        # Si es PyInstaller, comprobar también junto al ejecutable
        if getattr(sys, 'frozen', False):
            possible_paths.append(os.path.join(os.path.dirname(sys.executable), module_file))
        
        found = False
        for path in possible_paths:
            if os.path.exists(path):
                print(f"Módulo encontrado: {path}")
                found = True
                break
        
        if not found:
            missing_modules.append(module_file)
            print(f"ADVERTENCIA: No se encontró el archivo requerido: {module_file}")
            print(f"   Rutas buscadas: {', '.join(possible_paths)}")
    
    return missing_modules

def load_module_dynamically(module_name, file_path):
    """Carga dinámicamente un módulo desde una ruta de archivo con múltiples intentos."""
    print(f"Intentando cargar módulo '{module_name}' desde: {file_path}")
    
    # Lista de posibles ubicaciones donde buscar el archivo
    possible_paths = [
        file_path,  # La ruta original
        os.path.join(os.getcwd(), os.path.basename(file_path)),  # Directorio actual
    ]
    
    # Si estamos en PyInstaller, agregar más rutas
    if getattr(sys, 'frozen', False):
        possible_paths.append(os.path.join(os.path.dirname(sys.executable), os.path.basename(file_path)))
        possible_paths.append(os.path.join(sys._MEIPASS, os.path.basename(file_path)))
    
    loaded_module = None
    errors = []
    
    # Intentar cada ruta posible
    for path in possible_paths:
        print(f"Intentando cargar desde: {path}")
        if not os.path.exists(path):
            print(f"  El archivo no existe en esta ruta")
            continue
            
        try:
            spec = importlib.util.spec_from_file_location(module_name, path)
            if not spec or not spec.loader:
                print(f"  No se pudo crear especificación para {module_name} en {path}")
                continue

            module = importlib.util.module_from_spec(spec)
            # Registrar en sys.modules ANTES de ejecutar
            sys.modules[module_name] = module
            spec.loader.exec_module(module)
            print(f"  Módulo '{module_name}' cargado exitosamente desde {path}")
            loaded_module = module
            break  # Éxito, salir del bucle
        except Exception as e:
            error_msg = f"Error al cargar '{module_name}' desde {path}: {str(e)}"
            print(error_msg)
            errors.append(error_msg)
            traceback.print_exc()
            # Eliminar del sys.modules si la carga falló
            if module_name in sys.modules:
                del sys.modules[module_name]
    
    if loaded_module is None and errors:
        print(f"No se pudo cargar '{module_name}' desde ninguna ubicación. Errores: {errors}")
    
    return loaded_module

def show_error_and_exit(message):
    """Muestra un mensaje de error en una ventana de diálogo y termina la aplicación."""
    root = tk.Tk()
    root.withdraw()  # Ocultar ventana principal
    messagebox.showerror("Error crítico", message)
    root.destroy()
    sys.exit(1)

def main():
    print("\n--- Iniciando Lanzador del Integrador ---")
    # Detectar directorio base
    base_dir = determine_base_dir()
    
    # Añadir directorio base y directorio actual al sys.path
    add_to_sys_path(base_dir)
    add_to_sys_path(os.getcwd())
    
    # Si estamos en PyInstaller, también añadir el directorio del ejecutable
    if getattr(sys, 'frozen', False):
        add_to_sys_path(os.path.dirname(sys.executable))

    # Verificar módulos críticos
    critical_modules = [
        "gui_integrador_mejorado.py",
        "integrador_tesis_scjn.py",
        "scraper_compuesto7.py",
        "driver_helper.py"
    ]
    missing = check_critical_modules(base_dir, critical_modules)

    if "gui_integrador_mejorado.py" in missing:
        error_msg = "Falta el archivo principal de la GUI (gui_integrador_mejorado.py).\nLa aplicación no puede continuar."
        print("\nError crítico: " + error_msg)
        show_error_and_exit(error_msg)

    if any(m != "gui_integrador_mejorado.py" for m in missing):
        print("\nADVERTENCIA: Faltan algunos módulos. La aplicación podría no funcionar correctamente.")

    # Intentar cargar el módulo GUI dinámicamente
    gui_module_name = "gui_integrador_mejorado"
    gui_path = os.path.join(base_dir, "gui_integrador_mejorado.py")
    gui_module = load_module_dynamically(gui_module_name, gui_path)

    if not gui_module:
        error_msg = "No se pudo cargar el módulo principal de la GUI.\nLa aplicación no puede continuar."
        print("\nError crítico: " + error_msg)
        show_error_and_exit(error_msg)

    # Intentar iniciar la aplicación GUI
    try:
        # Buscar la clase principal de la GUI
        gui_class_name = "IntegradorTesisGUIAvanzado"
        if hasattr(gui_module, gui_class_name):
            print(f"Clase GUI '{gui_class_name}' encontrada. Iniciando interfaz...")
            # Crear la ventana principal de Tkinter
            root = tk.Tk()
            # Instanciar la clase de la GUI
            app = getattr(gui_module, gui_class_name)(root)
            # Iniciar el bucle principal de Tkinter
            root.mainloop()
            print("Aplicación cerrada.")
        else:
            error_msg = f"No se encontró la clase principal '{gui_class_name}' en el módulo GUI."
            print(f"Error: {error_msg}")
            show_error_and_exit(error_msg)

    except Exception as e:
        error_msg = f"Error inesperado al ejecutar la aplicación GUI:\n{str(e)}"
        print(f"\n{error_msg}")
        traceback.print_exc()
        show_error_and_exit(error_msg)

if __name__ == "__main__":
    main()