"""
Helper module para gestionar WebDriver de Chrome
Utiliza webdriver-manager para gestionar la descarga y actualización del driver
"""

import os
import sys
import logging
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options

# Configurar logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger("driver_helper")

def get_chrome_driver(headless=False):
    """Configura y devuelve un driver de Chrome con webdriver-manager"""
    try:
        # Intentar importar webdriver-manager
        try:
            from webdriver_manager.chrome import ChromeDriverManager
            use_webdriver_manager = True
        except ImportError:
            logger.warning("webdriver-manager no está disponible, usando método alternativo")
            use_webdriver_manager = False
        
        # Configurar opciones del navegador
        options = Options()
        if headless:
            options.add_argument("--headless=new")
        
        # Opciones para rendimiento y estabilidad
        options.add_argument("--disable-gpu")
        options.add_argument("--no-sandbox")
        options.add_argument("--disable-dev-shm-usage")
        options.add_argument("--window-size=1920,1080")
        
        # Desactivar imágenes para mejorar rendimiento
        options.add_argument("--blink-settings=imagesEnabled=false")
        options.add_experimental_option("prefs", {
            "profile.default_content_setting_values.images": 2,  # Bloquear imágenes
            "profile.default_content_setting_values.cookies": 1,  # Permitir cookies
            "profile.default_content_setting_values.notifications": 2,  # Bloquear notificaciones
        })
        
        # User-Agent para simular navegador moderno
        options.add_argument("--user-agent=Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/114.0.0.0 Safari/537.36")
        
        # Determinar el método de inicialización del driver
        if use_webdriver_manager:
            # Usar webdriver-manager para gestionar el driver
            service = Service(ChromeDriverManager().install())
            driver = webdriver.Chrome(service=service, options=options)
        else:
            # Método alternativo: intentar usar el driver disponible en el sistema
            driver = webdriver.Chrome(options=options)
        
        logger.info("WebDriver Chrome inicializado correctamente")
        return driver
    
    except Exception as e:
        logger.error(f"Error al inicializar Chrome WebDriver: {str(e)}")
        raise