"""
Controlador de Precedentes para el Integrador de Tesis SCJN
Módulo intermediario que sirve como puente entre la GUI y el extractor de precedentes.
Facilita la comunicación, gestiona logs y maneja el estado del proceso.
"""

import os
import sys
import time
import json
import logging
import threading
import traceback
from datetime import datetime
import importlib.util

# Configuración de logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("controlador_precedentes.log"),
        logging.StreamHandler()
    ]
)
controlador_logger = logging.getLogger("ControladorPrecedentes")

# Determinar si estamos ejecutando como script o como ejecutable
if getattr(sys, 'frozen', False):
    # Si es ejecutable, usar sys._MEIPASS para encontrar recursos
    base_path = sys._MEIPASS
else:
    # Si es script, usar el directorio actual
    base_path = os.path.dirname(os.path.abspath(__file__))

# Agregar la ruta base al sys.path para que pueda encontrar los módulos
sys.path.insert(0, base_path)

# Variables globales para control de estado
EXTRACTOR_IMPORTADO = False
error_importacion_extractor = None

# Intentar importar el extractor de precedentes
try:
    # Importación directa
    from extractor_precedentes import ExtractorPrecedentes, ejecutar_extraccion_precedentes
    EXTRACTOR_IMPORTADO = True
    controlador_logger.info("Importación directa de 'extractor_precedentes' exitosa.")
except ImportError as e:
    error_importacion_extractor = f"Error importación directa: {str(e)}"
    controlador_logger.error(f"Fallo importación directa: {error_importacion_extractor}")
    
    try:
        # Intentar carga dinámica desde la ruta del ejecutable/script
        extractor_path = os.path.join(base_path, "extractor_precedentes.py")
        controlador_logger.info(f"Intentando carga dinámica desde: {extractor_path}")
        
        if os.path.exists(extractor_path):
            spec = importlib.util.spec_from_file_location("extractor_precedentes", extractor_path)
            if spec and spec.loader:
                extractor_module = importlib.util.module_from_spec(spec)
                sys.modules["extractor_precedentes"] = extractor_module
                spec.loader.exec_module(extractor_module)
                
                # Verificar las clases/funciones necesarias
                if hasattr(extractor_module, 'ExtractorPrecedentes') and hasattr(extractor_module, 'ejecutar_extraccion_precedentes'):
                    ExtractorPrecedentes = extractor_module.ExtractorPrecedentes
                    ejecutar_extraccion_precedentes = extractor_module.ejecutar_extraccion_precedentes
                    EXTRACTOR_IMPORTADO = True
                    error_importacion_extractor = None
                    controlador_logger.info("Módulo 'extractor_precedentes' cargado dinámicamente con éxito")
                else:
                    error_importacion_extractor += " | Carga dinámica OK, pero faltan componentes necesarios"
            else:
                error_importacion_extractor += " | No se pudo crear especificación de módulo para carga dinámica"
        else:
            error_importacion_extractor += f" | Archivo no encontrado: {extractor_path}"
    except Exception as e2:
        error_importacion_extractor += f" | Error fatal en carga alternativa: {str(e2)}"
        controlador_logger.error(f"Error crítico al cargar extractor_precedentes.py: {str(e2)}")

class ControladorPrecedentes:
    """
    Clase controladora que sirve como intermediario entre la GUI y el extractor de precedentes.
    Proporciona métodos para verificar, iniciar y monitorear la extracción de precedentes.
    """
    
    def __init__(self, callback_log=None, callback_progreso=None, callback_finalizado=None):
        """
        Inicializa el controlador de precedentes.
        
        Args:
            callback_log (callable): Función para recibir mensajes de log
            callback_progreso (callable): Función para recibir actualizaciones de progreso
            callback_finalizado (callable): Función que se llama al finalizar el proceso
        """
        self.callback_log = callback_log
        self.callback_progreso = callback_progreso
        self.callback_finalizado = callback_finalizado
        
        self.extractor = None
        self.proceso_activo = False
        self.thread_extraccion = None
        self.tiempo_inicio = None
        self.total_jurisprudencias = 0
        
        # Verificar disponibilidad del extractor
        self.extractor_disponible = EXTRACTOR_IMPORTADO
        self.error_importacion = error_importacion_extractor
        
        # Estadísticas del proceso
        self.estadisticas = {
            "jurisprudencias_identificadas": 0,
            "precedentes_encontrados": 0,
            "precedentes_extraidos": 0,
            "errores": 0,
            "archivos_generados": [],
            "archivo_compilado": None,
            "tiempo_total": 0
        }
        
        if not self.extractor_disponible:
            self.log("⚠️ El extractor de precedentes no está disponible", "warning")
            if self.error_importacion:
                self.log(f"Error: {self.error_importacion}", "error")
    
    def verificar_estado(self):
        """
        Verifica el estado del extractor y devuelve información sobre su disponibilidad.
        
        Returns:
            dict: Diccionario con información sobre el estado del extractor
        """
        return {
            "disponible": self.extractor_disponible,
            "error": self.error_importacion,
            "opencv_disponible": self._verificar_opencv(),
            "proceso_activo": self.proceso_activo
        }
    
    def _verificar_opencv(self):
        """
        Verifica si OpenCV está disponible para el reconocimiento visual.
        
        Returns:
            bool: True si OpenCV está disponible, False en caso contrario
        """
        try:
            import cv2
            import numpy as np
            return True
        except ImportError:
            return False
        
    def verificar_archivo_json(self, ruta_archivo):
        """
        Verifica que el archivo JSON seleccionado contiene resultados válidos
        y devuelve información sobre jurisprudencias encontradas.
        
        Args:
            ruta_archivo (str): Ruta al archivo JSON a verificar
            
        Returns:
            dict: Información sobre el archivo JSON verificado
        """
        resultado = {
            "valido": False,
            "mensaje": "",
            "jurisprudencias": 0,
            "total_resultados": 0,
            "ruta_archivo": ruta_archivo
        }
        
        try:
            if not os.path.exists(ruta_archivo):
                resultado["mensaje"] = f"El archivo {ruta_archivo} no existe"
                return resultado
                
            with open(ruta_archivo, 'r', encoding='utf-8') as f:
                datos = json.load(f)
                
            # Verificar que contiene resultados
            resultados = datos.get('resultados', [])
            if not resultados:
                resultado["mensaje"] = "El archivo no contiene resultados"
                return resultado
                
            resultado["total_resultados"] = len(resultados)
                
            # Contar jurisprudencias
            jurisprudencias = []
            for resultado_item in resultados:
                # Verificar formato del número de tesis para identificar jurisprudencias
                numero_tesis = resultado_item.get('numero_tesis', '')
                if 'J' in numero_tesis or 'j' in numero_tesis:
                    jurisprudencias.append(resultado_item)
            
            # Actualizar resultados
            resultado["jurisprudencias"] = len(jurisprudencias)
            resultado["valido"] = True
            
            if len(jurisprudencias) == 0:
                resultado["mensaje"] = "El archivo no contiene jurisprudencias"
            else:
                resultado["mensaje"] = f"Encontradas {len(jurisprudencias)} jurisprudencias en {len(resultados)} resultados"
                
            return resultado
            
        except json.JSONDecodeError:
            resultado["mensaje"] = "El archivo no es un JSON válido"
            return resultado
        except Exception as e:
            resultado["mensaje"] = f"Error al verificar archivo: {str(e)}"
            return resultado
    
    def iniciar_extraccion(self, archivo_json, directorio_salida, config=None):
        """
        Inicia el proceso de extracción de precedentes en un hilo separado.
        
        Args:
            archivo_json (str): Ruta al archivo JSON con resultados
            directorio_salida (str): Directorio donde guardar los precedentes
            config (dict): Configuración adicional para el extractor
            
        Returns:
            bool: True si se inició correctamente, False en caso contrario
        """
        if not self.extractor_disponible:
            self.log("No se puede iniciar la extracción, el extractor no está disponible", "error")
            return False
            
        if self.proceso_activo:
            self.log("Ya hay un proceso de extracción en curso", "warning")
            return False
            
        # Verificar archivo y directorio
        if not os.path.exists(archivo_json):
            self.log(f"El archivo {archivo_json} no existe", "error")
            return False
            
        # Crear directorio si no existe
        if not os.path.exists(directorio_salida):
            try:
                os.makedirs(directorio_salida)
                self.log(f"Creado directorio: {directorio_salida}", "info")
            except Exception as e:
                self.log(f"Error al crear directorio: {str(e)}", "error")
                return False
        
        # Configuración por defecto
        if config is None:
            config = {}
        
        # Valores por defecto
        headless = config.get('headless', False)
        modo_debug = config.get('modo_debug', True)
        usar_reconocimiento_visual = config.get('usar_reconocimiento_visual', True)
        
        # Log de inicio
        self.log("Iniciando proceso de extracción de precedentes...", "info")
        self.log(f"Archivo JSON: {archivo_json}", "info")
        self.log(f"Directorio de salida: {directorio_salida}", "info")
        self.log(f"Modo headless: {headless}", "info")
        self.log(f"Modo debug: {modo_debug}", "info")
        self.log(f"Usar reconocimiento visual: {usar_reconocimiento_visual}", "info")
        
        # Reiniciar estadísticas
        self.estadisticas = {
            "jurisprudencias_identificadas": 0,
            "precedentes_encontrados": 0,
            "precedentes_extraidos": 0,
            "errores": 0,
            "archivos_generados": [],
            "archivo_compilado": None,
            "tiempo_total": 0
        }
        
        # Marcar inicio del proceso
        self.proceso_activo = True
        self.tiempo_inicio = time.time()
        
        # Notificar progreso inicial
        if self.callback_progreso:
            self.callback_progreso(0, "Iniciando extracción de precedentes...")
        
        # Iniciar hilo de extracción
        self.thread_extraccion = threading.Thread(
            target=self._ejecutar_extraccion_thread,
            args=(archivo_json, directorio_salida, headless, modo_debug, usar_reconocimiento_visual),
            daemon=True
        )
        self.thread_extraccion.start()
        
        return True
    
    def detener_extraccion(self):
        """
        Solicita detener el proceso de extracción en curso.
        
        Returns:
            bool: True si se solicitó la detención, False si no hay proceso activo
        """
        if not self.proceso_activo or not self.extractor:
            return False
            
        self.log("Solicitud de detención enviada. Espere a que finalice el proceso actual...", "warning")
        
        # Intentar cerrar el extractor de manera limpia
        try:
            if hasattr(self.extractor, 'cerrar_scraper'):
                self.extractor.cerrar_scraper()
            
            # Marcar como inactivo
            self.proceso_activo = False
            
            # Notificar progreso
            if self.callback_progreso:
                self.callback_progreso(100, "Deteniendo extracción...")
            
            # Calcular tiempo transcurrido
            if self.tiempo_inicio:
                tiempo_total = time.time() - self.tiempo_inicio
                self.estadisticas["tiempo_total"] = tiempo_total
            
            return True
        except Exception as e:
            self.log(f"Error al detener extracción: {str(e)}", "error")
            return False
    
    def obtener_estadisticas(self):
        """
        Devuelve las estadísticas actuales del proceso de extracción.
        
        Returns:
            dict: Estadísticas del proceso
        """
        # Si hay proceso activo, actualizar el tiempo total
        if self.proceso_activo and self.tiempo_inicio:
            tiempo_actual = time.time() - self.tiempo_inicio
            self.estadisticas["tiempo_total"] = tiempo_actual
            
        return self.estadisticas
    
    def _ejecutar_extraccion_thread(self, archivo_json, directorio_salida, headless, modo_debug, usar_reconocimiento_visual):
        """
        Ejecuta la extracción de precedentes en un hilo separado.
        Este método es privado y no debe llamarse directamente.
        
        Args:
            archivo_json (str): Ruta al archivo JSON con resultados
            directorio_salida (str): Directorio donde guardar los precedentes
            headless (bool): Si se ejecuta en modo headless
            modo_debug (bool): Si se activa el modo debug
            usar_reconocimiento_visual (bool): Si se utiliza reconocimiento visual
        """
        try:
            # Crear instancia del extractor
            self.extractor = ExtractorPrecedentes(
                headless=headless,
                timeout_default=45,
                modo_debug=modo_debug,
                directorio_salida=directorio_salida,
                max_intentos=3
            )
            
            # Desactivar reconocimiento visual si no se desea
            if not usar_reconocimiento_visual:
                self.extractor.reconocimiento_visual = False
                self.log("Reconocimiento visual desactivado manualmente", "info")
            
            # Registrar callback de log si está disponible
            if self.callback_log and hasattr(self.extractor, 'registrar_callback_log'):
                self.extractor.registrar_callback_log(self.callback_log)
            
            # Ejecutar extracción
            estadisticas_extractor = self.extractor.extraer_precedentes(archivo_json)
            
            # Actualizar estadísticas locales
            for clave in ['jurisprudencias_identificadas', 'precedentes_encontrados', 'precedentes_extraidos', 'errores']:
                if clave in estadisticas_extractor:
                    self.estadisticas[clave] = estadisticas_extractor[clave]
            
            # Archivos generados
            if 'archivos_generados' in estadisticas_extractor:
                self.estadisticas["archivos_generados"] = estadisticas_extractor['archivos_generados']
                
                # Buscar archivo compilado
                compilados = [f for f in estadisticas_extractor['archivos_generados'] if "COMPILADO" in f]
                if compilados:
                    self.estadisticas["archivo_compilado"] = compilados[0]
            
            # Tiempo total
            tiempo_total = time.time() - self.tiempo_inicio
            self.estadisticas["tiempo_total"] = tiempo_total
            
            # Marcar finalización
            self.proceso_activo = False
            
            # Log final
            self.log(f"✅ Proceso completado en {tiempo_total:.2f} segundos", "success")
            self.log(f"Jurisprudencias identificadas: {self.estadisticas['jurisprudencias_identificadas']}", "info")
            self.log(f"Precedentes encontrados: {self.estadisticas['precedentes_encontrados']}", "info")
            self.log(f"Precedentes extraídos correctamente: {self.estadisticas['precedentes_extraidos']}", "info")
            self.log(f"Errores: {self.estadisticas['errores']}", "info")
            
            # Notificar finalización
            if self.callback_finalizado:
                self.callback_finalizado(self.estadisticas)
            
            # Notificar progreso final
            if self.callback_progreso:
                self.callback_progreso(100, "Proceso finalizado")
            
        except Exception as e:
            # Capturar cualquier excepción no manejada
            self.log(f"❌ Error inesperado: {str(e)}", "error")
            traceback.print_exc()
            
            # Actualizar estadísticas con el error
            self.estadisticas["error"] = str(e)
            tiempo_total = time.time() - self.tiempo_inicio
            self.estadisticas["tiempo_total"] = tiempo_total
            
            # Marcar finalización
            self.proceso_activo = False
            
            # Notificar finalización con error
            if self.callback_finalizado:
                self.callback_finalizado(self.estadisticas)
            
            # Notificar progreso final
            if self.callback_progreso:
                self.callback_progreso(100, "Proceso interrumpido por error")
        
        finally:
            # Asegurar que se cierra el extractor
            if self.extractor:
                try:
                    if hasattr(self.extractor, 'cerrar_scraper'):
                        self.extractor.cerrar_scraper()
                except:
                    pass    

    def log(self, mensaje, tipo="info"):
        """
        Registra un mensaje en el log y lo envía al callback si está disponible.
        
        Args:
            mensaje (str): Mensaje a registrar
            tipo (str): Tipo de mensaje (info, success, warning, error)
        """
        # Normalizar tipo a minúsculas
        tipo = tipo.lower()
        
        # Mapear tipo a nivel de logging
        nivel_log = {
            "info": logging.INFO,
            "success": logging.INFO,
            "warning": logging.WARNING,
            "error": logging.ERROR,
            "debug": logging.DEBUG
        }.get(tipo, logging.INFO)
        
        # Registrar en el log
        controlador_logger.log(nivel_log, mensaje)
        
        # Enviar al callback si está disponible
        if self.callback_log:
            try:
                self.callback_log(mensaje, tipo)
            except Exception as e:
                controlador_logger.error(f"Error al enviar log a callback: {str(e)}")
    
    def actualizar_progreso(self, valor, mensaje=None):
        """
        Actualiza el valor de progreso y lo transmite al callback.
        
        Args:
            valor (int): Valor de progreso (0-100)
            mensaje (str): Mensaje de estado opcional
        """
        # Validar rango del valor
        valor = max(0, min(100, valor))
        
        # Enviar al callback si está disponible
        if self.callback_progreso:
            try:
                self.callback_progreso(valor, mensaje)
            except Exception as e:
                controlador_logger.error(f"Error al enviar progreso a callback: {str(e)}")
    
    def registrar_callback_log(self, callback):
        """
        Registra una función de callback para recibir mensajes de log.
        
        Args:
            callback (callable): Función que acepta (mensaje, tipo)
        """
        self.callback_log = callback
    
    def registrar_callback_progreso(self, callback):
        """
        Registra una función de callback para recibir actualizaciones de progreso.
        
        Args:
            callback (callable): Función que acepta (valor, mensaje)
        """
        self.callback_progreso = callback
    
    def registrar_callback_finalizado(self, callback):
        """
        Registra una función de callback para ser llamada al finalizar el proceso.
        
        Args:
            callback (callable): Función que acepta (estadisticas)
        """
        self.callback_finalizado = callback
    
    def desregistrar_callbacks(self):
        """Desregistra todas las funciones de callback."""
        self.callback_log = None
        self.callback_progreso = None
        self.callback_finalizado = None
    
    def monitorear_progreso_extractor(self):
        """
        Inicia un hilo de monitoreo para actualizar el progreso desde el extractor.
        Este método se llama automáticamente al iniciar la extracción.
        
        Returns:
            threading.Thread: Hilo de monitoreo
        """
        if not self.extractor or not self.proceso_activo:
            return None
            
        # Crear hilo de monitoreo
        monitor_thread = threading.Thread(
            target=self._monitorear_thread,
            daemon=True
        )
        monitor_thread.start()
        
        return monitor_thread
    
    def _monitorear_thread(self):
        """
        Función de hilo para monitorear el progreso del extractor.
        Este método es privado y no debe llamarse directamente.
        """
        # Intervalo de muestreo en segundos
        intervalo = 0.5
        
        # Etapas del proceso
        etapas_progreso = [
            "Iniciando extracción",
            "Procesando archivo JSON",
            "Identificando jurisprudencias",
            "Extrayendo URLs de precedentes",
            "Accediendo a precedentes",
            "Extrayendo texto de precedentes",
            "Compilando resultados",
            "Finalizando proceso"
        ]
        
        # Valor de progreso inicial
        valor_progreso = 0
        
        # Monitorear mientras el proceso esté activo
        while self.proceso_activo and self.extractor:
            try:
                # Si el extractor tiene estadísticas, actualizarlas
                if hasattr(self.extractor, 'estadisticas'):
                    # Copiar estadísticas del extractor
                    for clave, valor in self.extractor.estadisticas.items():
                        if clave in self.estadisticas:
                            self.estadisticas[clave] = valor
                
                # Calcular progreso basado en jurisprudencias procesadas
                if hasattr(self.extractor, 'estadisticas'):
                    # Obtener total de jurisprudencias y procesadas
                    total_juris = self.extractor.estadisticas.get("jurisprudencias_identificadas", 0)
                    extraidas = self.extractor.estadisticas.get("precedentes_extraidos", 0)
                    encontradas = self.extractor.estadisticas.get("precedentes_encontrados", 0)
                    
                    if total_juris > 0:
                        # Calcular progreso real si hay jurisprudencias
                        progreso_real = (extraidas / total_juris) * 100
                        
                        # Limitar al rango 10-90% para dejar espacio a las etapas
                        # de inicio y finalización
                        progreso_acotado = 10 + (progreso_real * 0.8)
                        valor_progreso = min(90, max(10, progreso_acotado))
                    else:
                        # Si aún no hay conteo de jurisprudencias, usar progreso simulado
                        valor_progreso = min(valor_progreso + 0.5, 20)  # Máximo 20% en esta fase
                else:
                    # Si no tiene estadísticas, simular progreso
                    valor_progreso = min(valor_progreso + 0.2, 80)  # Máximo 80% con simulación
                
                # Seleccionar mensaje de etapa basado en el progreso
                indice_etapa = int(min(valor_progreso / 12.5, len(etapas_progreso) - 1))
                mensaje_etapa = etapas_progreso[indice_etapa]
                
                # Enviar actualización de progreso
                if self.callback_progreso:
                    self.callback_progreso(int(valor_progreso), f"{mensaje_etapa}... ({int(valor_progreso)}%)")
                
                # Pausa antes de la siguiente actualización
                time.sleep(intervalo)
                
            except Exception as e:
                controlador_logger.error(f"Error en monitoreo: {str(e)}")
                time.sleep(intervalo * 2)  # Pausa más larga en caso de error
    
    def obtener_mensajes_depuracion(self):
        """
        Obtiene mensajes de depuración del estado interno del extractor.
        Útil para diagnósticos avanzados.
        
        Returns:
            dict: Información de depuración
        """
        info = {
            "proceso_activo": self.proceso_activo,
            "tiempo_transcurrido": 0,
            "estado_extractor": "No disponible",
            "opencv_disponible": self._verificar_opencv(),
            "estadisticas": self.estadisticas.copy() 
        }
        
        # Calcular tiempo transcurrido si hay proceso activo
        if self.tiempo_inicio:
            info["tiempo_transcurrido"] = time.time() - self.tiempo_inicio
        
        # Obtener estado del extractor si está disponible
        if self.extractor:
            if hasattr(self.extractor, 'estadisticas'):
                info["estadisticas_extractor"] = self.extractor.estadisticas
            
            info["estado_extractor"] = "Activo"
            
            # Verificar reconocimiento visual
            if hasattr(self.extractor, 'reconocimiento_visual'):
                info["reconocimiento_visual"] = self.extractor.reconocimiento_visual
        
        return info     

    def abrir_archivo_compilado(self):
        """
        Abre el archivo compilado de precedentes generado.
        
        Returns:
            bool: True si se abrió correctamente, False en caso contrario
        """
        archivo_compilado = self.estadisticas.get("archivo_compilado")
        
        if not archivo_compilado or not os.path.exists(archivo_compilado):
            self.log("No hay un archivo compilado disponible para abrir", "warning")
            return False
            
        try:
            self.log(f"Abriendo archivo compilado: {archivo_compilado}", "info")
            
            # Abrir el archivo según el sistema operativo
            if sys.platform.startswith('win'):
                os.startfile(archivo_compilado)
            elif sys.platform.startswith('darwin'):  # macOS
                import subprocess
                subprocess.run(['open', archivo_compilado])
            else:  # Linux y otros
                import subprocess
                subprocess.run(['xdg-open', archivo_compilado])
                
            return True
                
        except Exception as e:
            self.log(f"Error al abrir archivo: {str(e)}", "error")
            return False
    
    def analizar_json_jurisprudencias(self, ruta_archivo):
        """
        Analiza un archivo JSON para identificar jurisprudencias y extraer información relevante.
        
        Args:
            ruta_archivo (str): Ruta al archivo JSON a analizar
            
        Returns:
            tuple: (jurisprudencias, total_resultados)
        """
        try:
            if not os.path.exists(ruta_archivo):
                self.log(f"El archivo {ruta_archivo} no existe", "error")
                return [], 0
                
            with open(ruta_archivo, 'r', encoding='utf-8') as f:
                datos = json.load(f)
                
            # Verificar que contiene resultados
            resultados = datos.get('resultados', [])
            if not resultados:
                self.log("El archivo no contiene resultados", "warning")
                return [], 0
            
            # Identificar jurisprudencias
            jurisprudencias = []
            for resultado in resultados:
                # Verificar formato del número de tesis para identificar jurisprudencias
                numero_tesis = resultado.get('numero_tesis', '')
                if 'J' in numero_tesis or 'j' in numero_tesis:
                    jurisprudencias.append(resultado)
            
            # Log del análisis
            self.log(f"Análisis de archivo JSON: {len(jurisprudencias)} jurisprudencias en {len(resultados)} resultados", "info")
            
            return jurisprudencias, len(resultados)
            
        except json.JSONDecodeError:
            self.log("El archivo no es un JSON válido", "error")
            return [], 0
        except Exception as e:
            self.log(f"Error al analizar archivo JSON: {str(e)}", "error")
            return [], 0
    
    def generar_archivo_urls(self, jurisprudencias, archivo_salida):
        """
        Genera un archivo con las URLs de los precedentes identificados.
        
        Args:
            jurisprudencias (list): Lista de jurisprudencias extraídas del JSON
            archivo_salida (str): Ruta donde guardar el archivo de URLs
            
        Returns:
            tuple: (éxito, num_urls)
        """
        try:
            if not jurisprudencias:
                self.log("No hay jurisprudencias para generar archivo de URLs", "warning")
                return False, 0
            
            # Crear directorio si no existe
            directorio = os.path.dirname(archivo_salida)
            if directorio and not os.path.exists(directorio):
                os.makedirs(directorio)
            
            # Generar URLs para cada jurisprudencia
            urls = []
            for juris in jurisprudencias:
                registro = juris.get('registro_digital')
                if registro and registro != "No identificado":
                    # Verificar si ya tiene URL generada
                    if 'url_tesis' in juris and juris['url_tesis']:
                        urls.append(juris['url_tesis'])
                    else:
                        # Generar URL basada en el registro
                        if hasattr(self, 'generar_url_registro'):
                            url = self.generar_url_registro(registro)
                            urls.append(url)
            
            # Guardar URLs en archivo
            with open(archivo_salida, 'w', encoding='utf-8') as f:
                for url in urls:
                    f.write(f"{url}\n")
            
            self.log(f"Archivo de URLs generado con {len(urls)} direcciones: {archivo_salida}", "info")
            return True, len(urls)
            
        except Exception as e:
            self.log(f"Error al generar archivo de URLs: {str(e)}", "error")
            return False, 0
    
    def generar_url_registro(self, registro_digital):
        """
        Genera una URL para acceder a la información de una tesis basada en su registro digital.
        
        Args:
            registro_digital (str): Número de registro digital de la tesis
            
        Returns:
            str: URL completa para acceder a la tesis
        """
        plantilla_url = ("https://sjf2.scjn.gob.mx/services/sjftesismicroservice/api/public/tesis/reporte/"
                        "{registro}?nameDocto=Tesis&hostName=https://sjf2.scjn.gob.mx&soloParrafos=false&appSource=SJFAPP2020")
        return plantilla_url.format(registro=registro_digital)
    
    def es_valido_json(self, ruta_archivo):
        """
        Verifica rápidamente si un archivo es un JSON válido.
        
        Args:
            ruta_archivo (str): Ruta al archivo a verificar
            
        Returns:
            bool: True si es un JSON válido, False en caso contrario
        """
        try:
            if not os.path.exists(ruta_archivo):
                return False
                
            with open(ruta_archivo, 'r', encoding='utf-8') as f:
                json.load(f)
                
            return True
        except:
            return False
    
    def obtener_info_sistema(self):
        """
        Obtiene información del sistema para diagnóstico.
        
        Returns:
            dict: Información del sistema
        """
        info = {
            "plataforma": sys.platform,
            "version_python": sys.version,
            "fecha_hora": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
            "extractor_disponible": self.extractor_disponible,
            "opencv_disponible": self._verificar_opencv(),
            "ejecutable": getattr(sys, 'frozen', False)
        }
        
        # Agregar rutas
        info["directorio_base"] = base_path
        if getattr(sys, 'frozen', False):
            info["ruta_ejecutable"] = sys.executable
        
        # Verificar disponibilidad de módulos clave
        try:
            import tkinter
            info["tkinter_disponible"] = True
        except ImportError:
            info["tkinter_disponible"] = False
        
        try:
            import requests
            info["requests_disponible"] = True
        except ImportError:
            info["requests_disponible"] = False
        
        return info


def crear_controlador(callback_log=None, callback_progreso=None, callback_finalizado=None):
    """
    Función auxiliar para crear una instancia del controlador con los callbacks especificados.
    
    Args:
        callback_log (callable): Función para recibir mensajes de log
        callback_progreso (callable): Función para recibir actualizaciones de progreso
        callback_finalizado (callable): Función que se llama al finalizar el proceso
        
    Returns:
        ControladorPrecedentes: Instancia del controlador
    """
    return ControladorPrecedentes(
        callback_log=callback_log,
        callback_progreso=callback_progreso,
        callback_finalizado=callback_finalizado
    )


# Funciones de ayuda para uso independiente

def print_log(mensaje, tipo="info"):
    """Callback de ejemplo para log en consola."""
    # Mapear tipo a colores ANSI
    colores = {
        "info": "\033[0m",      # Normal
        "success": "\033[92m",  # Verde
        "warning": "\033[93m",  # Amarillo
        "error": "\033[91m"     # Rojo
    }
    
    # Seleccionar color o normal si no está disponible
    color = colores.get(tipo, "\033[0m")
    reset = "\033[0m"
    
    # Imprimir mensaje con color
    print(f"{color}{mensaje}{reset}")


def print_progreso(valor, mensaje=None):
    """Callback de ejemplo para mostrar progreso en consola."""
    barra = "█" * int(valor / 2) + " " * (50 - int(valor / 2))
    if mensaje:
        print(f"\r[{barra}] {valor}% {mensaje}", end="")
    else:
        print(f"\r[{barra}] {valor}%", end="")


def finalizado_proceso(estadisticas):
    """Callback de ejemplo para mostrar estadísticas al finalizar."""
    print("\n\n--- PROCESO FINALIZADO ---")
    print(f"Jurisprudencias identificadas: {estadisticas.get('jurisprudencias_identificadas', 0)}")
    print(f"Precedentes encontrados: {estadisticas.get('precedentes_encontrados', 0)}")
    print(f"Precedentes extraídos: {estadisticas.get('precedentes_extraidos', 0)}")
    print(f"Errores: {estadisticas.get('errores', 0)}")
    
    # Mostrar archivo compilado si existe
    if estadisticas.get("archivo_compilado"):
        print(f"Archivo compilado: {estadisticas['archivo_compilado']}")


def main():
    """Función principal para uso independiente del controlador."""
    print("\n" + "=" * 70)
    print("CONTROLADOR DE EXTRACCIÓN DE PRECEDENTES JURISPRUDENCIALES")
    print("=" * 70 + "\n")
    
    # Verificar si el extractor está disponible
    if not EXTRACTOR_IMPORTADO:
        print("⚠️ ADVERTENCIA: El extractor de precedentes no está disponible.")
        if error_importacion_extractor:
            print(f"Error: {error_importacion_extractor}")
        print("\nEl controlador no puede funcionar sin el extractor de precedentes.")
        print("Asegúrese de que el archivo extractor_precedentes.py esté disponible.")
        return
    
    # Crear controlador con callbacks de ejemplo
    controlador = ControladorPrecedentes(
        callback_log=print_log,
        callback_progreso=print_progreso,
        callback_finalizado=finalizado_proceso
    )
    
    # Solicitar archivo JSON
    archivo_json = input("Ingrese la ruta al archivo JSON de resultados: ")
    if not archivo_json:
        print("Error: Debe proporcionar una ruta al archivo JSON.")
        return
    
    # Verificar archivo
    info_archivo = controlador.verificar_archivo_json(archivo_json)
    if not info_archivo["valido"]:
        print(f"Error: {info_archivo['mensaje']}")
        return
    
    print(f"\nArchivo JSON válido: {info_archivo['mensaje']}")
    
    # Solicitar directorio de salida
    directorio_salida = input("Ingrese el directorio de salida [precedentes_extraidos]: ")
    if not directorio_salida:
        directorio_salida = "precedentes_extraidos"
    
    # Configuración
    headless = input("¿Ejecutar en modo sin interfaz gráfica (headless)? (s/N): ").lower().startswith('s')
    modo_debug = input("¿Activar modo debug con capturas? (S/n): ").lower()
    modo_debug = not modo_debug.startswith('n')  # Por defecto sí
    
    usar_reconocimiento = input("¿Usar reconocimiento visual? (S/n): ").lower()
    usar_reconocimiento = not usar_reconocimiento.startswith('n')  # Por defecto sí
    
    # Iniciar extracción
    print("\nIniciando extracción de precedentes...")
    
    config = {
        'headless': headless,
        'modo_debug': modo_debug,
        'usar_reconocimiento_visual': usar_reconocimiento
    }
    
    controlador.iniciar_extraccion(archivo_json, directorio_salida, config)
    
    # Esperar a que termine
    try:
        controlador.monitorear_progreso_extractor()
        
        while controlador.proceso_activo:
            time.sleep(0.1)
        
        print("\n\nProceso completado.")
        
        # Preguntar si quiere abrir el archivo compilado
        archivo_compilado = controlador.estadisticas.get("archivo_compilado")
        if archivo_compilado and os.path.exists(archivo_compilado):
            abrir = input("\n¿Desea abrir el archivo compilado? (S/n): ").lower()
            if not abrir.startswith('n'):  # Por defecto sí
                controlador.abrir_archivo_compilado()
    
    except KeyboardInterrupt:
        print("\n\nDetención solicitada por el usuario.")
        controlador.detener_extraccion()
    
    print("\nPrograma finalizado.")


if __name__ == "__main__":
    main()